/*
 * Decompiled with CFR 0.152.
 */
package me.lightspeed7.mongofs;

import com.mongodb.DB;
import com.mongodb.ReadPreference;
import com.mongodb.WriteConcern;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import me.lightspeed7.mongofs.MongoFile;
import me.lightspeed7.mongofs.MongoFileConstants;
import me.lightspeed7.mongofs.MongoFileCursor;
import me.lightspeed7.mongofs.MongoFileStoreConfig;
import me.lightspeed7.mongofs.MongoFileWriter;
import me.lightspeed7.mongofs.MongoManifest;
import me.lightspeed7.mongofs.MongoZipArchiveQuery;
import me.lightspeed7.mongofs.url.MongoFileUrl;
import me.lightspeed7.mongofs.url.StorageFormat;
import me.lightspeed7.mongofs.util.ChunkSize;
import me.lightspeed7.mongofs.util.TimeMachine;
import org.bson.types.ObjectId;
import org.mongodb.CommandResult;
import org.mongodb.Document;
import org.mongodb.Index;
import org.mongodb.MongoCollection;
import org.mongodb.MongoCollectionOptions;
import org.mongodb.MongoCursor;
import org.mongodb.MongoDatabase;
import org.mongodb.MongoException;
import org.mongodb.MongoView;
import org.mongodb.OrderBy;
import org.mongodb.WriteResult;
import org.mongodb.diagnostics.Loggers;
import org.mongodb.diagnostics.logging.Logger;

public class MongoFileStore {
    private static final Logger LOGGER = Loggers.getLogger("file");
    private final MongoCollection<Document> filesCollection;
    private final MongoCollection<Document> chunksCollection;
    private MongoFileStoreConfig config;

    public MongoFileStore(DB database, MongoFileStoreConfig config) {
        this(new MongoDatabase(database), config);
    }

    public MongoFileStore(MongoDatabase database, MongoFileStoreConfig config) {
        this.config = config;
        WriteConcern writeConcern = config.getWriteConcern() == null ? database.getSurrogate().getWriteConcern() : config.getWriteConcern();
        ReadPreference readPreference = config.getReadPreference() == null ? database.getSurrogate().getReadPreference() : config.getReadPreference();
        MongoCollectionOptions fileOptions = MongoCollectionOptions.builder().writeConcern(writeConcern).readPreference(readPreference).build();
        this.filesCollection = database.getCollection(config.getBucket() + ".files", fileOptions);
        MongoCollectionOptions chunksOptions = MongoCollectionOptions.builder().writeConcern(writeConcern).readPreference(readPreference).build();
        this.chunksCollection = database.getCollection(config.getBucket() + ".chunks", chunksOptions);
        this.checkForExpireAtIndex(this.filesCollection, 0);
        this.checkForExpireAtIndex(this.chunksCollection, 60);
        this.checkForManifestIndex(this.filesCollection);
        try {
            if (this.getCollectionStats(this.filesCollection) < 1000) {
                this.createIdIndexes(this.filesCollection, this.chunksCollection);
            }
        }
        catch (MongoException e) {
            LOGGER.info(String.format("Unable to ensure indices on GridFS collections in database %s", this.filesCollection.getDatabase().getName()));
        }
    }

    private int getCollectionStats(MongoCollection<Document> coll) {
        CommandResult result = coll.getDatabase().executeCommand(new Document("collStats", coll.getName()).append("scale", 1024));
        return result.isOk() ? result.getResponse().getInteger("size") : 0;
    }

    private void checkForExpireAtIndex(MongoCollection<Document> coll, int secondsDelay) {
        for (Document document : coll.tools().getIndexes()) {
            if (!"ttl".equals(document.get("name"))) continue;
            return;
        }
        Index idx = Index.builder().addKey("expireAt", OrderBy.ASC).expireAfterSeconds(secondsDelay).name("ttl").sparse().background(true).build();
        coll.tools().createIndexes(Collections.singletonList(idx));
    }

    private void checkForManifestIndex(MongoCollection<Document> coll) {
        String name = MongoFileConstants.manifestId.name();
        for (Document document : coll.tools().getIndexes()) {
            if (!name.equals(document.get("name"))) continue;
            return;
        }
        Index idx = Index.builder().addKey(name, OrderBy.ASC).name(name).sparse().background(true).build();
        coll.tools().createIndexes(Collections.singletonList(idx));
    }

    private void createIdIndexes(MongoCollection<Document> fileColl, MongoCollection<Document> chunksColl) {
        Index filesIdx = Index.builder().name("filename").addKey("filename", OrderBy.ASC).addKey("uploadDate", OrderBy.ASC).background(true).build();
        fileColl.tools().createIndexes(Collections.singletonList(filesIdx));
        Index chunksIdx = Index.builder().name("files_id").addKey("files_id", OrderBy.ASC).addKey("n", OrderBy.ASC).unique().background(true).build();
        chunksColl.tools().createIndexes(Collections.singletonList(chunksIdx));
    }

    public ChunkSize getChunkSize() {
        return this.config.getChunkSize();
    }

    MongoFileStoreConfig getConfig() {
        return this.config;
    }

    public boolean validateConnection() {
        try {
            Document doc = new Document().append("touch", this.config.getBucket() + ".files").append("data", Boolean.FALSE).append("index", Boolean.TRUE);
            CommandResult commandResult = this.filesCollection.getDatabase().executeCommand(doc);
            if (!commandResult.isOk()) {
                throw new MongoException(commandResult.getErrorMessage());
            }
            return true;
        }
        catch (Exception e) {
            throw new MongoException("Unable to run command on server", e);
        }
    }

    public MongoFileWriter createNew(String filename, String mediaType) throws IOException {
        return this.createNew(filename, mediaType, null, this.config.isCompressionEnabled());
    }

    public MongoFileWriter createNew(String filename, String mediaType, Date expiresAt, boolean compress) throws IOException {
        if (filename == null) {
            throw new IllegalArgumentException("filename cannot be null");
        }
        if (mediaType == null) {
            throw new IllegalArgumentException("mediaType cannot be null");
        }
        if (compress && !this.config.isCompressionEnabled()) {
            throw new IllegalStateException("This data store has compression disabled");
        }
        StorageFormat format = StorageFormat.detect(compress, this.config.isEncryptionEnabled());
        MongoFileUrl mongoFileUrl = MongoFileUrl.construct(new ObjectId(), filename, mediaType, format);
        MongoFile mongoFile = new MongoFile(this, mongoFileUrl, this.config.getChunkSize().getChunkSize());
        if (expiresAt != null) {
            mongoFile.setExpiresAt(expiresAt);
        }
        return new MongoFileWriter(this, mongoFileUrl, mongoFile, this.chunksCollection);
    }

    public MongoFile upload(File file, String mediaType) throws IOException {
        return this.upload(file, mediaType, this.config.isCompressionEnabled(), null);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public MongoFile upload(File file, String mediaType, boolean compress, Date expiresAt) throws IOException {
        if (file == null) {
            throw new IllegalArgumentException("passed in file cannot be null");
        }
        if (!file.exists()) {
            throw new FileNotFoundException("File does not exist or cannot be read by this library");
        }
        try (FileInputStream inputStream = new FileInputStream(file);){
            MongoFile mongoFile = this.upload(file.toPath().toString(), mediaType, expiresAt, compress, inputStream);
            return mongoFile;
        }
    }

    public MongoFile upload(String filename, String mediaType, InputStream inputStream) throws IOException {
        return this.upload(filename, mediaType, null, true, inputStream);
    }

    public MongoFile upload(String filename, String mediaType, Date expiresAt, boolean compress, InputStream inputStream) throws IOException {
        return this.createNew(filename, mediaType, expiresAt, compress).write(inputStream);
    }

    public MongoManifest getManifest(MongoFile file) {
        Document sort;
        if (file == null) {
            throw new IllegalArgumentException("file cannot be null");
        }
        if (!file.isExpandedZipFile()) {
            throw new IllegalStateException("");
        }
        Document query = new Document(MongoFileConstants.manifestId.name(), file.getId());
        MongoFileCursor mongoFileCursor = this.find(query, sort = new Document(MongoFileConstants.manifestNum.name(), 1));
        if (!mongoFileCursor.hasNext()) {
            throw new IllegalStateException("Cannot generate manifest correctly");
        }
        MongoManifest manifest = new MongoManifest(mongoFileCursor.next());
        while (mongoFileCursor.hasNext()) {
            manifest.addMongoFile(mongoFileCursor.next());
        }
        return manifest;
    }

    public MongoManifest getManifest(MongoFileUrl url) throws IOException {
        if (url == null) {
            throw new IllegalArgumentException("file cannot be null");
        }
        MongoFile mongoFile = this.findOne(url);
        if (mongoFile == null) {
            return null;
        }
        return this.getManifest(mongoFile);
    }

    public MongoFile findOne(URL url) {
        if (url == null) {
            throw new IllegalArgumentException("url cannot be null");
        }
        return this.findOne(MongoFileUrl.construct(url));
    }

    public MongoFile findOne(MongoFileUrl url) {
        if (url == null) {
            throw new IllegalArgumentException("url cannot be null");
        }
        MongoCursor<Document> cursor = this.filesCollection.find(new Document().append(MongoFileConstants._id.toString(), url.getMongoFileId())).get();
        if (!cursor.hasNext()) {
            return null;
        }
        Document file = this.deletedFileCheck(cursor.next());
        return file == null ? null : new MongoFile(this, file);
    }

    public boolean exists(MongoFileUrl url) {
        if (url == null) {
            throw new IllegalArgumentException("mongoFile cannot be null");
        }
        return this.findOne(url) != null;
    }

    public boolean exists(ObjectId id) {
        return null != this.findOne(id);
    }

    public MongoFile findOne(ObjectId id) {
        MongoCursor<Document> cursor = this.getFilesCollection().find(new Document("_id", id)).get();
        if (!cursor.hasNext()) {
            return null;
        }
        Document one = this.deletedFileCheck(cursor.next());
        if (one == null) {
            return null;
        }
        return new MongoFile(this, one);
    }

    public MongoFileCursor find(String filename) {
        return this.find(filename, null);
    }

    public MongoFileCursor find(String filename, Document sort) {
        return this.find(new Document(MongoFileConstants.filename.toString(), filename), sort);
    }

    public MongoFileCursor find(Document query) {
        return this.find(query, null);
    }

    public MongoFileCursor find(Document query, Document sort) {
        MongoView<Document> c = this.getFilesCollection().find(query);
        if (sort != null) {
            c.sort(sort);
        }
        MongoCursor<Document> cursor = c.get();
        return new MongoFileCursor(this, cursor);
    }

    public MongoZipArchiveQuery findInZipArchive(MongoFile zipFile) throws Exception {
        return this.findInZipArchive(zipFile.getURL());
    }

    public MongoZipArchiveQuery findInZipArchive(MongoFileUrl zipFileUrl) {
        return new MongoZipArchiveQuery(this, zipFileUrl);
    }

    public void expireFile(MongoFile file, Date when) throws MalformedURLException {
        MongoFileUrl url = file.getURL();
        Document filesQuery = new Document("_id", url.getMongoFileId());
        Document chunksQuery = new Document("files_id", url.getMongoFileId());
        this.setExpiresAt(filesQuery, chunksQuery, when, false);
    }

    public void remove(MongoFile mongoFile) throws IOException {
        this.remove(mongoFile, false);
    }

    public void remove(MongoFile mongoFile, boolean async) throws IOException {
        if (mongoFile == null) {
            throw new IllegalArgumentException("mongoFile cannot be null");
        }
        this.remove(mongoFile.getURL(), async);
    }

    public void remove(MongoFileUrl url) {
        this.remove(url, false);
    }

    public void remove(MongoFileUrl url, boolean async) {
        if (url == null) {
            throw new IllegalArgumentException("mongoFileUrl cannot be null");
        }
        this.remove(new Document("_id", url.getMongoFileId()), async);
    }

    public void remove(Document query) {
        this.remove(query, false);
    }

    public void remove(Document query, boolean async) {
        WriteResult writeResult;
        if (query == null) {
            throw new IllegalArgumentException("query can not be null");
        }
        ArrayList<ObjectId> filesIds = new ArrayList<ObjectId>();
        for (MongoFile f : this.find(query)) {
            if (f.isExpandedZipFile()) {
                for (MongoFile f2 : this.getManifest(f).getFiles()) {
                    filesIds.add(f2.getId());
                }
            }
            filesIds.add(f.getId());
        }
        Document filesQuery = new Document("_id", new Document("$in", filesIds));
        Document chunksQuery = new Document("files_id", new Document("$in", filesIds));
        this.setExpiresAt(filesQuery, chunksQuery, TimeMachine.now().backward(1).seconds().inTime(), true);
        if (!async && (writeResult = this.getFilesCollection().remove(filesQuery)).getCount() > 0) {
            this.getChunksCollection().remove(chunksQuery);
        }
    }

    private void setExpiresAt(Document filesQuery, Document chunksQuery, Date when, boolean multi) {
        Document filesUpdate = new Document().append(MongoFileConstants.expireAt.toString(), when).append(MongoFileConstants.deleted.toString(), when.before(new Date()));
        filesUpdate = new Document().append("$set", filesUpdate);
        this.getFilesCollection().find(filesQuery).withWriteConcern(WriteConcern.JOURNALED).update(filesUpdate);
        Document chunksUpdate = new Document().append(MongoFileConstants.expireAt.toString(), when);
        chunksUpdate = new Document("$set", chunksUpdate);
        this.getChunksCollection().find(chunksQuery).withWriteConcern(WriteConcern.JOURNALED).update(chunksUpdate);
    }

    private Document deletedFileCheck(Document file) {
        if (new MongoFile(this, file).isDeleted()) {
            return null;
        }
        return file;
    }

    MongoCollection<Document> getFilesCollection() {
        return this.filesCollection;
    }

    MongoCollection<Document> getChunksCollection() {
        return this.chunksCollection;
    }

    public String toString() {
        return String.format("MongoFileStore [filesCollection=%s, chunksCollection=%s,%n  config=%s%n]", this.filesCollection, this.chunksCollection, this.config.toString());
    }
}

