package net.md_5.bungee.protocol.packet;

import io.netty.buffer.ByteBuf;
import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.EqualsAndHashCode;
import lombok.NoArgsConstructor;
import net.md_5.bungee.protocol.AbstractPacketHandler;
import net.md_5.bungee.protocol.DefinedPacket;
import net.md_5.bungee.protocol.ProtocolConstants;
import net.md_5.bungee.protocol.nbt.NBTTag; // Waterfall (1.16: 20w21a+)

@Data
@NoArgsConstructor
@AllArgsConstructor
@EqualsAndHashCode(callSuper = false)
public class Login extends DefinedPacket
{

    private int entityId;
    private short gameMode;
    private int dimension;
    private long seed;
    private short difficulty;
    private short maxPlayers;
    private String levelType;
    private int viewDistance;
    private boolean reducedDebugInfo;
    private boolean normalRespawn;
    private boolean isDebug; // Waterfall (1.16: 20w20a+)
    private boolean isFlat; // Waterfall (1.16: 20w20a+)
    private String dimensionCodecName; // Waterfall (1.16: 20w21a+)
    private NBTTag codecTag; // Waterfall (1.16: 20w21a+)
    private String[] worlds; // Waterfall (1.16: 20w22a+)
    private String world; // Waterfall (1.16: 20w22a+)
    private short previousGamemode; // Waterfall (1.16: pre6)

    @Override
    public void read(ByteBuf buf, ProtocolConstants.Direction direction, int protocolVersion)
    {
        entityId = buf.readInt();
        gameMode = buf.readByte(); // Waterfall (1.16: pre6) Change to signed for consistency
        // Waterfall (1.16: 20w21a+) start:
        if ( protocolVersion >= ProtocolConstants.MINECRAFT_1_16 )
        {
            // Waterfall (1.16: pre6) previous gamemode (F3 menu)
            previousGamemode = buf.readByte();
            // Waterfall (1.16: 20w22a) world name array
            int worldNameArrayLen = readVarInt( buf );
            worlds = new String[worldNameArrayLen];
            for ( int iter = 0; iter < worldNameArrayLen; iter++ )
            {
                worlds[iter] = readString( buf );
            }
            codecTag = readNBT( buf );
            dimensionCodecName = readString( buf );
            world = readString( buf ); // Waterfall (1.16: 20w22a) world name
        } else
        {
            if ( protocolVersion > ProtocolConstants.MINECRAFT_1_9 )
            {
                dimension = buf.readInt();
            } else
            {
                dimension = buf.readByte();
            }
        }
        // Waterfall end
        if ( protocolVersion >= ProtocolConstants.MINECRAFT_1_15 )
        {
            seed = buf.readLong();
        }
        if ( protocolVersion < ProtocolConstants.MINECRAFT_1_14 )
        {
            difficulty = buf.readUnsignedByte();
        }
        maxPlayers = buf.readUnsignedByte();
        // Waterfall (1.16: 20w20a+) start:
        if( protocolVersion < ProtocolConstants.MINECRAFT_1_16 )
        {
            levelType = readString( buf );
        }
        // Waterfall end
        if ( protocolVersion >= ProtocolConstants.MINECRAFT_1_14 )
        {
            viewDistance = readVarInt( buf );
        }
        if ( protocolVersion >= 29 )
        {
            reducedDebugInfo = buf.readBoolean();
        }
        if ( protocolVersion >= ProtocolConstants.MINECRAFT_1_15 )
        {
            normalRespawn = buf.readBoolean();
        }
        // Waterfall (1.16: 20w20a+) start:
        if( protocolVersion >= ProtocolConstants.MINECRAFT_1_16 )
        {
            isDebug = buf.readBoolean();
            isFlat = buf.readBoolean();
        }
        // Waterfall end
    }

    @Override
    public void write(ByteBuf buf, ProtocolConstants.Direction direction, int protocolVersion)
    {
        buf.writeInt( entityId );
        buf.writeByte( gameMode );
        // Waterfall (1.16: 20w21a+) start:
        if ( protocolVersion >= ProtocolConstants.MINECRAFT_1_16 )
        {
            // Waterfall (1.16: pre6) previous gamemode (F3 menu)
            buf.writeByte( previousGamemode );
            // Waterfall (1.16: 20w22a) world name array
            writeVarInt( worlds.length, buf );
            for ( int iter = 0; iter < worlds.length; iter++ )
            {
                writeString( worlds[iter], buf );
            }
            writeNBT( codecTag, buf );
            writeString( dimensionCodecName, buf );
            writeString( world, buf ); // Waterfall (1.16: 20w22a) world name
        } else
        {
            if ( protocolVersion > ProtocolConstants.MINECRAFT_1_9 )
            {
                buf.writeInt( dimension );
            } else
            {
                buf.writeByte( dimension );
            }
        }
        // Waterfall end
        if ( protocolVersion >= ProtocolConstants.MINECRAFT_1_15 )
        {
            buf.writeLong( seed );
        }
        if ( protocolVersion < ProtocolConstants.MINECRAFT_1_14 )
        {
            buf.writeByte( difficulty );
        }
        buf.writeByte( maxPlayers );
        // Waterfall (1.16: 20w20a+) start:
        if( protocolVersion < ProtocolConstants.MINECRAFT_1_16 )
        {
            writeString( levelType, buf );
        }
        // Waterfall end
        if ( protocolVersion >= ProtocolConstants.MINECRAFT_1_14 )
        {
            writeVarInt( viewDistance, buf );
        }
        if ( protocolVersion >= 29 )
        {
            buf.writeBoolean( reducedDebugInfo );
        }
        if ( protocolVersion >= ProtocolConstants.MINECRAFT_1_15 )
        {
            buf.writeBoolean( normalRespawn );
        }
        // Waterfall (1.16: 20w20a+) start:
        if( protocolVersion >= ProtocolConstants.MINECRAFT_1_16 )
        {
            buf.writeBoolean( isDebug );
            buf.writeBoolean( isFlat );
        }
        // Waterfall end
    }

    @Override
    public void handle(AbstractPacketHandler handler) throws Exception
    {
        handler.handle( this );
    }
}
