package io.papermc.paper.event.entity;

import org.bukkit.entity.Entity;
import org.bukkit.event.Cancellable;
import org.bukkit.event.HandlerList;
import org.bukkit.event.entity.EntityEvent;
import org.bukkit.util.Vector;
import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.NotNull;

/**
 * Fired when an entity is pushed by another entity's attack. The acceleration vector can be
 * modified. If this event is cancelled, the entity will not get pushed.
 * <p>
 * Note: Some entities might trigger this multiple times on the same entity
 * as multiple acceleration calculations are done.
 */
public class EntityPushedByEntityAttackEvent extends EntityEvent implements Cancellable {

    private static final HandlerList HANDLER_LIST = new HandlerList();

    private final @NotNull Entity pushedBy;
    private @NotNull Vector acceleration;
    private boolean cancelled;

    @ApiStatus.Internal
    public EntityPushedByEntityAttackEvent(@NotNull Entity entity, @NotNull Entity pushedBy, @NotNull Vector acceleration) {
        super(entity);
        this.pushedBy = pushedBy;
        this.acceleration = acceleration;
    }

    /**
     * Gets the entity which pushed the affected entity.
     *
     * @return the pushing entity
     */
    @NotNull
    public Entity getPushedBy() {
        return this.pushedBy;
    }

    /**
     * Gets the acceleration that will be applied to the affected entity.
     *
     * @return the acceleration vector
     */
    @NotNull
    public Vector getAcceleration() {
        return this.acceleration; // TODO Clone in 1.21 to not instantly break what was technically already modifiable
    }

    /**
     * Sets the relative acceleration that will be applied to the affected entity.
     *
     * @param acceleration the new acceleration vector
     */
    public void setAcceleration(final @NotNull Vector acceleration) {
        this.acceleration = acceleration.clone();
    }

    @Override
    public boolean isCancelled() {
        return this.cancelled;
    }

    @Override
    public void setCancelled(boolean cancel) {
        this.cancelled = cancel;
    }

    @NotNull
    @Override
    public HandlerList getHandlers() {
        return HANDLER_LIST;
    }

    @NotNull
    public static HandlerList getHandlerList() {
        return HANDLER_LIST;
    }
}
