package com.destroystokyo.paper.event.server;

import org.bukkit.event.Event;
import org.bukkit.event.HandlerList;
import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.NotNull;

/**
 * Called when the server has finished ticking the main loop
 */
public class ServerTickEndEvent extends Event {

    private static final HandlerList HANDLER_LIST = new HandlerList();

    private final int tickNumber;
    private final double tickDuration;
    private final long timeEnd;

    @ApiStatus.Internal
    public ServerTickEndEvent(int tickNumber, double tickDuration, long timeRemaining) {
        this.tickNumber = tickNumber;
        this.tickDuration = tickDuration;
        this.timeEnd = System.nanoTime() + timeRemaining;
    }

    /**
     * @return What tick this was since start (first tick = 1)
     */
    public int getTickNumber() {
        return this.tickNumber;
    }

    /**
     * @return Time in milliseconds of how long this tick took
     */
    public double getTickDuration() {
        return this.tickDuration;
    }

    /**
     * Amount of nanoseconds remaining before the next tick should start.
     * <p>
     * If this value is negative, then that means the server has exceeded the tick time limit and TPS has been lost.
     * <p>
     * Method will continuously return the updated time remaining value. (return value is not static)
     *
     * @return Amount of nanoseconds remaining before the next tick should start
     */
    public long getTimeRemaining() {
        return this.timeEnd - System.nanoTime();
    }

    @NotNull
    public HandlerList getHandlers() {
        return HANDLER_LIST;
    }

    @NotNull
    public static HandlerList getHandlerList() {
        return HANDLER_LIST;
    }
}
