/*
 * Decompiled with CFR 0.152.
 */
package org.kingdoms.managers.backup;

import java.io.BufferedOutputStream;
import java.io.IOException;
import java.nio.channels.OverlappingFileLockException;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Collection;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.stream.Stream;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import java.util.zip.ZipOutputStream;
import org.kingdoms.locale.MessageHandler;
import org.kingdoms.main.KLogger;
import org.kingdoms.managers.BackupInfo;
import org.kingdoms.managers.backup.ZipEntryCreator;
import org.kingdoms.utils.fs.FSUtil;
import org.kingdoms.utils.fs.walker.visitors.PathVisit;
import org.kingdoms.utils.fs.walker.visitors.PathVisitor;
import org.kingdoms.utils.internal.iterator.Iterables;
import org.kingdoms.utils.internal.runnables.ThrowingRunnable;
import org.kingdoms.utils.time.TimeUtils;

public abstract class BackupManager {
    protected static final DateTimeFormatter DATE_PATTERN = DateTimeFormatter.ofPattern("yyyy-MM-dd");
    public static final int METHOD = 9;
    private static final String EXTENSION = ".zip";
    protected static boolean useMultiBackups = true;
    private final boolean deleteOldBackups;
    protected final Path backups;
    protected final Path toBackup;
    protected final String prefix;
    protected boolean isAutomatic;
    private final ReadWriteLock lock;

    public BackupManager(Path backups, Path toBackup, boolean deleteOldBackups, String prefix) {
        this.backups = Objects.requireNonNull(backups, "Backups directory cannot be null");
        this.toBackup = Objects.requireNonNull(toBackup, "Cannot backup null directory");
        this.deleteOldBackups = deleteOldBackups;
        this.prefix = prefix;
        this.lock = new ReentrantReadWriteLock();
    }

    private static void validateDir(Path zipFolder) {
        try {
            Files.createDirectories(zipFolder, new FileAttribute[0]);
        }
        catch (IOException e) {
            throw new RuntimeException("Failed to create backups directory: " + zipFolder, e);
        }
    }

    public boolean isAutomatic() {
        return this.isAutomatic;
    }

    public void setAutomatic(boolean automatic) {
        this.isAutomatic = automatic;
    }

    public String getDate() {
        return LocalDateTime.now().format(DATE_PATTERN);
    }

    public ReadWriteLock getLock() {
        return this.lock;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void unzip(Path zip, Path unzipTo, boolean replaceExisting) {
        Objects.requireNonNull(zip, "Cannot unzip null directory.");
        Objects.requireNonNull(unzipTo, "Cannot unzip to null directory");
        if (!zip.toString().toLowerCase().endsWith(EXTENSION)) {
            throw new IllegalArgumentException("ZIP path must refer to a ZIP file");
        }
        if (Files.exists(unzipTo, new LinkOption[0]) && !Files.isDirectory(unzipTo, new LinkOption[0])) {
            throw new IllegalArgumentException("Cannot unzip to a non-directory");
        }
        this.lock.readLock().lock();
        try {
            try {
                Files.createDirectories(unzipTo, new FileAttribute[0]);
            }
            catch (IOException e) {
                e.printStackTrace();
            }
            BackupInfo info = BackupManager.unzip(zip, unzipTo);
            this.configureRestorer(info);
            for (BackupInfo.BackupFile file : info.files.values()) {
                if (!file.exists()) continue;
                if (file.restorer == null) {
                    file.copy(file.path);
                    continue;
                }
                file.restorer.accept(file);
            }
        }
        finally {
            this.lock.readLock().unlock();
        }
    }

    private static BackupInfo unzip(Path zip, Path root) {
        Objects.requireNonNull(zip, "Cannot unzip null directory.");
        if (!zip.toString().toLowerCase().endsWith(EXTENSION)) {
            throw new IllegalArgumentException("ZIP path must refer to a ZIP file");
        }
        BackupInfo info = new BackupInfo(root);
        try (ZipFile zipFile = new ZipFile(zip.toString());){
            Enumeration<? extends ZipEntry> entries = zipFile.entries();
            while (entries.hasMoreElements()) {
                ZipEntry entry = entries.nextElement();
                info.putFile(zipFile, entry);
            }
        }
        catch (IOException e) {
            e.printStackTrace();
        }
        return info;
    }

    public void restore(Path from, Path to, boolean replaceIfExists) {
        this.unzip(from, to, replaceIfExists);
    }

    public final void takeBackup() {
        this.takeBackup(this.getZip());
    }

    public void takeBackup(Path to) {
        if (!useMultiBackups && this.hasBackupToday()) {
            return;
        }
        if (this.deleteOldBackups) {
            this.deleteOldBackups(30, TimeUnit.DAYS);
        }
        this.zipFiles(to);
    }

    public int zipFiles(Path zipTo) {
        this.lock.writeLock().lock();
        try {
            int n = this.zipFiles0(zipTo);
            return n;
        }
        finally {
            this.lock.writeLock().unlock();
        }
    }

    private int zipFiles0(Path zipTo) {
        BackupManager.validateDir(zipTo.getParent());
        try {
            Files.createFile(zipTo, new FileAttribute[0]);
        }
        catch (IOException ex) {
            throw new RuntimeException("Error while attempting to create ZIP file: " + zipTo, ex);
        }
        HashSet zipPaths = new HashSet();
        try (ZipOutputStream zs = new ZipOutputStream(new BufferedOutputStream(Files.newOutputStream(zipTo, new OpenOption[0])));){
            zs.setLevel(9);
            zs.setComment("A backup file for Kingdoms minecraft plugin data.\nThese backups contain language file configs and config.yml, kingdoms, players and lands data\ndepending on the options specified in the config.\n\nNote that you have to stop the server before restoring one of these backups.\nBackup taken at: " + TimeUtils.getDateAndTime());
            for (PathVisit pathVisit : Iterables.of(this.getPathVisitor().stream(this.toBackup).filter(x -> x.getVisitType() == PathVisit.Type.ENTRY))) {
                Path file = pathVisit.getPath();
                if (!pathVisit.getAttributes().isRegularFile()) continue;
                ZipEntry zipEntry = new ZipEntry(this.toBackup.relativize(file).toString());
                BackupManager.tryAddEntry(file, () -> {
                    FSUtil.lockAndTransfer(file, zs, () -> zs.putNextEntry(zipEntry));
                    zs.closeEntry();
                    zipPaths.add(zipEntry.getName());
                });
            }
            Collection<ZipEntryCreator> additionalEntries = this.addToZip();
            if (additionalEntries != null) {
                block14: for (ZipEntryCreator additionalEntry : additionalEntries) {
                    String zipPath = additionalEntry.zipPath(this.toBackup);
                    if (zipPaths.contains(zipPath)) {
                        switch (additionalEntry.getDuplicateStrategy()) {
                            case SKIP: {
                                continue block14;
                            }
                            case ERROR: {
                                throw new IllegalStateException("Duplicate zip entry: " + zipPath + " for " + additionalEntry);
                            }
                        }
                        throw new AssertionError((Object)("Unknown duplicate strategy for " + additionalEntry + " - " + (Object)((Object)additionalEntry.getDuplicateStrategy())));
                    }
                    BackupManager.tryAddEntry(additionalEntry, () -> {
                        additionalEntry.processStream(this.toBackup, zs);
                        zs.closeEntry();
                        zipPaths.add(zipPath);
                    });
                }
            }
        }
        catch (IOException e) {
            MessageHandler.sendConsolePluginMessage("&4Error while attempting to take a backup...");
            e.printStackTrace();
        }
        return zipPaths.size();
    }

    private static void tryAddEntry(Object subject, ThrowingRunnable runnable) {
        try {
            runnable.run();
        }
        catch (OverlappingFileLockException e) {
            KLogger.warn("Could not take a backup of " + subject + " because the file is already in use: " + e.getMessage());
        }
        catch (Throwable e) {
            MessageHandler.sendConsolePluginMessage("&4Error while attempting to backup a file&8: &e" + subject);
            e.printStackTrace();
        }
    }

    protected String multiZipSuffix(int number) {
        return " (" + number + ')';
    }

    protected Path getMultiZipName() {
        Path file = this.getRegularZipPath();
        Path parent = file.getParent();
        String name = file.getFileName().toString();
        name = name.substring(0, name.lastIndexOf(46));
        int count = 1;
        while (Files.exists(file, new LinkOption[0])) {
            file = parent.resolve(name + this.multiZipSuffix(count++) + EXTENSION);
        }
        return file;
    }

    protected Collection<ZipEntryCreator> addToZip() {
        return Collections.emptyList();
    }

    public abstract PathVisitor getPathVisitor();

    protected boolean shouldBeDeleted(Path path, int time, TimeUnit timeUnit) {
        long created;
        try {
            created = Files.getLastModifiedTime(path, new LinkOption[0]).to(TimeUnit.MILLISECONDS);
        }
        catch (IOException e) {
            e.printStackTrace();
            return false;
        }
        long diff = System.currentTimeMillis() - created;
        return diff >= timeUnit.toMillis(time);
    }

    public CompletableFuture<Void> deleteOldBackups(int time, TimeUnit timeUnit) {
        if (time <= 0) {
            return null;
        }
        return CompletableFuture.runAsync(() -> {
            if (!Files.exists(this.backups, new LinkOption[0])) {
                return;
            }
            this.lock.writeLock().lock();
            try (Stream<Path> walker = Files.walk(this.backups, new FileVisitOption[0]);){
                walker.filter(x$0 -> Files.isRegularFile(x$0, new LinkOption[0])).filter(x -> x.getFileName().endsWith(EXTENSION)).forEach(f -> {
                    if (this.shouldBeDeleted((Path)f, time, timeUnit)) {
                        try {
                            MessageHandler.sendConsolePluginMessage("&2Deleting old backup... &6" + f.getFileName());
                            Files.delete(f);
                        }
                        catch (IOException e) {
                            e.printStackTrace();
                        }
                    }
                });
            }
            catch (IOException e) {
                e.printStackTrace();
            }
            finally {
                this.lock.writeLock().unlock();
            }
        });
    }

    protected Path getRegularZipPath() {
        String name = this.prefix == null ? this.getDate() : this.prefix;
        return this.backups.resolve(name + EXTENSION);
    }

    public Path getBackupsFolder() {
        return this.backups;
    }

    public Path getToBackupRootFolder() {
        return this.toBackup;
    }

    protected Path getZip() {
        return useMultiBackups ? this.getMultiZipName() : this.getRegularZipPath();
    }

    public boolean hasBackupToday() {
        return Files.exists(this.getZip(), new LinkOption[0]);
    }

    public void configureRestorer(BackupInfo info) {
    }
}

