package at.borkowski.spicej.proxy;

import java.math.BigDecimal;
import java.math.RoundingMode;

import at.borkowski.spicej.ticks.TickSource;

/**
 * A utility class for calculating parameters necessary for rate limitation of
 * bytes per seconds.
 * 
 * The way spiceJ works internally, three parameters are required to set up a
 * byte-rate limited stream:
 * 
 * <ul>
 * <li>Tick interval: How many nanoseconds are between two ticks (see
 * {@link TickSource})</li>
 * <li>Bytes per tick: How many bytes are allowed to be transmitted/received per
 * tick</li>
 * <li>Prescale: By how much to divide the incoming ticks (
 * <code>1 - prescale</code> ticks are skipped before actually allowing data to
 * be sent or received). This is a possibility of skipping ticks to allow for
 * very low bitrates without having extensive tick intervals.</li>
 * </ul>
 * 
 * This class allows for easy calculation of bytes per second to these three
 * parametes while respecting some bounds:
 * 
 * <ul>
 * <li>Minimum interval: The interval must not be shorter than
 * {@link #MIN_INTERVAL_NS} nanoseconds to avoid overhead by CPU usage.</li>
 * <li>Maximum interval: An upper bound must not be exceeded for the tick
 * interval, {@link #MAX_INTERVAL_NS} ns.</li>
 * </ul>
 * 
 * The error generated by this class is influenced by the scale used in
 * calculations (see {@link #SCALE}) is tested by
 * <code>RateCalculatorTest</code> (in test sources).
 * 
 */
public class RateCalculator {

   private RateCalculator() {}

   /**
    * The minimum byte rate representable with our data model (this is not
    * necessarily the exact minimum value, but a value close enough to the
    * minimum lower bound.
    */
   public static final float MIN_RATE = 7.6E-10F;

   /**
    * The highest byte rate representable with our data model (this is not
    * necessarily the exact maximum value, but a value close enough to the
    * maximum upper bound.
    */
   public static final float MAX_RATE = 8.7E+13F;

   /**
    * Nanoseconds per second.
    */
   public static final int NS_PER_S = 1000000000;

   /**
    * The default enforced lower bound for the interval. Intervals lower than
    * this might not be reliably possible on some hardware.
    */
   public static final int MIN_INTERVAL_NS = 100;

   /**
    * The enforced upper bound for the interval. Rates requiring a higher
    * interval will be forced to be realized use prescaling.
    */
   public static final int MAX_INTERVAL_NS = NS_PER_S;

   /**
    * The enforced upper bound for bytes per tick. Higher values result in a
    * bigger buffer and higher memory usage.
    */
   public static final int MAX_BYTES_PER_TICK = 10 * 1024 * 1024;

   /**
    * The scale (see {@link BigDecimal}) used for internal calculations.
    */
   public static final int SCALE = 70;

   /**
    * The rounding mode (see {@link BigDecimal}) used for internal calculation.
    */
   public static final RoundingMode ROUND = RoundingMode.HALF_UP;

   /**
    * The error margin permitted for calculations.
    */
   public static final double MAX_ERROR = 0.005;

   /**
    * Performs a calculation with the default minimal interval (see
    * {@link #MIN_INTERVAL_NS}). See {@link RateCalculator} for details.
    * 
    * @param rateBytesPerSecond
    *           The rate in bytes per second to target
    * @return the calculation result
    */
   public static Result calculate(float rateBytesPerSecond) {
      return calculate(rateBytesPerSecond, MIN_INTERVAL_NS);
   }

   /**
    * Performs a calculation with the given minimal interval. See
    * {@link RateCalculator} for details.
    * 
    * @param rateBytesPerSecond
    *           The rate in bytes per second to target
    * @param minInterval
    *           The minimal interval in nanoseconds.
    * @return the calculation result
    */
   public static Result calculate(float rateBytesPerSecond, int minInterval) {
      BigDecimal B_MIN = BigDecimal.ONE;
      BigDecimal B_MAX = BigDecimal.valueOf(MAX_BYTES_PER_TICK);
      BigDecimal I_MIN = BigDecimal.valueOf(minInterval);
      BigDecimal I_MAX = BigDecimal.valueOf(MAX_INTERVAL_NS);
      BigDecimal Id_MAX = BigDecimal.valueOf(Integer.MAX_VALUE);

      BigDecimal ONEG = BigDecimal.valueOf(NS_PER_S);
      BigDecimal TWO = BigDecimal.valueOf(2);
      BigDecimal E = BigDecimal.valueOf(Math.E);

      BigDecimal R = BigDecimal.valueOf(rateBytesPerSecond).divide(ONEG, SCALE, ROUND);
      BigDecimal I = I_MAX;

      BigDecimal B = R.multiply(I);

      BigDecimal ERROR;
      BigDecimal ERROR_MAX = BigDecimal.valueOf(MAX_ERROR);

      BigDecimal R_set = BigDecimal.valueOf(rateBytesPerSecond).divide(ONEG, SCALE, ROUND);
      BigDecimal R_eff;

      // bytes per I_MAX <= 1
      if (B.compareTo(B_MIN) <= 0) {
         B = BigDecimal.ONE;
         I = B.divide(R, SCALE, ROUND);

         BigDecimal Id = BigDecimal.ONE;
         while (true) {
            while (I.compareTo(I_MAX) > 0) {
               I = I.divide(E, SCALE, ROUND);
               Id = Id.multiply(E);
            }

            if (Id.compareTo(Id_MAX) > 0)
               throw new IllegalArgumentException("rate too low (" + rateBytesPerSecond + "): prescaler too high");

            BigDecimal B_int = BigDecimal.valueOf(B.setScale(0, ROUND).intValueExact());
            BigDecimal I_int = BigDecimal.valueOf(I.setScale(0, ROUND).intValueExact());
            BigDecimal Id_int = BigDecimal.valueOf(Id.setScale(0, ROUND).intValueExact());

            R_eff = B_int.divide(I_int, SCALE, ROUND).divide(Id_int, SCALE, ROUND);
            ERROR = R_eff.divide(R_set, SCALE, ROUND).subtract(BigDecimal.ONE).abs();
            if (ERROR.compareTo(ERROR_MAX) < 0)
               break;

            I = I.multiply(E);
            B = B.multiply(E);
         }

         return new Result(B.setScale(0, ROUND).intValueExact(), Id.setScale(0, ROUND).intValueExact(), I.setScale(0, ROUND).intValueExact());
      } else {
         BigDecimal Id = BigDecimal.ONE;

         while (true) {
            while (I.compareTo(I_MAX) > 0) {
               I = I.divide(E, SCALE, ROUND);
               Id = Id.multiply(E);
            }

            while (B.compareTo(B_MAX) > 0) {
               B = B.divide(TWO, SCALE, ROUND);
               I = I.divide(TWO, SCALE, ROUND);
            }

            if (I.compareTo(BigDecimal.ONE) < 0)
               throw new IllegalArgumentException("rate too high (" + rateBytesPerSecond + "): interval too low");

            BigDecimal B_int = BigDecimal.valueOf(B.setScale(0, ROUND).intValueExact());
            BigDecimal I_int = BigDecimal.valueOf(I.setScale(0, ROUND).intValueExact());
            BigDecimal Id_int = BigDecimal.valueOf(Id.setScale(0, ROUND).intValueExact());

            R_eff = B_int.divide(I_int, SCALE, ROUND).divide(Id_int, SCALE, ROUND);
            ERROR = R_eff.divide(R_set, SCALE, ROUND).subtract(BigDecimal.ONE).abs();
            if (ERROR.compareTo(ERROR_MAX) < 0)
               break;

            I = I.multiply(E);
            B = B.multiply(E);
         }

         if (I.compareTo(I_MIN) < 0)
            throw new IllegalArgumentException("rate too high (" + rateBytesPerSecond + "): interval too low");

         return new Result(B.setScale(0, ROUND).intValueExact(), Id.setScale(0, ROUND).intValueExact(), I.setScale(0, ROUND).intValueExact());
      }
   }

   public static class Result {
      private final int bytesPerTick;
      private final int prescale;
      private final int tickNanosecondInterval;

      public Result(int bytesPerTick, int prescale, int tickNanosecondInterval) {
         this.bytesPerTick = bytesPerTick;
         this.prescale = prescale;
         this.tickNanosecondInterval = tickNanosecondInterval;
      }

      public int getBytesPerTick() {
         return bytesPerTick;
      }

      public int getPrescale() {
         return prescale;
      }

      public int getTickNanosecondsInterval() {
         return tickNanosecondInterval;
      }
   }
}
