package ai.grakn.kgms;

import ai.grakn.GraknConfigKey;
import ai.grakn.engine.GraknConfig;
import ai.grakn.kgms.authentication.service.SignedToken;
import ai.grakn.kgms.console.AuthenticatedCommandBus;
import ai.grakn.kgms.console.AuthenticatedCommandBusImpl;
import ai.grakn.kgms.console.PrintWriterNotRecognised;
import ai.grakn.kgms.console.authentication.createuser.CreateUserCommand;
import ai.grakn.kgms.console.authentication.createuser.CreateUserDisplay;
import ai.grakn.kgms.console.authentication.deleteuser.DeleteUserCommand;
import ai.grakn.kgms.console.authentication.deleteuser.DeleteUserDisplay;
import ai.grakn.kgms.console.authentication.retrieveallusers.RetrieveAllUserCommand;
import ai.grakn.kgms.console.authentication.retrieveallusers.RetrieveAllUserDisplay;
import ai.grakn.kgms.console.authentication.retrieveuser.RetrieveUserCommand;
import ai.grakn.kgms.console.authentication.retrieveuser.RetrieveUserDisplay;
import ai.grakn.kgms.console.authentication.updateuser.UpdateUserCommand;
import ai.grakn.kgms.console.authentication.updateuser.UpdateUserDisplay;
import ai.grakn.kgms.rpc.GrpcClient;
import ai.grakn.util.SimpleURI;
import jline.console.ConsoleReader;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.CommandLineParser;
import org.apache.commons.cli.DefaultParser;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nullable;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.PrintWriter;
import java.nio.file.Path;

import static ai.grakn.GraknConfigKey.GRPC_PORT;

/**
 * KGMS Authenticated gRPC Shell that can be run from command line
 *
 * @author Marco Scoppetta
 */


public class KGMSClientShell {

    private static final Logger LOG = LoggerFactory.getLogger(KGMSClientShell.class);
    private static final GraknConfigKey<Path> KGMS_TRUST_STORE_PATH = GraknConfigKey.key("kgms.client.trust-store", GraknConfigKey.PATH);

    private static final String PROMPT = ">>> ";
    private SignedToken authToken;
    private PrintWriter outPrinter;

    private final ConsoleReader console;
    private final GrpcClient grpcClient;
    private final AuthenticatedCommandBus commandBus;

    public static void main(String[] args) {
        int exitCode = (startShell(args, System.in, System.out, GraknConfig.create())) ? 0 : 1;
        System.exit(exitCode);
    }

    public static boolean startShell(String[] args, InputStream in, OutputStream out, GraknConfig config) {
        Options options = new Options();

        options.addOption("u", "username", true, "username of admin user");
        options.addOption("p", "password", true, "password of admin user");
        options.addOption("c", "contact-point", true, "contact point for gRPC server");
        CommandLineParser parser = new DefaultParser();
        CommandLine cmd;
        try {
            cmd = parser.parse(options, args);
            int grpcPort;
            String grpcHost;
            if (cmd.hasOption('c')) {
                SimpleURI contactPoint = new SimpleURI(cmd.getOptionValue('c'));
                grpcHost = contactPoint.getHost();
                grpcPort = contactPoint.getPort();
            }else{
                grpcHost = config.getProperty(GraknConfigKey.SERVER_HOST_NAME);
                grpcPort = config.getProperty(GRPC_PORT);
            }
            GrpcClient grpcClient = new GrpcClient(grpcHost, grpcPort, config.getProperty(KGMS_TRUST_STORE_PATH));
            KGMSClientShell shell = new KGMSClientShell(in, out, grpcClient);
            String username = cmd.getOptionValue('u');
            String password = cmd.getOptionValue('p');
            shell.start(username, password);
        } catch (IOException | ParseException e){
            LOG.error("Unable to start KGMS console", e);
            return false;
        }

        return true;
    }


    KGMSClientShell(InputStream in, OutputStream out, GrpcClient grpcClient) throws IOException {
        this.console = new ConsoleReader(in, out);
        this.outPrinter = new PrintWriter(console.getOutput());
        this.grpcClient = grpcClient;
        this.commandBus = initialiseCommandBus();
    }

    void start(@Nullable String username, @Nullable String password) throws IOException {
        if (username == null) {
            username = console.readLine("Username: ");
        }
        if (password == null) {
            password = console.readLine("Password: ", '*');
        }
        authToken = grpcClient.authenticateUser(username, password);

        authenticatedShell();
    }

    private void authenticatedShell() throws IOException {
        console.setPrompt(PROMPT);
        String line;
        while ((line = console.readLine()) != null) {
            commandBus.run(line, authToken);
        }
    }

    private AuthenticatedCommandBus initialiseCommandBus() {
        return new AuthenticatedCommandBusImpl(new PrintWriterNotRecognised(this.outPrinter),
                new CreateUserCommand(grpcClient.createNewUser(), new CreateUserDisplay(this.outPrinter)),
                new RetrieveAllUserCommand(grpcClient.retrieveAllUser(), new RetrieveAllUserDisplay(this.outPrinter)),
                new DeleteUserCommand(grpcClient.deleteUser(), new DeleteUserDisplay(this.outPrinter)),
                new RetrieveUserCommand(grpcClient.retrieveUser(), new RetrieveUserDisplay(this.outPrinter)),
                new UpdateUserCommand(grpcClient.updateUser(), new UpdateUserDisplay(this.outPrinter)));
    }


}