/*
 * Configurate
 * Copyright (C) zml and Configurate contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.spongepowered.configurate;

import static java.util.Objects.requireNonNull;

import org.checkerframework.checker.nullness.qual.Nullable;

import java.util.Arrays;
import java.util.Iterator;

final class NodePathImpl implements NodePath {

    static final NodePath EMPTY = new NodePathImpl(new Object[0], false);

    private final Object[] arr;

    NodePathImpl(final Object[] arr, final boolean copy) {
        requireNonNull(arr);
        this.arr = copy ? Arrays.copyOf(arr, arr.length) : arr;
    }

    @Override
    public Object get(final int i) {
        return this.arr[i];
    }

    @Override
    public int size() {
        return this.arr.length;
    }

    @Override
    public NodePath withAppendedChild(final Object childKey) {
        requireNonNull(childKey, "childKey");
        final Object[] arr = this.arr;
        if (arr.length == 0) {
            return new NodePathImpl(new Object[] {childKey}, false);
        }

        final Object[] childPath = Arrays.copyOf(arr, arr.length + 1);
        childPath[childPath.length - 1] = childKey;

        return new NodePathImpl(childPath, false);
    }

    @Override
    public NodePath with(final int index, final Object value) throws IndexOutOfBoundsException {
        requireNonNull(value, "value");
        final Object[] arr = this.arr;
        if (index < 0 || index >= arr.length) {
            throw new IndexOutOfBoundsException("Index " + index + " is not within limit of [0," + arr.length + ")");
        }
        final Object[] newPath = Arrays.copyOf(arr, arr.length);
        newPath[index] = value;
        return new NodePathImpl(newPath, false);
    }

    @Override
    public NodePath plus(final NodePath other) {
        requireNonNull(other, "other");

        // Avoid copies for empty paths
        if (this.arr.length == 0) {
            return other;
        } else if (other.size() == 0) {
            return this;
        }

        final Object[] otherArr = (other instanceof NodePathImpl) ? ((NodePathImpl) other).arr : other.array();
        final Object[] result = new Object[this.arr.length + otherArr.length];
        System.arraycopy(this.arr, 0, result, 0, this.arr.length);
        System.arraycopy(otherArr, 0, result, this.arr.length, otherArr.length);
        return new NodePathImpl(result, false);
    }

    @Override
    public Object[] array() {
        return Arrays.copyOf(this.arr, this.arr.length);
    }

    @Override
    public Iterator<Object> iterator() {
        return Arrays.asList(this.arr).iterator();
    }

    @Override
    public NodePath copy() {
        return new NodePathImpl(this.arr, true);
    }

    @Override
    public boolean equals(final @Nullable Object other) {
        if (this == other) {
            return true;
        }

        if (other == null || getClass() != other.getClass()) {
            return false;
        }

        final NodePathImpl that = (NodePathImpl) other;
        return Arrays.equals(this.arr, that.arr);
    }

    @Override
    public int hashCode() {
        return Arrays.hashCode(this.arr);
    }

    @Override
    public String toString() {
        return Arrays.toString(this.arr);
    }

}
