/*
 * Configurate
 * Copyright (C) zml and Configurate contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package ninja.leaping.configurate.objectmapping.serialize;

import com.google.common.reflect.TypeToken;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.function.Function;
import java.util.function.Predicate;

import static java.util.Objects.requireNonNull;

/**
 * A calculated collection of {@link TypeSerializer}s
 */
public class TypeSerializerCollection {
    private final TypeSerializerCollection parent;
    private final SerializerList serializers = new SerializerList();
    private final Map<TypeToken<?>, TypeSerializer<?>> typeMatches = new ConcurrentHashMap<>();

    TypeSerializerCollection(TypeSerializerCollection parent) {
        this.parent = parent;
    }

    /**
     * Gets the default {@link TypeSerializer}s. While this collection is mutable, starting with
     * Configurate 4.0 type serializer collections will be immutable, so mutability should not be
     * relied on. Instead, a new child of this collection should be used to register any custom
     * serializers.
     *
     * @return The default serializers
     */
    public static TypeSerializerCollection defaults() {
        return TypeSerializers.DEFAULT_SERIALIZERS;
    }

    public static TypeSerializerCollection create() {
        return defaults().newChild();
    }

    @SuppressWarnings("unchecked")
    public <T> TypeSerializer<T> get(TypeToken<T> type) {
        type = requireNonNull(type).wrap();

        TypeSerializer<?> serial = typeMatches.computeIfAbsent(type, serializers);
        if (serial == null && parent != null) {
            serial = parent.get(type);
        }

        return (TypeSerializer) serial;
    }

    /**
     * Register a type serializer for a given type. Serializers registered will match all subclasses
     * of the provided type, as well as unwrapped primitive equivalents of the type.
     *
     * @param type       The type to accept
     * @param serializer The serializer that will be serialized with
     * @param <T>        The type to generify around
     * @return this
     * @deprecated Use #register(TypeToken, TypeSerializer) instead
     */
    @Deprecated
    public <T> TypeSerializerCollection registerType(TypeToken<T> type, TypeSerializer<? super T> serializer) {
        return register(type, serializer);

    }

    /**
     * Register a type serializer for a given type. Serializers registered will match all subclasses
     * of the provided type, as well as unwrapped primitive equivalents of the type.
     *
     * @param type       The type to accept
     * @param serializer The serializer that will be serialized with
     * @param <T>        The type to generify around
     * @return this
     */
    public <T> TypeSerializerCollection register(TypeToken<T> type, TypeSerializer<? super T> serializer) {
        serializers.add(new RegisteredSerializer(requireNonNull(type, "type"), requireNonNull(serializer)));
        typeMatches.clear();
        return this;
    }

    /**
     * Register a type serializer matching against a given predicate.
     *
     * @param test       The predicate to match types against
     * @param serializer The serializer to serialize matching types with
     * @param <T>        The type parameter
     * @return this
     * @deprecated Use {@link #register(Predicate, TypeSerializer)} instead
     */
    @Deprecated
    public <T> TypeSerializerCollection registerPredicate(Predicate<TypeToken<T>> test, TypeSerializer<? super T> serializer) {
        return register(test, serializer);
    }

    /**
     * Register a type serializer matching against a given predicate.
     *
     * @param test       The predicate to match types against
     * @param serializer The serializer to serialize matching types with
     * @param <T>        The type parameter
     * @return this
     */
    @SuppressWarnings("unchecked")
    public <T> TypeSerializerCollection register(Predicate<TypeToken<T>> test, TypeSerializer<? super T> serializer) {
        serializers.add(new RegisteredSerializer((Predicate) requireNonNull(test, "test"), requireNonNull(serializer, "serializer")));
        typeMatches.clear();
        return this;
    }

    /**
     * Register a scalar serializer under its appropriate type. Serializers registered will match all subclasses
     *
     * @param serializer The serializer that will be serialized with
     * @param <T>        The type to generify around
     * @return this
     */
    public <T> TypeSerializerCollection register(ScalarSerializer<T> serializer) {
        return register(serializer.type(), serializer);
    }

    public TypeSerializerCollection newChild() {
        return new TypeSerializerCollection(this);
    }

    private static final class SerializerList extends CopyOnWriteArrayList<RegisteredSerializer> implements Function<TypeToken<?>, TypeSerializer<?>> {

        @Override
        public TypeSerializer<?> apply(TypeToken<?> type) {
            for (RegisteredSerializer ent : this) {
                if (ent.predicate.test(type)) {
                    return ent.serializer;
                }
            }
            return null;
        }
    }

}
