/*
 * Decompiled with CFR 0.152.
 */
package org.nd4j.linalg.util;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.math.MathContext;
import java.security.ProviderException;
import org.nd4j.linalg.util.Bernoulli;
import org.nd4j.linalg.util.Factorial;
import org.nd4j.linalg.util.Rational;

public class BigDecimalMath {
    static BigDecimal E = new BigDecimal("2.71828182845904523536028747135266249775724709369995957496696762772407663035354759457138217852516642742746639193200305992181741359662904357290033429526059563073813232862794349076323382988075319525101901157383418793070215408914993488416750924476146066808226480016847741185374234544243710753907774499206955170276183860626133138458300075204493382656029760673711320070932870912744374704723069697720931014169283681902551510865746377211125238978442505695369677078544996996794686445490598793163688923009879312773617821542499922957635148220826989519366803318252886939849646510582093923982948879332036250944311730123819706841614039701983767932068328237646480429531180232878250981945581530175671736133206981125099618188159304169035159888851934580727386673858942287922849989208680582574927961048419844436346324496848756023362482704197862320900216099023530436994184914631409343173814364054625315209618369088870701676839642437814059271456354906130310720851038375051011574770417189861068739696552126715468895703503540212340784981933432106817012100562788023519303322474501585390473041995777709350366041699732972508868769664035557071622684471625607988265178713419512466520103059212366771943252786753985589448969709640975459185695638023637016211204774272283648961342251644507818244235294863637214174023889344124796357437026375529444833799801612549227850925778256209262264832627793338656648162772516401910590049164499828931");
    static BigDecimal PI = new BigDecimal("3.14159265358979323846264338327950288419716939937510582097494459230781640628620899862803482534211706798214808651328230664709384460955058223172535940812848111745028410270193852110555964462294895493038196442881097566593344612847564823378678316527120190914564856692346034861045432664821339360726024914127372458700660631558817488152092096282925409171536436789259036001133053054882046652138414695194151160943305727036575959195309218611738193261179310511854807446237996274956735188575272489122793818301194912983367336244065664308602139494639522473719070217986094370277053921717629317675238467481846766940513200056812714526356082778577134275778960917363717872146844090122495343014654958537105079227968925892354201995611212902196086403441815981362977477130996051870721134999999837297804995105973173281609631859502445945534690830264252230825334468503526193118817101000313783875288658753320838142061717766914730359825349042875546873115956286388235378759375195778185778053217122680661300192787661119590921642019893809525720106548586327886593615338182796823030195203530185296899577362259941389124972177528347913151557485724245415069595082953311686172785588907509838175463746493931925506040092770167113900984882401285836160356370766010471018194295559619894676783744944825537977472684710404753464620804668425906949129331367702898915210475216205696602405803815019351125338243003558764024749647326391419927260426992279678235478163600934172164121992458631503028618297455570674983850549458858692699569092721079750930295532116534498720275596023648066549911988183479775356636980742654252786255181841757467289097777279380008164706001614524919217321721477235014");
    static BigDecimal GAMMA = new BigDecimal("0.57721566490153286060651209008240243104215933593992359880576723488486772677766467093694706329174674951463144724980708248096050401448654283622417399764492353625350033374293733773767394279259525824709491600873520394816567085323315177661152862119950150798479374508570574002992135478614669402960432542151905877553526733139925401296742051375413954911168510280798423487758720503843109399736137255306088933126760017247953783675927135157722610273492913940798430103417771778088154957066107501016191663340152278935867965497252036212879226555953669628176388792726801324310104765059637039473949576389065729679296010090151251959509222435014093498712282479497471956469763185066761290638110518241974448678363808617494551698927923018773910729457815543160050021828440960537724342032854783670151773943987003023703395183286900015581939880427074115422278197165230110735658339673487176504919418123000406546931429992977795693031005030863034185698032310836916400258929708909854868257773642882539549258736295961332985747393023734388470703702844129201664178502487333790805627549984345907616431671031467107223700218107450444186647591348036690255324586254422253451813879124345735013612977822782881489459098638460062931694718871495875254923664935204732436410972682761608775950880951262084045444779922991572482925162512784276596570832146102982146179519579590959227042089896279712553632179488737642106606070659825619901028807561251991375116782176436190570584407835735015800560774579342131449885007864151716151945");
    static BigDecimal LOG2 = new BigDecimal("0.69314718055994530941723212145817656807550013436025525412068000949339362196969471560586332699641868754200148102057068573368552023575813055703267075163507596193072757082837143519030703862389167347112335011536449795523912047517268157493206515552473413952588295045300709532636664265410423915781495204374043038550080194417064167151864471283996817178454695702627163106454615025720740248163777338963855069526066834113727387372292895649354702576265209885969320196505855476470330679365443254763274495125040606943814710468994650622016772042452452961268794654619316517468139267250410380254625965686914419287160829380317271436778265487756648508567407764845146443994046142260319309673540257444607030809608504748663852313818167675143866747664789088143714198549423151997354880375165861275352916610007105355824987941472950929311389715599820565439287170007218085761025236889213244971389320378439353088774825970171559107088236836275898425891853530243634214367061189236789192372314672321720534016492568727477823445353476481149418642386776774406069562657379600867076257199184734022651462837904883062033061144630073719489002743643965002580936519443041191150608094879306786515887090060520346842973619384128965255653968602219412292420757432175748909770675268711581705113700915894266547859596489065305846025866838294002283300538207400567705304678700184162404418833232798386349001563121889560650553151272199398332030751408426091479001265168243443893572472788205486271552741877243002489794540196187233980860831664811490930667519339312890431641370681397776498176974868903887789991296503619270710889264105230924783917373501229842420499568935992206602204654941510613");
    private static int TAYLOR_NTERM = 8;

    public static BigDecimal pi(MathContext mc) {
        if (mc.getPrecision() < PI.precision()) {
            return PI.round(mc);
        }
        int[] a = new int[]{1, 0, 0, -1, -1, -1, 0, 0};
        BigDecimal S = BigDecimalMath.broadhurstBBP(1, 1, a, mc);
        return BigDecimalMath.multiplyRound(S, 8);
    }

    public static BigDecimal gamma(MathContext mc) {
        if (mc.getPrecision() < GAMMA.precision()) {
            return GAMMA.round(mc);
        }
        double eps = BigDecimalMath.prec2err(0.577, mc.getPrecision());
        MathContext mcloc = new MathContext(2 + mc.getPrecision());
        BigDecimal resul = BigDecimal.ONE;
        resul = resul.add(BigDecimalMath.log(2, mcloc));
        resul = resul.subtract(BigDecimalMath.log(3, mcloc));
        int kmax = (int)((Math.log(eps / 0.7) - 2.0) / 4.0);
        mcloc = new MathContext(1 + BigDecimalMath.err2prec(1.2, eps / (double)kmax));
        int n = 1;
        while (true) {
            BigDecimal c = BigDecimalMath.zeta(2 * n + 1, mcloc).subtract(BigDecimal.ONE);
            BigInteger fourn = BigInteger.valueOf(2 * n + 1);
            fourn = fourn.shiftLeft(2 * n);
            c = BigDecimalMath.divideRound(c, fourn);
            resul = resul.subtract(c);
            if (c.doubleValue() < 0.1 * eps) break;
            ++n;
        }
        return resul.round(mc);
    }

    public static BigDecimal sqrt(BigDecimal x) {
        if (x.compareTo(BigDecimal.ZERO) < 0) {
            throw new ArithmeticException("negative argument " + x.toString() + " of square root");
        }
        return BigDecimalMath.root(2, x);
    }

    public static BigDecimal cbrt(BigDecimal x) {
        if (x.compareTo(BigDecimal.ZERO) < 0) {
            return BigDecimalMath.root(3, x.negate()).negate();
        }
        return BigDecimalMath.root(3, x);
    }

    public static BigDecimal root(int n, BigDecimal x) {
        BigDecimal c;
        if (x.compareTo(BigDecimal.ZERO) < 0) {
            throw new ArithmeticException("negative argument " + x.toString() + " of root");
        }
        if (n <= 0) {
            throw new ArithmeticException("negative power " + n + " of root");
        }
        if (n == 1) {
            return x;
        }
        BigDecimal s = new BigDecimal(Math.pow(x.doubleValue(), 1.0 / (double)n));
        BigDecimal nth = new BigDecimal(n);
        BigDecimal xhighpr = BigDecimalMath.scalePrec(x, 2);
        MathContext mc = new MathContext(2 + x.precision());
        double eps = x.ulp().doubleValue() / ((double)(2 * n) * x.doubleValue());
        do {
            c = xhighpr.divide(s.pow(n - 1), mc);
            c = s.subtract(c);
            MathContext locmc = new MathContext(c.precision());
            c = c.divide(nth, locmc);
            s = s.subtract(c);
        } while (!(Math.abs(c.doubleValue() / s.doubleValue()) < eps));
        return s.round(new MathContext(BigDecimalMath.err2prec(eps)));
    }

    public static BigDecimal hypot(BigDecimal x, BigDecimal y) {
        BigDecimal z = x.pow(2).add(y.pow(2));
        BigDecimal zerr = x.abs().multiply(x.ulp()).add(y.abs().multiply(y.ulp()));
        MathContext mc = new MathContext(2 + BigDecimalMath.err2prec(z, zerr));
        z = BigDecimalMath.sqrt(z.round(mc));
        mc = new MathContext(BigDecimalMath.err2prec(z.doubleValue(), 0.5 * zerr.doubleValue() / z.doubleValue()));
        return z.round(mc);
    }

    public static BigDecimal hypot(int n, BigDecimal x) {
        BigDecimal z = new BigDecimal(n).pow(2).add(x.pow(2));
        double zerr = x.doubleValue() * x.ulp().doubleValue();
        MathContext mc = new MathContext(2 + BigDecimalMath.err2prec(z.doubleValue(), zerr));
        z = BigDecimalMath.sqrt(z.round(mc));
        mc = new MathContext(BigDecimalMath.err2prec(z.doubleValue(), 0.5 * zerr / z.doubleValue()));
        return z.round(mc);
    }

    public static BigDecimal exp(BigDecimal x) {
        if (x.compareTo(BigDecimal.ZERO) < 0) {
            BigDecimal invx = BigDecimalMath.exp(x.negate());
            MathContext mc = new MathContext(invx.precision());
            return BigDecimal.ONE.divide(invx, mc);
        }
        if (x.compareTo(BigDecimal.ZERO) == 0) {
            return BigDecimalMath.scalePrec(BigDecimal.ONE, -((int)Math.log10(x.ulp().doubleValue())));
        }
        double xDbl = x.doubleValue();
        double xUlpDbl = x.ulp().doubleValue();
        if (Math.pow(xDbl, TAYLOR_NTERM) < (double)TAYLOR_NTERM * ((double)TAYLOR_NTERM - 1.0) * ((double)TAYLOR_NTERM - 2.0) * xUlpDbl) {
            BigDecimal resul = BigDecimal.ONE;
            BigDecimal xpowi = BigDecimal.ONE;
            BigInteger ifac = BigInteger.ONE;
            MathContext mcTay = new MathContext(BigDecimalMath.err2prec(1.0, xUlpDbl / (double)TAYLOR_NTERM));
            for (int i = 1; i <= TAYLOR_NTERM; ++i) {
                ifac = ifac.multiply(BigInteger.valueOf(i));
                xpowi = xpowi.multiply(x);
                BigDecimal c = xpowi.divide(new BigDecimal(ifac), mcTay);
                resul = resul.add(c);
                if (Math.abs(xpowi.doubleValue()) < (double)i && Math.abs(c.doubleValue()) < 0.5 * xUlpDbl) break;
            }
            MathContext mc = new MathContext(BigDecimalMath.err2prec(xUlpDbl / 2.0));
            return resul.round(mc);
        }
        int exSc = (int)(1.0 - Math.log10((double)TAYLOR_NTERM * ((double)TAYLOR_NTERM - 1.0) * ((double)TAYLOR_NTERM - 2.0) * xUlpDbl / Math.pow(xDbl, TAYLOR_NTERM)) / ((double)TAYLOR_NTERM - 1.0));
        BigDecimal xby10 = x.scaleByPowerOfTen(-exSc);
        BigDecimal expxby10 = BigDecimalMath.exp(xby10);
        MathContext mc = new MathContext(expxby10.precision() - exSc);
        while (exSc > 0) {
            int exsub = Math.min(8, exSc);
            exSc -= exsub;
            MathContext mctmp = new MathContext(expxby10.precision() - exsub + 2);
            int pex = 1;
            while (exsub-- > 0) {
                pex *= 10;
            }
            expxby10 = expxby10.pow(pex, mctmp);
        }
        return expxby10.round(mc);
    }

    public static BigDecimal exp(MathContext mc) {
        if (mc.getPrecision() < E.precision()) {
            return E.round(mc);
        }
        BigDecimal uni = BigDecimalMath.scalePrec(BigDecimal.ONE, mc.getPrecision());
        return BigDecimalMath.exp(uni);
    }

    public static BigDecimal log(BigDecimal x) {
        if (x.compareTo(BigDecimal.ZERO) < 0) {
            throw new ArithmeticException("Cannot take log of negative " + x.toString());
        }
        if (x.compareTo(BigDecimal.ONE) == 0) {
            return BigDecimalMath.scalePrec(BigDecimal.ZERO, x.precision() - 1);
        }
        if (Math.abs(x.doubleValue() - 1.0) <= 0.3) {
            BigDecimal z;
            BigDecimal zpown = z = BigDecimalMath.scalePrec(x.subtract(BigDecimal.ONE), 2);
            double eps = 0.5 * x.ulp().doubleValue() / Math.abs(x.doubleValue());
            BigDecimal resul = z;
            int k = 2;
            while (true) {
                zpown = BigDecimalMath.multiplyRound(zpown, z);
                BigDecimal c = BigDecimalMath.divideRound(zpown, k);
                resul = k % 2 == 0 ? resul.subtract(c) : resul.add(c);
                if (Math.abs(c.doubleValue()) < eps) break;
                ++k;
            }
            MathContext mc = new MathContext(BigDecimalMath.err2prec(resul.doubleValue(), eps));
            return resul.round(mc);
        }
        double xDbl = x.doubleValue();
        double xUlpDbl = x.ulp().doubleValue();
        int r = (int)(Math.log(xDbl) / 0.2);
        r = Math.max(2, r);
        BigDecimal xhighpr = BigDecimalMath.scalePrec(x, 2);
        BigDecimal resul = BigDecimalMath.root(r, xhighpr);
        resul = BigDecimalMath.log(resul).multiply(new BigDecimal(r));
        MathContext mc = new MathContext(BigDecimalMath.err2prec(resul.doubleValue(), xUlpDbl / xDbl));
        return resul.round(mc);
    }

    public static BigDecimal log(int n, MathContext mc) {
        if (n <= 0) {
            throw new ArithmeticException("Cannot take log of negative " + n);
        }
        if (n == 1) {
            return BigDecimal.ZERO;
        }
        if (n == 2) {
            if (mc.getPrecision() < LOG2.precision()) {
                return LOG2.round(mc);
            }
            int[] a = new int[]{2, -5, -2, -7, -2, -5, 2, -3};
            BigDecimal S = BigDecimalMath.broadhurstBBP(2, 1, a, new MathContext(1 + mc.getPrecision()));
            S = S.multiply(new BigDecimal(8));
            S = BigDecimalMath.sqrt(BigDecimalMath.divideRound(S, 3));
            return S.round(mc);
        }
        if (n == 3) {
            Rational tmp;
            int kmax = (int)((double)mc.getPrecision() / 1.87);
            MathContext mcloc = new MathContext(mc.getPrecision() + 1 + (int)Math.log10((double)kmax * 0.693 / 1.098));
            BigDecimal log3 = BigDecimalMath.multiplyRound(BigDecimalMath.log(2, mcloc), 19);
            double eps = BigDecimalMath.prec2err(1.098, mc.getPrecision()) / (double)kmax;
            Rational r = new Rational(7153, 524288);
            Rational pk = new Rational(7153, 524288);
            int k = 1;
            while (!((tmp = pk.divide(k)).doubleValue() < eps)) {
                mcloc = new MathContext(BigDecimalMath.err2prec(tmp.doubleValue(), eps));
                BigDecimal c = pk.divide(k).BigDecimalValue(mcloc);
                log3 = k % 2 != 0 ? log3.add(c) : log3.subtract(c);
                pk = pk.multiply(r);
                ++k;
            }
            log3 = BigDecimalMath.divideRound(log3, 12);
            return log3.round(mc);
        }
        if (n == 5) {
            Rational tmp;
            int kmax = (int)((double)mc.getPrecision() / 1.33);
            MathContext mcloc = new MathContext(mc.getPrecision() + 1 + (int)Math.log10((double)kmax * 0.693 / 1.609));
            BigDecimal log5 = BigDecimalMath.multiplyRound(BigDecimalMath.log(2, mcloc), 14);
            double eps = BigDecimalMath.prec2err(1.6, mc.getPrecision()) / (double)kmax;
            Rational r = new Rational(759, 16384);
            Rational pk = new Rational(759, 16384);
            int k = 1;
            while (!((tmp = pk.divide(k)).doubleValue() < eps)) {
                mcloc = new MathContext(BigDecimalMath.err2prec(tmp.doubleValue(), eps));
                BigDecimal c = pk.divide(k).BigDecimalValue(mcloc);
                log5 = log5.subtract(c);
                pk = pk.multiply(r);
                ++k;
            }
            log5 = BigDecimalMath.divideRound(log5, 6);
            return log5.round(mc);
        }
        if (n == 7) {
            Rational tmp;
            int kmax = (int)((double)mc.getPrecision() / 0.903);
            MathContext mcloc = new MathContext(mc.getPrecision() + 1 + (int)Math.log10((double)(kmax * 3) * 0.693 / 1.098));
            BigDecimal log7 = BigDecimalMath.multiplyRound(BigDecimalMath.log(2, mcloc), 3);
            double eps = BigDecimalMath.prec2err(1.9, mc.getPrecision()) / (double)kmax;
            Rational r = new Rational(1, 8);
            Rational pk = new Rational(1, 8);
            int k = 1;
            while (!((tmp = pk.divide(k)).doubleValue() < eps)) {
                mcloc = new MathContext(BigDecimalMath.err2prec(tmp.doubleValue(), eps));
                BigDecimal c = pk.divide(k).BigDecimalValue(mcloc);
                log7 = log7.subtract(c);
                pk = pk.multiply(r);
                ++k;
            }
            return log7.round(mc);
        }
        double res = Math.log(n);
        double eps = BigDecimalMath.prec2err(res, mc.getPrecision());
        MathContext mcloc = new MathContext(1 + BigDecimalMath.err2prec(n, eps *= (double)n));
        BigDecimal nb = BigDecimalMath.scalePrec(new BigDecimal(n), mcloc);
        return BigDecimalMath.log(nb);
    }

    public static BigDecimal log(Rational r, MathContext mc) {
        if (r.compareTo(Rational.ZERO) <= 0) {
            throw new ArithmeticException("Cannot take log of negative " + r.toString());
        }
        if (r.compareTo(Rational.ONE) == 0) {
            return BigDecimal.ZERO;
        }
        double eps = BigDecimalMath.prec2err(Math.log(r.doubleValue()), mc.getPrecision());
        MathContext mcloc = new MathContext(1 + BigDecimalMath.err2prec(eps));
        BigDecimal resul = BigDecimalMath.log(r.BigDecimalValue(mcloc));
        return resul.round(mc);
    }

    public static BigDecimal pow(BigDecimal x, BigDecimal y) {
        if (x.compareTo(BigDecimal.ZERO) < 0) {
            throw new ArithmeticException("Cannot power negative " + x.toString());
        }
        if (x.compareTo(BigDecimal.ZERO) == 0) {
            return BigDecimal.ZERO;
        }
        BigDecimal logx = BigDecimalMath.log(x);
        BigDecimal ylogx = y.multiply(logx);
        BigDecimal resul = BigDecimalMath.exp(ylogx);
        double errR = Math.abs(logx.doubleValue() * y.ulp().doubleValue() / 2.0) + Math.abs(y.doubleValue() * x.ulp().doubleValue() / 2.0 / x.doubleValue());
        MathContext mcR = new MathContext(BigDecimalMath.err2prec(1.0, errR));
        return resul.round(mcR);
    }

    public static BigDecimal powRound(BigDecimal x, int n) {
        MathContext mc = new MathContext(x.precision() - (int)Math.log10(Math.abs(n)));
        return x.pow(n, mc);
    }

    public static BigDecimal sin(BigDecimal x) {
        if (x.compareTo(BigDecimal.ZERO) < 0) {
            return BigDecimalMath.sin(x.negate()).negate();
        }
        if (x.compareTo(BigDecimal.ZERO) == 0) {
            return BigDecimal.ZERO;
        }
        BigDecimal res = BigDecimalMath.mod2pi(x);
        double errpi = 0.5 * Math.abs(x.ulp().doubleValue());
        int val = 2 + BigDecimalMath.err2prec(Math.PI, errpi);
        MathContext mc = new MathContext(val);
        BigDecimal p = BigDecimalMath.pi(mc);
        mc = new MathContext(x.precision());
        if (res.compareTo(p) > 0) {
            return BigDecimalMath.sin(BigDecimalMath.subtractRound(res, p)).negate();
        }
        if (res.multiply(new BigDecimal(2)).compareTo(p) > 0) {
            return BigDecimalMath.sin(BigDecimalMath.subtractRound(p, res));
        }
        if (res.multiply(new BigDecimal(4)).compareTo(p) > 0) {
            return BigDecimalMath.cos(BigDecimalMath.subtractRound(p.divide(new BigDecimal(2)), res));
        }
        BigDecimal resul = res;
        BigDecimal xpowi = res;
        BigInteger ifac = BigInteger.ONE;
        double xUlpDbl = res.ulp().doubleValue();
        int k = (int)((double)res.precision() / Math.log10(1.0 / res.doubleValue())) / 2;
        MathContext mcTay = new MathContext(BigDecimalMath.err2prec(res.doubleValue(), xUlpDbl / (double)k));
        int i = 1;
        while (true) {
            ifac = ifac.multiply(BigInteger.valueOf(2 * i));
            ifac = ifac.multiply(BigInteger.valueOf(2 * i + 1));
            xpowi = xpowi.multiply(res).multiply(res).negate();
            BigDecimal corr = xpowi.divide(new BigDecimal(ifac), mcTay);
            resul = resul.add(corr);
            if (corr.abs().doubleValue() < 0.5 * xUlpDbl) break;
            ++i;
        }
        mc = new MathContext(res.precision());
        return resul.round(mc);
    }

    public static BigDecimal cos(BigDecimal x) {
        if (x.compareTo(BigDecimal.ZERO) < 0) {
            return BigDecimalMath.cos(x.negate());
        }
        if (x.compareTo(BigDecimal.ZERO) == 0) {
            return BigDecimal.ONE;
        }
        BigDecimal res = BigDecimalMath.mod2pi(x);
        double errpi = 0.5 * Math.abs(x.ulp().doubleValue());
        int val = BigDecimalMath.err2prec(Math.PI, errpi);
        MathContext mc = new MathContext(val);
        BigDecimal p = BigDecimalMath.pi(mc);
        mc = new MathContext(x.precision());
        if (res.compareTo(p) > 0) {
            return BigDecimalMath.cos(BigDecimalMath.subtractRound(res, p)).negate();
        }
        if (res.multiply(new BigDecimal(2)).compareTo(p) > 0) {
            return BigDecimalMath.cos(BigDecimalMath.subtractRound(p, res)).negate();
        }
        if (res.multiply(new BigDecimal(4)).compareTo(p) > 0) {
            return BigDecimalMath.sin(BigDecimalMath.subtractRound(p.divide(new BigDecimal(2)), res));
        }
        BigDecimal resul = BigDecimal.ONE;
        BigDecimal xpowi = BigDecimal.ONE;
        BigInteger ifac = BigInteger.ONE;
        double xUlpDbl = 0.5 * res.ulp().doubleValue() * res.doubleValue();
        int k = (int)(Math.log(xUlpDbl) / Math.log(res.doubleValue())) / 2;
        MathContext mcTay = new MathContext(BigDecimalMath.err2prec(1.0, xUlpDbl / (double)k));
        int i = 1;
        while (true) {
            ifac = ifac.multiply(BigInteger.valueOf(2 * i - 1));
            ifac = ifac.multiply(BigInteger.valueOf(2 * i));
            xpowi = xpowi.multiply(res).multiply(res).negate();
            BigDecimal corr = xpowi.divide(new BigDecimal(ifac), mcTay);
            resul = resul.add(corr);
            if (corr.abs().doubleValue() < 0.5 * xUlpDbl) break;
            ++i;
        }
        mc = new MathContext(BigDecimalMath.err2prec(resul.doubleValue(), xUlpDbl));
        return resul.round(mc);
    }

    public static BigDecimal tan(BigDecimal x) {
        if (x.compareTo(BigDecimal.ZERO) == 0) {
            return BigDecimal.ZERO;
        }
        if (x.compareTo(BigDecimal.ZERO) < 0) {
            return BigDecimalMath.tan(x.negate()).negate();
        }
        BigDecimal res = BigDecimalMath.modpi(x);
        double xDbl = res.doubleValue();
        double xUlpDbl = x.ulp().doubleValue() / 2.0;
        double eps = xUlpDbl / 2.0 / Math.pow(Math.cos(xDbl), 2.0);
        if (xDbl > 0.8) {
            BigDecimal co = BigDecimalMath.cot(x);
            MathContext mc = new MathContext(BigDecimalMath.err2prec(1.0 / co.doubleValue(), eps));
            return BigDecimal.ONE.divide(co, mc);
        }
        BigDecimal xhighpr = BigDecimalMath.scalePrec(res, 2);
        BigDecimal xhighprSq = BigDecimalMath.multiplyRound(xhighpr, xhighpr);
        BigDecimal result = xhighpr.plus();
        BigDecimal xpowi = xhighpr;
        Bernoulli b = new Bernoulli();
        BigInteger fourn = BigInteger.valueOf(4L);
        BigInteger fac = BigInteger.valueOf(2L);
        int i = 2;
        while (true) {
            Rational f = b.at(2 * i).abs();
            fourn = fourn.shiftLeft(2);
            fac = fac.multiply(BigInteger.valueOf(2 * i)).multiply(BigInteger.valueOf(2 * i - 1));
            f = f.multiply(fourn).multiply(fourn.subtract(BigInteger.ONE)).divide(fac);
            xpowi = BigDecimalMath.multiplyRound(xpowi, xhighprSq);
            BigDecimal c = BigDecimalMath.multiplyRound(xpowi, f);
            result = result.add(c);
            if (Math.abs(c.doubleValue()) < 0.1 * eps) break;
            ++i;
        }
        MathContext mc = new MathContext(BigDecimalMath.err2prec(result.doubleValue(), eps));
        return result.round(mc);
    }

    public static BigDecimal cot(BigDecimal x) {
        if (x.compareTo(BigDecimal.ZERO) == 0) {
            throw new ArithmeticException("Cannot take cot of zero " + x.toString());
        }
        if (x.compareTo(BigDecimal.ZERO) < 0) {
            return BigDecimalMath.cot(x.negate()).negate();
        }
        BigDecimal res = BigDecimalMath.modpi(x);
        double xDbl = res.doubleValue();
        double xUlpDbl = x.ulp().doubleValue() / 2.0;
        double eps = xUlpDbl / 2.0 / Math.pow(Math.sin(xDbl), 2.0);
        BigDecimal xhighpr = BigDecimalMath.scalePrec(res, 2);
        BigDecimal xhighprSq = BigDecimalMath.multiplyRound(xhighpr, xhighpr);
        MathContext mc = new MathContext(BigDecimalMath.err2prec(xhighpr.doubleValue(), eps));
        BigDecimal resul = BigDecimal.ONE.divide(xhighpr, mc);
        BigDecimal xpowi = xhighpr;
        Bernoulli b = new Bernoulli();
        BigInteger fourn = BigInteger.valueOf(4L);
        BigInteger fac = BigInteger.ONE;
        int i = 1;
        while (true) {
            Rational f = b.at(2 * i);
            fac = fac.multiply(BigInteger.valueOf(2 * i)).multiply(BigInteger.valueOf(2 * i - 1));
            f = f.multiply(fourn).divide(fac);
            BigDecimal c = BigDecimalMath.multiplyRound(xpowi, f);
            resul = i % 2 == 0 ? resul.add(c) : resul.subtract(c);
            if (Math.abs(c.doubleValue()) < 0.1 * eps) break;
            fourn = fourn.shiftLeft(2);
            xpowi = BigDecimalMath.multiplyRound(xpowi, xhighprSq);
            ++i;
        }
        mc = new MathContext(BigDecimalMath.err2prec(resul.doubleValue(), eps));
        return resul.round(mc);
    }

    public static BigDecimal asin(BigDecimal x) {
        if (x.compareTo(BigDecimal.ONE) > 0 || x.compareTo(BigDecimal.ONE.negate()) < 0) {
            throw new ArithmeticException("Out of range argument " + x.toString() + " of asin");
        }
        if (x.compareTo(BigDecimal.ZERO) == 0) {
            return BigDecimal.ZERO;
        }
        if (x.compareTo(BigDecimal.ONE) == 0) {
            double errpi = Math.sqrt(x.ulp().doubleValue());
            MathContext mc = new MathContext(BigDecimalMath.err2prec(3.14159, errpi));
            return BigDecimalMath.pi(mc).divide(new BigDecimal(2));
        }
        if (x.compareTo(BigDecimal.ZERO) < 0) {
            return BigDecimalMath.asin(x.negate()).negate();
        }
        if (x.doubleValue() > 0.7) {
            BigDecimal xCompl = BigDecimal.ONE.subtract(x);
            double xDbl = x.doubleValue();
            double xUlpDbl = x.ulp().doubleValue() / 2.0;
            double eps = xUlpDbl / 2.0 / Math.sqrt(1.0 - Math.pow(xDbl, 2.0));
            BigDecimal xhighpr = BigDecimalMath.scalePrec(xCompl, 3);
            BigDecimal xhighprV = BigDecimalMath.divideRound(xhighpr, 4);
            BigDecimal resul = BigDecimal.ONE;
            BigDecimal xpowi = BigDecimal.ONE;
            BigInteger ifacN = BigInteger.ONE;
            BigInteger ifacD = BigInteger.ONE;
            int i = 1;
            while (true) {
                ifacN = ifacN.multiply(BigInteger.valueOf(2 * i - 1));
                ifacD = ifacD.multiply(BigInteger.valueOf(i));
                xpowi = i == 1 ? xhighprV : BigDecimalMath.multiplyRound(xpowi, xhighprV);
                BigDecimal c = BigDecimalMath.divideRound(BigDecimalMath.multiplyRound(xpowi, ifacN), ifacD.multiply(BigInteger.valueOf(2 * i + 1)));
                resul = resul.add(c);
                if (Math.abs(c.doubleValue()) < xUlpDbl / 120.0) break;
                ++i;
            }
            xpowi = BigDecimalMath.sqrt(xhighpr.multiply(new BigDecimal(2)));
            resul = BigDecimalMath.multiplyRound(xpowi, resul);
            MathContext mc = new MathContext(resul.precision());
            BigDecimal pihalf = BigDecimalMath.pi(mc).divide(new BigDecimal(2));
            mc = new MathContext(BigDecimalMath.err2prec(resul.doubleValue(), eps));
            return pihalf.subtract(resul, mc);
        }
        double xDbl = x.doubleValue();
        double xUlpDbl = x.ulp().doubleValue() / 2.0;
        double eps = xUlpDbl / 2.0 / Math.sqrt(1.0 - Math.pow(xDbl, 2.0));
        BigDecimal xhighpr = BigDecimalMath.scalePrec(x, 2);
        BigDecimal xhighprSq = BigDecimalMath.multiplyRound(xhighpr, xhighpr);
        BigDecimal resul = xhighpr.plus();
        BigDecimal xpowi = xhighpr;
        BigInteger ifacN = BigInteger.ONE;
        BigInteger ifacD = BigInteger.ONE;
        int i = 1;
        while (true) {
            ifacN = ifacN.multiply(BigInteger.valueOf(2 * i - 1));
            ifacD = ifacD.multiply(BigInteger.valueOf(2 * i));
            xpowi = BigDecimalMath.multiplyRound(xpowi, xhighprSq);
            BigDecimal c = BigDecimalMath.divideRound(BigDecimalMath.multiplyRound(xpowi, ifacN), ifacD.multiply(BigInteger.valueOf(2 * i + 1)));
            resul = resul.add(c);
            if (Math.abs(c.doubleValue()) < 0.1 * eps) break;
            ++i;
        }
        MathContext mc = new MathContext(BigDecimalMath.err2prec(resul.doubleValue(), eps));
        return resul.round(mc);
    }

    public static BigDecimal atan(BigDecimal x) {
        if (x.compareTo(BigDecimal.ZERO) < 0) {
            return BigDecimalMath.atan(x.negate()).negate();
        }
        if (x.compareTo(BigDecimal.ZERO) == 0) {
            return BigDecimal.ZERO;
        }
        if (x.doubleValue() > 0.7 && x.doubleValue() < 3.0) {
            BigDecimal y = BigDecimalMath.scalePrec(x, 2);
            BigDecimal newx = BigDecimalMath.divideRound(BigDecimalMath.hypot(1, y).subtract(BigDecimal.ONE), y);
            BigDecimal resul = BigDecimalMath.multiplyRound(BigDecimalMath.atan(newx), 2);
            double eps = x.ulp().doubleValue() / (2.0 * Math.hypot(1.0, x.doubleValue()));
            MathContext mc = new MathContext(BigDecimalMath.err2prec(resul.doubleValue(), eps));
            return resul.round(mc);
        }
        if (x.doubleValue() < 0.71) {
            BigDecimal xhighpr = BigDecimalMath.scalePrec(x, 2);
            BigDecimal xhighprSq = BigDecimalMath.multiplyRound(xhighpr, xhighpr).negate();
            BigDecimal resul = xhighpr.plus();
            BigDecimal xpowi = xhighpr;
            double eps = x.ulp().doubleValue() / (2.0 * Math.hypot(1.0, x.doubleValue()));
            int i = 1;
            while (true) {
                xpowi = BigDecimalMath.multiplyRound(xpowi, xhighprSq);
                BigDecimal c = BigDecimalMath.divideRound(xpowi, 2 * i + 1);
                resul = resul.add(c);
                if (Math.abs(c.doubleValue()) < 0.1 * eps) break;
                ++i;
            }
            MathContext mc = new MathContext(BigDecimalMath.err2prec(resul.doubleValue(), eps));
            return resul.round(mc);
        }
        double eps = x.ulp().doubleValue() / (2.0 * Math.hypot(1.0, x.doubleValue()));
        MathContext mc = new MathContext(2 + BigDecimalMath.err2prec(3.1416, eps));
        BigDecimal onepi = BigDecimalMath.pi(mc);
        BigDecimal resul = onepi.divide(new BigDecimal(2));
        BigDecimal xhighpr = BigDecimalMath.divideRound(-1, BigDecimalMath.scalePrec(x, 2));
        BigDecimal xhighprSq = BigDecimalMath.multiplyRound(xhighpr, xhighpr).negate();
        BigDecimal xpowi = xhighpr;
        int i = 0;
        while (true) {
            BigDecimal c = BigDecimalMath.divideRound(xpowi, 2 * i + 1);
            resul = resul.add(c);
            if (Math.abs(c.doubleValue()) < 0.1 * eps) break;
            xpowi = BigDecimalMath.multiplyRound(xpowi, xhighprSq);
            ++i;
        }
        mc = new MathContext(BigDecimalMath.err2prec(resul.doubleValue(), eps));
        return resul.round(mc);
    }

    public static BigDecimal cosh(BigDecimal x) {
        if (x.compareTo(BigDecimal.ZERO) < 0) {
            return BigDecimalMath.cos(x.negate());
        }
        if (x.compareTo(BigDecimal.ZERO) == 0) {
            return BigDecimal.ONE;
        }
        if (x.doubleValue() > 1.5) {
            return BigDecimalMath.hypot(1, BigDecimalMath.sinh(x));
        }
        BigDecimal xhighpr = BigDecimalMath.scalePrec(x, 2);
        BigDecimal resul = BigDecimal.ONE;
        BigDecimal xpowi = BigDecimal.ONE;
        BigInteger ifac = BigInteger.ONE;
        double xUlpDbl = 0.5 * x.ulp().doubleValue() * x.doubleValue();
        int k = (int)(Math.log(xUlpDbl) / Math.log(x.doubleValue())) / 2;
        MathContext mcTay = new MathContext(BigDecimalMath.err2prec(1.0, xUlpDbl / (double)k));
        int i = 1;
        while (true) {
            ifac = ifac.multiply(BigInteger.valueOf(2 * i - 1));
            ifac = ifac.multiply(BigInteger.valueOf(2 * i));
            xpowi = xpowi.multiply(xhighpr).multiply(xhighpr);
            BigDecimal corr = xpowi.divide(new BigDecimal(ifac), mcTay);
            resul = resul.add(corr);
            if (corr.abs().doubleValue() < 0.5 * xUlpDbl) break;
            ++i;
        }
        MathContext mc = new MathContext(BigDecimalMath.err2prec(resul.doubleValue(), xUlpDbl));
        return resul.round(mc);
    }

    public static BigDecimal sinh(BigDecimal x) {
        BigDecimal xhighpr;
        if (x.compareTo(BigDecimal.ZERO) < 0) {
            return BigDecimalMath.sinh(x.negate()).negate();
        }
        if (x.compareTo(BigDecimal.ZERO) == 0) {
            return BigDecimal.ZERO;
        }
        if (x.doubleValue() > 2.4) {
            BigDecimal two = new BigDecimal(2);
            BigDecimal xhalf = x.divide(two);
            BigDecimal resul = BigDecimalMath.sinh(xhalf).multiply(BigDecimalMath.cosh(xhalf)).multiply(two);
            double eps = Math.tanh(x.doubleValue());
            MathContext mc = new MathContext(BigDecimalMath.err2prec(0.5 * x.ulp().doubleValue() / eps));
            return resul.round(mc);
        }
        BigDecimal resul = xhighpr = BigDecimalMath.scalePrec(x, 2);
        BigDecimal xpowi = xhighpr;
        BigInteger ifac = BigInteger.ONE;
        double xUlpDbl = x.ulp().doubleValue();
        int k = (int)((double)x.precision() / Math.log10(1.0 / xhighpr.doubleValue())) / 2;
        MathContext mcTay = new MathContext(BigDecimalMath.err2prec(x.doubleValue(), xUlpDbl / (double)k));
        int i = 1;
        while (true) {
            ifac = ifac.multiply(BigInteger.valueOf(2 * i));
            ifac = ifac.multiply(BigInteger.valueOf(2 * i + 1));
            xpowi = xpowi.multiply(xhighpr).multiply(xhighpr);
            BigDecimal corr = xpowi.divide(new BigDecimal(ifac), mcTay);
            resul = resul.add(corr);
            if (corr.abs().doubleValue() < 0.5 * xUlpDbl) break;
            ++i;
        }
        MathContext mc = new MathContext(x.precision());
        return resul.round(mc);
    }

    public static BigDecimal tanh(BigDecimal x) {
        if (x.compareTo(BigDecimal.ZERO) < 0) {
            return BigDecimalMath.tanh(x.negate()).negate();
        }
        if (x.compareTo(BigDecimal.ZERO) == 0) {
            return BigDecimal.ZERO;
        }
        BigDecimal xhighpr = BigDecimalMath.scalePrec(x, 2);
        BigDecimal exp2x = BigDecimalMath.exp(xhighpr.multiply(new BigDecimal(-2)));
        double eps = 0.5 * x.ulp().doubleValue() / Math.pow(Math.cosh(x.doubleValue()), 2.0);
        MathContext mc = new MathContext(BigDecimalMath.err2prec(Math.tanh(x.doubleValue()), eps));
        return BigDecimal.ONE.subtract(exp2x).divide(BigDecimal.ONE.add(exp2x), mc);
    }

    public static BigDecimal asinh(BigDecimal x) {
        if (x.compareTo(BigDecimal.ZERO) == 0) {
            return BigDecimal.ZERO;
        }
        BigDecimal xhighpr = BigDecimalMath.scalePrec(x, 2);
        BigDecimal logx = BigDecimalMath.log(BigDecimalMath.hypot(1, xhighpr).add(xhighpr));
        double xDbl = x.doubleValue();
        double eps = 0.5 * x.ulp().doubleValue() / Math.hypot(1.0, xDbl);
        MathContext mc = new MathContext(BigDecimalMath.err2prec(logx.doubleValue(), eps));
        return logx.round(mc);
    }

    public static BigDecimal acosh(BigDecimal x) {
        if (x.compareTo(BigDecimal.ONE) < 0) {
            throw new ArithmeticException("Out of range argument cosh " + x.toString());
        }
        if (x.compareTo(BigDecimal.ONE) == 0) {
            return BigDecimal.ZERO;
        }
        BigDecimal xhighpr = BigDecimalMath.scalePrec(x, 2);
        BigDecimal logx = BigDecimalMath.log(BigDecimalMath.sqrt(xhighpr.pow(2).subtract(BigDecimal.ONE)).add(xhighpr));
        double xDbl = x.doubleValue();
        double eps = 0.5 * x.ulp().doubleValue() / Math.sqrt(xDbl * xDbl - 1.0);
        MathContext mc = new MathContext(BigDecimalMath.err2prec(logx.doubleValue(), eps));
        return logx.round(mc);
    }

    public static BigDecimal Gamma(BigDecimal x) {
        if (x.compareTo(BigDecimal.ZERO) < 0) {
            return BigDecimalMath.divideRound(BigDecimalMath.Gamma(x.add(BigDecimal.ONE)), x);
        }
        if (x.doubleValue() > 1.5) {
            int n = (int)(x.doubleValue() - 0.5);
            BigDecimal xmin1 = x.subtract(new BigDecimal(n));
            return BigDecimalMath.multiplyRound(BigDecimalMath.Gamma(xmin1), BigDecimalMath.pochhammer(xmin1, n));
        }
        BigDecimal z = x.subtract(BigDecimal.ONE);
        z = BigDecimalMath.scalePrec(z, 2);
        MathContext mcloc = new MathContext(z.precision());
        double eps = x.ulp().doubleValue() / x.doubleValue();
        BigDecimal resul = BigDecimalMath.log(BigDecimalMath.scalePrec(x, 2)).negate();
        if (x.compareTo(BigDecimal.ONE) != 0) {
            BigDecimal gammCompl = BigDecimal.ONE.subtract(BigDecimalMath.gamma(mcloc));
            resul = resul.add(BigDecimalMath.multiplyRound(z, gammCompl));
            int n = 2;
            while (true) {
                BigDecimal c = BigDecimalMath.divideRound(z.pow(n, mcloc), n);
                MathContext m = new MathContext(BigDecimalMath.err2prec((double)n * z.ulp().doubleValue() / 2.0 / z.doubleValue()));
                m = eps / 100.0 / (c = c.round(m)).doubleValue() < 0.01 ? new MathContext(BigDecimalMath.err2prec(eps / 100.0 / c.doubleValue())) : new MathContext(2);
                BigDecimal zetm1 = BigDecimalMath.zeta(n, m).subtract(BigDecimal.ONE);
                c = BigDecimalMath.multiplyRound(c, zetm1);
                resul = n % 2 == 0 ? resul.add(c) : resul.subtract(c);
                if (Math.abs(c.doubleValue()) < eps) break;
                ++n;
            }
        }
        double psi = 0.5772156649;
        double zdbl = z.doubleValue();
        for (int n = 1; n < 5; ++n) {
            psi += zdbl / (double)n / ((double)n + zdbl);
        }
        eps = psi * x.ulp().doubleValue() / 2.0;
        mcloc = new MathContext(BigDecimalMath.err2prec(eps));
        return BigDecimalMath.exp(resul).round(mcloc);
    }

    public static BigDecimal pochhammer(BigDecimal x, int n) {
        BigDecimal xhighpr;
        if (n < 0) {
            throw new ProviderException("Unimplemented pochhammer with negative index " + n);
        }
        if (n == 0) {
            return BigDecimal.ONE;
        }
        BigDecimal resul = xhighpr = BigDecimalMath.scalePrec(x, 2);
        double xUlpDbl = x.ulp().doubleValue();
        double xDbl = x.doubleValue();
        double eps = 0.5 * xUlpDbl / Math.abs(xDbl);
        for (int i = 1; i < n; ++i) {
            resul = resul.multiply(xhighpr.add(new BigDecimal(i)));
            MathContext mcloc = new MathContext(4 + BigDecimalMath.err2prec(eps += 0.5 * xUlpDbl / Math.abs(xDbl + (double)i)));
            resul = resul.round(mcloc);
        }
        return resul.round(new MathContext(BigDecimalMath.err2prec(eps)));
    }

    public static BigDecimal mod2pi(BigDecimal x) {
        int k = (int)(0.5 * x.doubleValue() / Math.PI);
        double err2pi = k != 0 ? 0.25 * Math.abs(x.ulp().doubleValue() / (double)k) : 0.5 * Math.abs(x.ulp().doubleValue());
        MathContext mc = new MathContext(2 + BigDecimalMath.err2prec(6.283, err2pi));
        BigDecimal twopi = BigDecimalMath.pi(mc).multiply(new BigDecimal(2));
        BigDecimal res = x.remainder(twopi);
        if (res.compareTo(BigDecimal.ZERO) < 0) {
            res = res.add(twopi);
        }
        mc = new MathContext(BigDecimalMath.err2prec(res.doubleValue(), x.ulp().doubleValue() / 2.0));
        return res.round(mc);
    }

    public static BigDecimal modpi(BigDecimal x) {
        int k = (int)(x.doubleValue() / Math.PI);
        double errpi = k != 0 ? 0.5 * Math.abs(x.ulp().doubleValue() / (double)k) : 0.5 * Math.abs(x.ulp().doubleValue());
        MathContext mc = new MathContext(2 + BigDecimalMath.err2prec(3.1416, errpi));
        BigDecimal onepi = BigDecimalMath.pi(mc);
        BigDecimal pihalf = onepi.divide(new BigDecimal(2));
        BigDecimal res = x.remainder(onepi);
        if (res.compareTo(pihalf) > 0) {
            res = res.subtract(onepi);
        } else if (res.compareTo(pihalf.negate()) < 0) {
            res = res.add(onepi);
        }
        mc = new MathContext(BigDecimalMath.err2prec(res.doubleValue(), x.ulp().doubleValue() / 2.0));
        return res.round(mc);
    }

    public static BigDecimal zeta(int n, MathContext mc) {
        if (n <= 0) {
            throw new ProviderException("Unimplemented zeta at negative argument " + n);
        }
        if (n == 1) {
            throw new ArithmeticException("Pole at zeta(1) ");
        }
        if (n % 2 == 0) {
            Rational b = new Bernoulli().at(n).abs();
            b = b.divide(new Factorial().at(n));
            b = b.multiply(BigInteger.ONE.shiftLeft(n - 1));
            MathContext mcpi = new MathContext(mc.getPrecision() + (int)Math.log10(10.0 * (double)n));
            BigDecimal piton = BigDecimalMath.pi(mcpi).pow(n, mc);
            return BigDecimalMath.multiplyRound(piton, b);
        }
        if (n == 3) {
            int[] a31 = new int[]{1, -7, -1, 10, -1, -7, 1, 0};
            int[] a33 = new int[]{1, 1, -1, -2, -1, 1, 1, 0};
            BigDecimal S31 = BigDecimalMath.broadhurstBBP(3, 1, a31, mc);
            BigDecimal S33 = BigDecimalMath.broadhurstBBP(3, 3, a33, mc);
            S31 = S31.multiply(new BigDecimal(48));
            S33 = S33.multiply(new BigDecimal(32));
            return S31.add(S33).divide(new BigDecimal(7), mc);
        }
        if (n == 5) {
            int[] a51 = new int[]{31, -1614, -31, -6212, -31, -1614, 31, 74552};
            int[] a53 = new int[]{173, 284, -173, -457, -173, 284, 173, -111};
            int[] a55 = new int[]{1, 0, -1, -1, -1, 0, 1, 1};
            BigDecimal S51 = BigDecimalMath.broadhurstBBP(5, 1, a51, new MathContext(2 + mc.getPrecision()));
            BigDecimal S53 = BigDecimalMath.broadhurstBBP(5, 3, a53, new MathContext(2 + mc.getPrecision()));
            BigDecimal S55 = BigDecimalMath.broadhurstBBP(5, 5, a55, new MathContext(1 + mc.getPrecision()));
            S51 = S51.multiply(new BigDecimal(18432));
            S53 = S53.multiply(new BigDecimal(14336));
            S55 = S55.multiply(new BigDecimal(0x171000));
            return S51.add(S53).subtract(S55).divide(new BigDecimal(62651), mc);
        }
        Rational betsum = new Rational();
        Bernoulli bern = new Bernoulli();
        Factorial fact = new Factorial();
        for (int npr = 0; npr <= (n + 1) / 2; ++npr) {
            Rational b = bern.at(2 * npr).multiply(bern.at(n + 1 - 2 * npr));
            b = b.divide(fact.at(2 * npr)).divide(fact.at(n + 1 - 2 * npr));
            b = b.multiply(1 - 2 * npr);
            betsum = npr % 2 == 0 ? betsum.add(b) : betsum.subtract(b);
        }
        betsum = betsum.divide(n - 1);
        MathContext mcloc = new MathContext(2 + mc.getPrecision() + (int)Math.log10(n));
        BigDecimal ftrm = BigDecimalMath.pi(mcloc).multiply(new BigDecimal(2));
        ftrm = ftrm.pow(n);
        ftrm = BigDecimalMath.multiplyRound(ftrm, betsum.BigDecimalValue(mcloc));
        BigDecimal exps = new BigDecimal(0);
        double eps = Math.pow(10.0, -mc.getPrecision());
        if (n % 4 == 3) {
            int kmax = mc.getPrecision() / 3;
            BigDecimal exp2p = BigDecimalMath.pi(new MathContext(3 + BigDecimalMath.err2prec(3.14, (eps /= (double)kmax) / 0.0075)));
            exp2p = BigDecimalMath.exp(exp2p.multiply(new BigDecimal(2)));
            BigDecimal c = exp2p.subtract(BigDecimal.ONE);
            exps = BigDecimalMath.divideRound(1, c);
            for (int npr = 2; npr <= kmax; ++npr) {
                c = BigDecimalMath.powRound(exp2p, npr).subtract(BigDecimal.ONE);
                c = BigDecimalMath.multiplyRound(c, BigInteger.valueOf(npr).pow(n));
                c = BigDecimalMath.divideRound(1, c);
                exps = exps.add(c);
            }
        } else {
            int kmax = (1 + mc.getPrecision()) / 3;
            BigDecimal twop = BigDecimalMath.pi(new MathContext(3 + BigDecimalMath.err2prec(3.14, (eps /= (double)kmax) / 0.017)));
            twop = twop.multiply(new BigDecimal(2));
            BigDecimal exp2p = BigDecimalMath.exp(twop);
            BigDecimal c = exp2p.subtract(BigDecimal.ONE);
            exps = BigDecimalMath.divideRound(1, c);
            c = BigDecimal.ONE.subtract(BigDecimalMath.divideRound(1, exp2p));
            c = BigDecimalMath.divideRound(twop, c).multiply(new BigDecimal(2));
            c = BigDecimalMath.divideRound(c, n - 1).add(BigDecimal.ONE);
            exps = BigDecimalMath.multiplyRound(exps, c);
            for (int npr = 2; npr <= kmax; ++npr) {
                c = BigDecimalMath.powRound(exp2p, npr).subtract(BigDecimal.ONE);
                c = BigDecimalMath.multiplyRound(c, BigInteger.valueOf(npr).pow(n));
                BigDecimal d = BigDecimalMath.divideRound(1, exp2p.pow(npr));
                d = BigDecimal.ONE.subtract(d);
                d = BigDecimalMath.divideRound(twop, d).multiply(new BigDecimal(2 * npr));
                d = BigDecimalMath.divideRound(d, n - 1).add(BigDecimal.ONE);
                d = BigDecimalMath.divideRound(d, c);
                exps = exps.add(d);
            }
        }
        exps = exps.multiply(new BigDecimal(2));
        return ftrm.subtract(exps, mc);
    }

    public static double zeta1(int n) {
        double[] zmin1 = new double[]{0.0, 0.0, 0.6449340668482264, 0.2020569031595943, 0.08232323371113819, 0.03692775514336993, 0.01734306198444914, 0.008349277381922827, 0.00407735619794434, 0.0020083928260822143, 9.945751278180853E-4, 4.941886041194645E-4, 2.460865533080483E-4, 1.2271334757848915E-4, 6.124813505870483E-5, 3.058823630702049E-5, 1.528225940865187E-5, 7.637197637899763E-6, 3.81729326499984E-6, 1.908212716553939E-6, 9.539620338727962E-7, 4.769329867878064E-7, 2.38450502727733E-7, 1.1921992596531106E-7, 5.960818905125948E-8, 2.980350351465228E-8, 1.4901554828365043E-8, 7.45071178983543E-9, 3.725334024788457E-9, 1.862659723513049E-9, 9.313274324196682E-10, 4.656629065033784E-10, 2.3283118336765053E-10, 1.164155017270052E-10, 5.820772087902701E-11, 2.9103850444971E-11, 1.4551921891041985E-11, 7.275959835057482E-12, 3.637979547378651E-12, 1.818989650307066E-12, 9.094947840263888E-13, 4.547473783042154E-13, 2.2737368458246524E-13, 1.136868407680228E-13, 5.684341987627585E-14, 2.842170976889302E-14, 1.4210854828031608E-14, 7.105427395210853E-15, 3.552713691337114E-15, 1.7763568435791204E-15, 8.881784210930816E-16, 4.440892103143813E-16, 2.220446050798042E-16, 1.1102230251410661E-16, 5.551115124845481E-17, 2.775557562136124E-17, 1.3877787809725232E-17, 6.938893904544153E-18, 3.4694469521659225E-18, 1.7347234760475765E-18, 8.673617380119933E-19, 4.336808690020651E-19, 2.16840434499722E-19, 1.0842021724942414E-19, 5.421010862456646E-20, 2.710505431223469E-20, 1.3552527156101164E-20, 6.77626357804519E-21, 3.388131789020797E-21, 1.694065894509799E-21, 8.470329472546998E-22, 4.235164736272833E-22, 2.1175823681361947E-22, 1.0587911840680233E-22, 5.293955920339871E-23, 2.646977960169853E-23, 1.323488980084899E-23, 6.617444900424404E-24, 3.308722450212172E-24, 1.6543612251060756E-24, 8.271806125530345E-25, 4.135903062765161E-25, 2.0679515313825765E-25, 1.0339757656912871E-25, 5.169878828456431E-26, 2.5849394142282144E-26, 1.2924697071141066E-26, 6.462348535570532E-27, 3.231174267785265E-27, 1.6155871338926325E-27, 8.077935669463163E-28, 4.038968E-28f, 2.019484E-28f, 1.009742E-28f, 5.04871E-29f, 2.524354896707238E-29, 1.262177448353619E-29, 6.310887241768095E-30, 3.1554436208840472E-30, 1.5777218104420236E-30, 7.888609052210118E-31};
        if (n <= 0) {
            throw new ProviderException("Unimplemented zeta at negative argument " + n);
        }
        if (n == 1) {
            throw new ArithmeticException("Pole at zeta(1) ");
        }
        if (n < zmin1.length) {
            return zmin1[n];
        }
        double eps = 1.0E-18 * Math.pow(2.0, -n);
        MathContext mc = new MathContext(BigDecimalMath.err2prec(eps));
        return BigDecimalMath.zeta(n, mc).subtract(BigDecimal.ONE).doubleValue();
    }

    protected static BigDecimal broadhurstBBP(int n, int p, int[] a, MathContext mc) {
        double x = 0.0;
        for (int k = 1; k < 10; ++k) {
            x += (double)a[(k - 1) % 8] / Math.pow(2.0, p * (k + 1) / 2) / Math.pow(k, n);
        }
        double eps = BigDecimalMath.prec2err(x, mc.getPrecision());
        int kmax = (int)(6.6 * (double)mc.getPrecision() / (double)p);
        eps /= (double)kmax;
        BigDecimal res = BigDecimal.ZERO;
        int c = 0;
        while (true) {
            Rational r = new Rational();
            for (int k = 0; k < 8; ++k) {
                Rational tmp = new Rational(BigInteger.valueOf(a[k]), BigInteger.valueOf(1 + 8 * c + k).pow(n));
                int pk1h = p * (2 + 8 * c + k) / 2;
                tmp = tmp.divide(BigInteger.ONE.shiftLeft(pk1h));
                r = r.add(tmp);
            }
            if (Math.abs(r.doubleValue()) < eps) break;
            MathContext mcloc = new MathContext(1 + BigDecimalMath.err2prec(r.doubleValue(), eps));
            res = res.add(r.BigDecimalValue(mcloc));
            ++c;
        }
        return res.round(mc);
    }

    public static BigDecimal addRound(BigDecimal x, BigDecimal y) {
        BigDecimal resul = x.add(y);
        double errR = Math.abs(y.ulp().doubleValue() / 2.0) + Math.abs(x.ulp().doubleValue() / 2.0);
        MathContext mc = new MathContext(BigDecimalMath.err2prec(resul.doubleValue(), errR));
        return resul.round(mc);
    }

    public static BigDecimal subtractRound(BigDecimal x, BigDecimal y) {
        BigDecimal resul = x.subtract(y);
        double errR = Math.abs(y.ulp().doubleValue() / 2.0) + Math.abs(x.ulp().doubleValue() / 2.0);
        MathContext mc = new MathContext(BigDecimalMath.err2prec(resul.doubleValue(), errR));
        return resul.round(mc);
    }

    public static BigDecimal multiplyRound(BigDecimal x, BigDecimal y) {
        BigDecimal resul = x.multiply(y);
        MathContext mc = new MathContext(Math.min(x.precision(), y.precision()));
        return resul.round(mc);
    }

    public static BigDecimal multiplyRound(BigDecimal x, Rational f) {
        if (f.compareTo(BigInteger.ZERO) == 0) {
            return BigDecimal.ZERO;
        }
        MathContext mc = new MathContext(2 + x.precision());
        BigDecimal fbd = f.BigDecimalValue(mc);
        return BigDecimalMath.multiplyRound(x, fbd);
    }

    public static BigDecimal multiplyRound(BigDecimal x, int n) {
        BigDecimal resul = x.multiply(new BigDecimal(n));
        MathContext mc = new MathContext(n != 0 ? x.precision() : 0);
        return resul.round(mc);
    }

    public static BigDecimal multiplyRound(BigDecimal x, BigInteger n) {
        BigDecimal resul = x.multiply(new BigDecimal(n));
        MathContext mc = new MathContext(n.compareTo(BigInteger.ZERO) != 0 ? x.precision() : 0);
        return resul.round(mc);
    }

    public static BigDecimal divideRound(BigDecimal x, BigDecimal y) {
        MathContext mc = new MathContext(Math.min(x.precision(), y.precision()));
        return x.divide(y, mc);
    }

    public static BigDecimal divideRound(BigDecimal x, int n) {
        MathContext mc = new MathContext(x.precision());
        return x.divide(new BigDecimal(n), mc);
    }

    public static BigDecimal divideRound(BigDecimal x, BigInteger n) {
        MathContext mc = new MathContext(x.precision());
        return x.divide(new BigDecimal(n), mc);
    }

    public static BigDecimal divideRound(BigInteger n, BigDecimal x) {
        MathContext mc = new MathContext(x.precision());
        return new BigDecimal(n).divide(x, mc);
    }

    public static BigDecimal divideRound(int n, BigDecimal x) {
        MathContext mc = new MathContext(x.precision());
        return new BigDecimal(n).divide(x, mc);
    }

    public static BigDecimal scalePrec(BigDecimal x, int d) {
        return x.setScale(d + x.scale());
    }

    public static BigDecimal scalePrec(BigDecimal x, MathContext mc) {
        int diffPr = mc.getPrecision() - x.precision();
        if (diffPr > 0) {
            return BigDecimalMath.scalePrec(x, diffPr);
        }
        return x;
    }

    public static int err2prec(BigDecimal x, BigDecimal xerr) {
        return BigDecimalMath.err2prec(xerr.divide(x, MathContext.DECIMAL64).doubleValue());
    }

    public static int err2prec(double x, double xerr) {
        return 1 + (int)Math.log10(Math.abs(0.5 * x / xerr));
    }

    public static int err2prec(double xerr) {
        return 1 + (int)Math.log10(Math.abs(0.5 / xerr));
    }

    public static double prec2err(double x, int prec) {
        return 5.0 * Math.abs(x) * Math.pow(10.0, -prec);
    }
}

