/*
 *  ******************************************************************************
 *  *
 *  *
 *  * This program and the accompanying materials are made available under the
 *  * terms of the Apache License, Version 2.0 which is available at
 *  * https://www.apache.org/licenses/LICENSE-2.0.
 *  *
 *  *  See the NOTICE file distributed with this work for additional
 *  *  information regarding copyright ownership.
 *  * Unless required by applicable law or agreed to in writing, software
 *  * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 *  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 *  * License for the specific language governing permissions and limitations
 *  * under the License.
 *  *
 *  * SPDX-License-Identifier: Apache-2.0
 *  *****************************************************************************
 */

package org.nd4j.linalg.learning.config;

import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.Data;
import org.nd4j.linalg.api.ndarray.INDArray;
import org.nd4j.linalg.learning.AdaDeltaUpdater;
import org.nd4j.linalg.learning.GradientUpdater;
import org.nd4j.linalg.schedule.ISchedule;

import java.util.Arrays;
import java.util.Map;

@Data
@AllArgsConstructor
@Builder(builderClassName = "Builder")
public class AdaDelta implements IUpdater {
    public static final double DEFAULT_ADADELTA_RHO = 0.95;
    public static final double DEFAULT_ADADELTA_EPSILON = 1e-6;

    @lombok.Builder.Default private double rho = DEFAULT_ADADELTA_RHO;
    @lombok.Builder.Default private double epsilon = DEFAULT_ADADELTA_EPSILON;

    public AdaDelta() {
        this(DEFAULT_ADADELTA_RHO, DEFAULT_ADADELTA_EPSILON);
    }

    @Override
    public long stateSize(long numParams) {
        return 2 * numParams;
    }

    @Override
    public GradientUpdater instantiate(INDArray viewArray, boolean initializeViewArray) {
        AdaDeltaUpdater u = new AdaDeltaUpdater(this);
        viewArray = viewArray.reshape(viewArray.length());
        long[] gradientShape = viewArray.shape();
        gradientShape = Arrays.copyOf(gradientShape, gradientShape.length);
        gradientShape[0] /= 2;
        u.setStateViewArray(viewArray, gradientShape, viewArray.ordering(), initializeViewArray);
        return u;
    }

    @Override
    public GradientUpdater instantiate(Map<String, INDArray> updaterState, boolean initializeStateArrays) {
        AdaDeltaUpdater u = new AdaDeltaUpdater(this);
        u.setState(updaterState, initializeStateArrays);
        return u;
    }

    @Override
    public AdaDelta clone() {
        return new AdaDelta(rho, epsilon);
    }

    @Override
    public double getLearningRate(int iteration, int epoch) {
        return Double.NaN;  //No LR for  this updater
    }

    @Override
    public boolean hasLearningRate() {
        return false;
    }

    @Override
    public void setLrAndSchedule(double lr, ISchedule lrSchedule) {
        throw new UnsupportedOperationException("Cannot set learning rate or LR schedule: AdaDelta does not have a learning rate");
    }

    //Partial builder implementation to give public no-arg constructor
    public static class Builder {
        public Builder(){ }
    }
}
