/*
 * Decompiled with CFR 0.152.
 */
package net.time4j;

import java.time.Clock;
import java.time.DateTimeException;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.Period;
import java.time.ZoneId;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.TimeZone;
import net.time4j.CalendarUnit;
import net.time4j.ClockUnit;
import net.time4j.Duration;
import net.time4j.Moment;
import net.time4j.OldApiTimezone;
import net.time4j.PlainDate;
import net.time4j.PlainTime;
import net.time4j.PlainTimestamp;
import net.time4j.ZonalDateTime;
import net.time4j.base.MathUtils;
import net.time4j.base.TimeSource;
import net.time4j.engine.ChronoException;
import net.time4j.engine.ChronoUnit;
import net.time4j.engine.Converter;
import net.time4j.scale.TimeScale;
import net.time4j.tz.Timezone;
import net.time4j.tz.ZonalOffset;

public abstract class TemporalType<S, T>
implements Converter<S, T> {
    private static final int MIO = 1000000;
    public static final TemporalType<Date, Moment> JAVA_UTIL_DATE = new JavaUtilDateRule();
    public static final TemporalType<Long, Moment> MILLIS_SINCE_UNIX = new MillisSinceUnixRule();
    public static final TemporalType<Calendar, ZonalDateTime> JAVA_UTIL_CALENDAR = new CalendarRule();
    public static final TemporalType<TimeZone, Timezone> JAVA_UTIL_TIMEZONE = new ZoneRule();
    public static final TemporalType<LocalDate, PlainDate> LOCAL_DATE = new LocalDateRule();
    public static final TemporalType<LocalTime, PlainTime> LOCAL_TIME = new LocalTimeRule();
    public static final TemporalType<LocalDateTime, PlainTimestamp> LOCAL_DATE_TIME = new LocalDateTimeRule();
    public static final TemporalType<Instant, Moment> INSTANT = new InstantRule();
    public static final TemporalType<ZonedDateTime, ZonalDateTime> ZONED_DATE_TIME = new ZonedDateTimeRule();
    public static final TemporalType<java.time.Duration, Duration<ClockUnit>> THREETEN_DURATION = new DurationRule();
    public static final TemporalType<Period, Duration<CalendarUnit>> THREETEN_PERIOD = new PeriodRule();
    public static final TemporalType<Clock, TimeSource<?>> CLOCK = new ClockRule();
    private static final String JUT_PROVIDER = "java.util.TimeZone~";

    protected TemporalType() {
    }

    @Override
    public abstract T translate(S var1);

    @Override
    public abstract S from(T var1);

    private static class JavaUtilDateRule
    extends TemporalType<Date, Moment> {
        private JavaUtilDateRule() {
        }

        @Override
        public Moment translate(Date date) {
            long l = date.getTime();
            long l2 = MathUtils.floorDivide(l, 1000);
            int n = MathUtils.floorModulo(l, 1000) * 1000000;
            return Moment.of(l2, n, TimeScale.POSIX);
        }

        @Override
        public Date from(Moment moment) {
            long l = moment.getPosixTime();
            int n = moment.getNanosecond();
            long l2 = MathUtils.safeAdd(MathUtils.safeMultiply(l, 1000L), (long)(n / 1000000));
            return new Date(l2);
        }

        @Override
        public Class<Date> getSourceType() {
            return Date.class;
        }
    }

    private static class MillisSinceUnixRule
    extends TemporalType<Long, Moment> {
        private MillisSinceUnixRule() {
        }

        @Override
        public Moment translate(Long l) {
            long l2 = l;
            long l3 = MathUtils.floorDivide(l2, 1000);
            int n = MathUtils.floorModulo(l2, 1000) * 1000000;
            return Moment.of(l3, n, TimeScale.POSIX);
        }

        @Override
        public Long from(Moment moment) {
            long l = moment.getPosixTime();
            int n = moment.getNanosecond();
            return MathUtils.safeAdd(MathUtils.safeMultiply(l, 1000L), (long)(n / 1000000));
        }

        @Override
        public Class<Long> getSourceType() {
            return Long.class;
        }
    }

    private static class CalendarRule
    extends TemporalType<Calendar, ZonalDateTime> {
        private CalendarRule() {
        }

        @Override
        public ZonalDateTime translate(Calendar calendar) {
            Moment moment = JAVA_UTIL_DATE.translate(calendar.getTime());
            Timezone timezone = JAVA_UTIL_TIMEZONE.translate(calendar.getTimeZone());
            return ZonalDateTime.of(moment, timezone);
        }

        @Override
        public Calendar from(ZonalDateTime zonalDateTime) {
            Date date = JAVA_UTIL_DATE.from(zonalDateTime.toMoment());
            TimeZone timeZone = JAVA_UTIL_TIMEZONE.from(zonalDateTime.getTimezone0());
            GregorianCalendar gregorianCalendar = new GregorianCalendar();
            gregorianCalendar.setGregorianChange(new Date(Long.MIN_VALUE));
            gregorianCalendar.setFirstDayOfWeek(2);
            gregorianCalendar.setMinimalDaysInFirstWeek(4);
            gregorianCalendar.setTimeZone(timeZone);
            gregorianCalendar.setTime(date);
            return gregorianCalendar;
        }

        @Override
        public Class<Calendar> getSourceType() {
            return Calendar.class;
        }
    }

    private static class ZoneRule
    extends TemporalType<TimeZone, Timezone> {
        private ZoneRule() {
        }

        @Override
        public Timezone translate(TimeZone timeZone) {
            if (timeZone instanceof OldApiTimezone) {
                return ((OldApiTimezone)timeZone).getDelegate();
            }
            return Timezone.of(TemporalType.JUT_PROVIDER + timeZone.getID());
        }

        @Override
        public TimeZone from(Timezone timezone) {
            if (timezone.getHistory() == null) {
                String string = timezone.getID().canonical();
                if (string.startsWith(TemporalType.JUT_PROVIDER)) {
                    string = string.substring(TemporalType.JUT_PROVIDER.length());
                }
                return TimeZone.getTimeZone(string);
            }
            return new OldApiTimezone(timezone);
        }

        @Override
        public Class<TimeZone> getSourceType() {
            return TimeZone.class;
        }
    }

    private static class LocalDateRule
    extends TemporalType<LocalDate, PlainDate> {
        private LocalDateRule() {
        }

        @Override
        public PlainDate translate(LocalDate localDate) {
            return PlainDate.of(localDate.getYear(), localDate.getMonthValue(), localDate.getDayOfMonth());
        }

        @Override
        public LocalDate from(PlainDate plainDate) {
            return LocalDate.of(plainDate.getYear(), plainDate.getMonth(), plainDate.getDayOfMonth());
        }

        @Override
        public Class<LocalDate> getSourceType() {
            return LocalDate.class;
        }
    }

    private static class LocalTimeRule
    extends TemporalType<LocalTime, PlainTime> {
        private LocalTimeRule() {
        }

        @Override
        public PlainTime translate(LocalTime localTime) {
            return PlainTime.of(localTime.getHour(), localTime.getMinute(), localTime.getSecond(), localTime.getNano());
        }

        @Override
        public LocalTime from(PlainTime plainTime) {
            if (plainTime.getHour() == 24) {
                return LocalTime.MIDNIGHT;
            }
            return LocalTime.of(plainTime.getHour(), plainTime.getMinute(), plainTime.getSecond(), plainTime.getNanosecond());
        }

        @Override
        public Class<LocalTime> getSourceType() {
            return LocalTime.class;
        }
    }

    private static class LocalDateTimeRule
    extends TemporalType<LocalDateTime, PlainTimestamp> {
        private LocalDateTimeRule() {
        }

        @Override
        public PlainTimestamp translate(LocalDateTime localDateTime) {
            return PlainTimestamp.of(PlainDate.of(localDateTime.getYear(), localDateTime.getMonthValue(), localDateTime.getDayOfMonth()), PlainTime.of(localDateTime.getHour(), localDateTime.getMinute(), localDateTime.getSecond(), localDateTime.getNano()));
        }

        @Override
        public LocalDateTime from(PlainTimestamp plainTimestamp) {
            return LocalDateTime.of(plainTimestamp.getYear(), plainTimestamp.getMonth(), plainTimestamp.getDayOfMonth(), plainTimestamp.getHour(), plainTimestamp.getMinute(), plainTimestamp.getSecond(), plainTimestamp.getNanosecond());
        }

        @Override
        public Class<LocalDateTime> getSourceType() {
            return LocalDateTime.class;
        }
    }

    private static class InstantRule
    extends TemporalType<Instant, Moment> {
        private InstantRule() {
        }

        @Override
        public Moment translate(Instant instant) {
            return Moment.of(instant.getEpochSecond(), instant.getNano(), TimeScale.POSIX);
        }

        @Override
        public Instant from(Moment moment) {
            return Instant.ofEpochSecond(moment.getPosixTime(), moment.getNanosecond());
        }

        @Override
        public Class<Instant> getSourceType() {
            return Instant.class;
        }
    }

    private static class ZonedDateTimeRule
    extends TemporalType<ZonedDateTime, ZonalDateTime> {
        private ZonedDateTimeRule() {
        }

        @Override
        public ZonalDateTime translate(ZonedDateTime zonedDateTime) {
            Moment moment = INSTANT.translate(zonedDateTime.toInstant());
            return moment.inZonalView(zonedDateTime.getZone().getId());
        }

        @Override
        public ZonedDateTime from(ZonalDateTime zonalDateTime) {
            ZoneId zoneId;
            Instant instant = INSTANT.from(zonalDateTime.toMoment());
            try {
                zoneId = ZoneId.of(zonalDateTime.getTimezone().canonical());
            }
            catch (DateTimeException dateTimeException) {
                ZonalOffset zonalOffset = Timezone.of(zonalDateTime.getTimezone()).getOffset(zonalDateTime.toMoment());
                zoneId = ZoneOffset.of(zonalOffset.toString());
            }
            return ZonedDateTime.ofInstant(instant, zoneId);
        }

        @Override
        public Class<ZonedDateTime> getSourceType() {
            return ZonedDateTime.class;
        }
    }

    private static class DurationRule
    extends TemporalType<java.time.Duration, Duration<ClockUnit>> {
        private DurationRule() {
        }

        @Override
        public Duration<ClockUnit> translate(java.time.Duration duration) {
            Duration<ClockUnit> duration2 = Duration.of(duration.getSeconds(), ClockUnit.SECONDS).plus(duration.getNano(), ClockUnit.NANOS);
            return duration2.with(Duration.STD_CLOCK_PERIOD);
        }

        @Override
        public java.time.Duration from(Duration<ClockUnit> duration) {
            java.time.Duration duration2 = java.time.Duration.ZERO;
            block7: for (ClockUnit clockUnit : ClockUnit.values()) {
                java.time.temporal.ChronoUnit chronoUnit;
                switch (clockUnit) {
                    case HOURS: {
                        chronoUnit = java.time.temporal.ChronoUnit.HOURS;
                        break;
                    }
                    case MINUTES: {
                        chronoUnit = java.time.temporal.ChronoUnit.MINUTES;
                        break;
                    }
                    case SECONDS: {
                        chronoUnit = java.time.temporal.ChronoUnit.SECONDS;
                        break;
                    }
                    case MILLIS: 
                    case MICROS: {
                        continue block7;
                    }
                    case NANOS: {
                        chronoUnit = java.time.temporal.ChronoUnit.NANOS;
                        break;
                    }
                    default: {
                        throw new UnsupportedOperationException(clockUnit.name());
                    }
                }
                long l = duration.getPartialAmount((ChronoUnit)clockUnit);
                duration2 = duration2.plus(l, chronoUnit);
            }
            if (duration.isNegative()) {
                duration2 = duration2.negated();
            }
            return duration2;
        }

        @Override
        public Class<java.time.Duration> getSourceType() {
            return java.time.Duration.class;
        }
    }

    private static class PeriodRule
    extends TemporalType<Period, Duration<CalendarUnit>> {
        private PeriodRule() {
        }

        @Override
        public Duration<CalendarUnit> translate(Period period) {
            try {
                Duration<CalendarUnit> duration = Duration.ofCalendarUnits(period.getYears(), period.getMonths(), period.getDays());
                return duration.with(Duration.STD_CALENDAR_PERIOD);
            }
            catch (RuntimeException runtimeException) {
                throw new ChronoException("Cannot convert period: " + period, runtimeException);
            }
        }

        @Override
        public Period from(Duration<CalendarUnit> duration) {
            Period period = Period.ZERO;
            block10: for (CalendarUnit calendarUnit : CalendarUnit.values()) {
                long l = duration.getPartialAmount((ChronoUnit)calendarUnit);
                if (l == 0L) continue;
                if (duration.isNegative()) {
                    l = Math.negateExact(l);
                }
                switch (calendarUnit) {
                    case MILLENNIA: {
                        period = period.plusYears(Math.multiplyExact(l, 1000L));
                        continue block10;
                    }
                    case CENTURIES: {
                        period = period.plusYears(Math.multiplyExact(l, 100L));
                        continue block10;
                    }
                    case DECADES: {
                        period = period.plusYears(Math.multiplyExact(l, 10L));
                        continue block10;
                    }
                    case YEARS: {
                        period = period.plusYears(l);
                        continue block10;
                    }
                    case QUARTERS: {
                        period = period.plusMonths(Math.multiplyExact(l, 3L));
                        continue block10;
                    }
                    case MONTHS: {
                        period = period.plusMonths(l);
                        continue block10;
                    }
                    case WEEKS: {
                        period = period.plusDays(Math.multiplyExact(l, 7L));
                        continue block10;
                    }
                    case DAYS: {
                        period = period.plusDays(l);
                        continue block10;
                    }
                    default: {
                        throw new UnsupportedOperationException(calendarUnit.name());
                    }
                }
            }
            return period;
        }

        @Override
        public Class<Period> getSourceType() {
            return Period.class;
        }
    }

    private static class ClockRule
    extends TemporalType<Clock, TimeSource<?>> {
        private ClockRule() {
        }

        @Override
        public TimeSource<?> translate(Clock clock) {
            return () -> INSTANT.translate(clock.instant());
        }

        @Override
        public Clock from(TimeSource<?> timeSource) {
            return new DelegateClock(ZoneId.systemDefault(), timeSource);
        }

        @Override
        public Class<Clock> getSourceType() {
            return Clock.class;
        }
    }

    private static class DelegateClock
    extends Clock {
        private final ZoneId zoneId;
        private final TimeSource<?> source;

        private DelegateClock(ZoneId zoneId, TimeSource<?> timeSource) {
            if (timeSource == null) {
                throw new NullPointerException("Missing time source.");
            }
            this.zoneId = zoneId;
            this.source = timeSource;
        }

        @Override
        public ZoneId getZone() {
            return this.zoneId;
        }

        @Override
        public Clock withZone(ZoneId zoneId) {
            if (zoneId.equals(this.zoneId)) {
                return this;
            }
            return new DelegateClock(zoneId, this.source);
        }

        @Override
        public Instant instant() {
            return INSTANT.from(Moment.from(this.source.currentTime()));
        }
    }
}

