package me.saharnooby.lib.query.query;

import lombok.NonNull;
import me.saharnooby.lib.query.set.ResultSetMapper;
import me.saharnooby.lib.query.set.ResultSetWrapper;

import javax.sql.DataSource;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.util.List;
import java.util.Optional;

/**
 * Represents an SQL query with some parameters to be filled in placeholders.
 * @author saharNooby
 * @since 13:08 14.11.2019
 */
public abstract class AbstractQuery {

	/**
	 * @return The SQL query.
	 */
	public abstract String getSQL();

	/**
	 * @return Values of the query parameters, in order of occurence (denoted by <code>?</code> in the SQL query).
	 */
	public abstract List<Object> getParams();

	/**
	 * Creates a prepared statement and sets parameter values.
	 * @param con Connection to the database.
	 * @return Prepared statement. Must be closed by the caller.
	 * @throws SQLException On SQL error.
	 */
	public PreparedStatement prepareStatement(@NonNull Connection con) throws SQLException {
		return prepareStatement(con, PreparedStatement.NO_GENERATED_KEYS);
	}

	/**
	 * Creates a prepared statement and sets parameter values.
	 * @param con Connection to the database.
	 * @param autoGeneratedKeys A value to be passed to <code>autoGeneratedKeys</code> parameter of <code>prepareStatement</code>.
	 * @return Prepared statement. Must be closed by the caller.
	 * @throws SQLException On SQL error.
	 */
	public PreparedStatement prepareStatement(@NonNull Connection con, int autoGeneratedKeys) throws SQLException {
		PreparedStatement s = con.prepareStatement(getSQL(), autoGeneratedKeys);

		List<Object> params = getParams();
		for (int i = 0; i < params.size(); i++) {
			s.setObject(i + 1, params.get(i));
		}

		return s;
	}

	/**
	 * Creates a prepared statement and executes an update.
	 * @param con Connection to the database.
	 * @return Result of <code>executeUpdate</code>.
	 * @throws SQLException On SQL error.
	 */
	public int update(@NonNull Connection con) throws SQLException {
		try (PreparedStatement s = prepareStatement(con)) {
			return s.executeUpdate();
		}
	}

	/**
	 * Creates a prepared statement and executes an update.
	 * A connection will be retrieved from the source and closed after the update.
	 * @param source Source of connections.
	 * @return Result of <code>executeUpdate</code>.
	 * @throws SQLException On SQL error.
	 */
	public int update(@NonNull DataSource source) throws SQLException {
		try (Connection con = source.getConnection()) {
			return update(con);
		}
	}

	/**
	 * Executes an update, returning generated keys.
	 * @param con Connection to the database.
	 * @return Generated keys result set.
	 * @throws SQLException On SQL error.
	 */
	public ResultSetWrapper updateWithKeys(@NonNull Connection con) throws SQLException {
		PreparedStatement p = prepareStatement(con, PreparedStatement.RETURN_GENERATED_KEYS);
		p.closeOnCompletion();
		p.executeUpdate();
		return new ResultSetWrapper(p.getGeneratedKeys());
	}

	/**
	 * Obtains a connection from the source, executes an update, returns generated keys mapped using specified mapper.
	 * @param source Source of connections.
	 * @param mapper Mapper.
	 * @param <T> Type of the result.
	 * @return Generated keys result set.
	 * @throws SQLException On SQL error.
	 */
	public <T> Optional<T> updateWithKeysAndMap(@NonNull DataSource source, @NonNull ResultSetMapper<T> mapper) throws SQLException {
		try (Connection con = source.getConnection()) {
			return updateWithKeys(con).map(mapper);
		}
	}

	/**
	 * Obtains a connection from the source, executes an update, returns generated keys mapped using specified mapper.
	 * @param source Source of connections.
	 * @param mapper Mapper.
	 * @param <T> Type of list elements.
	 * @return Generated keys result set.
	 * @throws SQLException On SQL error.
	 */
	public <T> List<T> updateWithKeysAndMapAll(@NonNull DataSource source, @NonNull ResultSetMapper<T> mapper) throws SQLException {
		try (Connection con = source.getConnection()) {
			return updateWithKeys(con).mapAll(mapper);
		}
	}

	/**
	 * Creates a prepared statement and executes a query.
	 * @param con Connection to database.
	 * @return Wrapped result set. Must be closed by the caller.
	 * @throws SQLException On SQL error.
	 */
	public ResultSetWrapper query(@NonNull Connection con) throws SQLException {
		PreparedStatement p = prepareStatement(con);
		ResultSetWrapper wrapper = new ResultSetWrapper(p.executeQuery());
		p.closeOnCompletion();
		return wrapper;
	}

	/**
	 * Obtains a connection from the source, performs a query, calls
	 * {@link ResultSetWrapper#map(ResultSetMapper)} with the specified
	 * mapper and returns the result.
	 * @param source Source of connections.
	 * @param mapper Mapper.
	 * @param <T> Result type.
	 * @return Result.
	 * @throws SQLException On SQL error.
	 */
	public <T> Optional<T> queryAndMap(@NonNull DataSource source, @NonNull ResultSetMapper<T> mapper) throws SQLException {
		try (Connection con = source.getConnection()) {
			return query(con).map(mapper);
		}
	}

	/**
	 * Obtains a connection from the source, performs a query, calls
	 * {@link ResultSetWrapper#mapAll(ResultSetMapper)} with the specified
	 * mapper and returns the result.
	 * @param source Source of connections.
	 * @param mapper Mapper.
	 * @param <T> Result type.
	 * @return Result.
	 * @throws SQLException On SQL error.
	 */
	public <T> List<T> queryAndMapAll(@NonNull DataSource source, @NonNull ResultSetMapper<T> mapper) throws SQLException {
		try (Connection con = source.getConnection()) {
			return query(con).mapAll(mapper);
		}
	}

}
