/*
 * This file is part of helper, licensed under the MIT License.
 *
 *  Copyright (c) lucko (Luck) <luck@lucko.me>
 *  Copyright (c) contributors
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a copy
 *  of this software and associated documentation files (the "Software"), to deal
 *  in the Software without restriction, including without limitation the rights
 *  to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 *  copies of the Software, and to permit persons to whom the Software is
 *  furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in all
 *  copies or substantial portions of the Software.
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 *  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 *  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 *  SOFTWARE.
 */

package me.lucko.helper.text;

import me.lucko.helper.utils.NmsUtil;

import net.kyori.text.Component;
import net.kyori.text.serializer.ComponentSerializers;

import org.bukkit.command.CommandSender;
import org.bukkit.entity.Player;

import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.Collections;

/**
 * Utility for sending JSON components to players.
 *
 * @author https://github.com/lucko/LuckPerms/blob/master/bukkit/src/main/java/me/lucko/luckperms/bukkit/compat/BukkitJsonMessageHandler.java
 */
class BukkitTextUtils {

    private static final boolean CHAT_COMPATIBLE = !NmsUtil.getServerVersion().startsWith("v1_7_");

    private static boolean setup = false;
    private static boolean triedAndFailed = false;

    private static Method GET_HANDLE_METHOD;
    private static Field PLAYER_CONNECTION_FIELD;
    private static Method SEND_PACKET_METHOD;
    private static Constructor<?> PACKET_CHAT_CONSTRUCTOR;
    private static Method SERIALIZE_METHOD;

    private static void setup(Object player) throws Exception {
        Class<?> craftPlayerClass = player.getClass();
        GET_HANDLE_METHOD = craftPlayerClass.getDeclaredMethod("getHandle");

        Object handleObject = GET_HANDLE_METHOD.invoke(player);
        Class<?> handleClass = handleObject.getClass();

        PLAYER_CONNECTION_FIELD = handleClass.getDeclaredField("playerConnection");

        Object playerConnectionObject = PLAYER_CONNECTION_FIELD.get(handleObject);

        Method[] playerConnectionMethods = playerConnectionObject.getClass().getDeclaredMethods();
        for (Method m : playerConnectionMethods) {
            if (m.getName().equals("sendPacket")) {
                SEND_PACKET_METHOD = m;
                break;
            }
        }

        Class<?> packetChatClass = NmsUtil.nmsClass("PacketPlayOutChat");
        Constructor[] packetConstructors = packetChatClass.getDeclaredConstructors();
        for (Constructor c : packetConstructors) {
            Class<?>[] parameters = c.getParameterTypes();
            if (parameters.length == 1 && parameters[0].getName().endsWith("IChatBaseComponent")) {
                PACKET_CHAT_CONSTRUCTOR = c;
                break;
            }
        }

        Class<?> baseComponentClass = NmsUtil.nmsClass("IChatBaseComponent");
        Class<?> chatSerializerClass;

        if (baseComponentClass.getClasses().length > 0) {
            chatSerializerClass = baseComponentClass.getClasses()[0];
        } else {
            // 1.7 class is here instead.
            chatSerializerClass = NmsUtil.nmsClass("ChatSerializer");
        }

        SERIALIZE_METHOD = chatSerializerClass.getDeclaredMethod("a", String.class);
    }

    private static synchronized boolean trySetup(Object player) {
        if (setup) return true;
        if (triedAndFailed) return false;

        try {
            setup(player);
            setup = true;
            return true;
        } catch (Throwable e) {
            e.printStackTrace();
            triedAndFailed = true;
            return false;
        }
    }

    private static Object serializeJsonMessage(Player player, Component message) {
        if (!trySetup(player)) {
            return false;
        }

        try {
            return PACKET_CHAT_CONSTRUCTOR.newInstance(SERIALIZE_METHOD.invoke(null, ComponentSerializers.JSON.serialize(message)));
        } catch (Exception e) {
            e.printStackTrace();
            return null;
        }
    }

    private static boolean sendJsonMessage(Player player, Object packet) {
        if (!trySetup(player)) {
            return false;
        }

        try {
            Object connection = PLAYER_CONNECTION_FIELD.get(GET_HANDLE_METHOD.invoke(player));
            SEND_PACKET_METHOD.invoke(connection, packet);
            return true;
        } catch (Exception e) {
            e.printStackTrace();
            return false;
        }
    }

    public static void sendJsonMessage(CommandSender sender, Component message) {
        sendJsonMessage(Collections.singleton(sender), message);
    }

    public static void sendJsonMessage(Iterable<CommandSender> senders, Component message) {
        Object packet = null;
        boolean tried = false;
        String legacy = null;

        for (CommandSender sender : senders) {

            attempt:
            if (CHAT_COMPATIBLE && sender instanceof Player) {
                Player player = (Player) sender;

                if (!tried) {
                    tried = true;
                    packet = serializeJsonMessage(player, message);
                }

                // give up
                if (packet == null) {
                    break attempt;
                }

                // try sending
                if (!sendJsonMessage(player, packet)) {
                    break attempt;
                }

                // sent successfully, process the next sender
                continue;
            }

            // Fallback to legacy text
            if (legacy == null) {
                legacy = TextUtils.toLegacy(message);
            }
            sender.sendMessage(legacy);
        }
    }

}
