/*
 * This file is part of helper, licensed under the MIT License.
 *
 *  Copyright (c) lucko (Luck) <luck@lucko.me>
 *  Copyright (c) contributors
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a copy
 *  of this software and associated documentation files (the "Software"), to deal
 *  in the Software without restriction, including without limitation the rights
 *  to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 *  copies of the Software, and to permit persons to whom the Software is
 *  furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in all
 *  copies or substantial portions of the Software.
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 *  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 *  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 *  SOFTWARE.
 */

package me.lucko.helper.cooldown;

import com.google.common.base.Preconditions;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;

import java.util.Map;
import java.util.OptionalLong;
import java.util.concurrent.TimeUnit;

import javax.annotation.Nonnull;

class CooldownMapImpl<T> implements CooldownMap<T> {

    private final Cooldown base;
    private final LoadingCache<T, Cooldown> cache;

    CooldownMapImpl(Cooldown base) {
        this.base = base;
        this.cache = CacheBuilder.newBuilder()
                // remove from the cache 10 seconds after the cooldown expires
                .expireAfterAccess(base.getTimeout() + 10000L, TimeUnit.MILLISECONDS)
                .build(new CacheLoader<T, Cooldown>() {
                    @Override
                    public Cooldown load(@Nonnull T key) {
                        return base.copy();
                    }
                });
    }

    @Nonnull
    @Override
    public Cooldown getBase() {
        return base;
    }

    /**
     * Gets the internal cooldown instance associated with the given key
     *
     * <p>The inline Cooldown methods in this class should be used to access the functionality of the cooldown as opposed
     * to calling the methods directly via the instance returned by this method.</p>
     *
     * @param key the key
     * @return a cooldown instance
     */
    @Nonnull
    public Cooldown get(@Nonnull T key) {
        Preconditions.checkNotNull(key, "key");
        return cache.getUnchecked(key);
    }

    @Override
    public void put(@Nonnull T key, @Nonnull Cooldown cooldown) {
        Preconditions.checkNotNull(key, "key");
        Preconditions.checkArgument(cooldown.getTimeout() == base.getTimeout(), "different timeout");
        cache.put(key, cooldown);
    }

    /**
     * Gets the cooldowns contained within this collection.
     *
     * @return the backing map
     */
    @Nonnull
    public Map<T, Cooldown> getAll() {
        return cache.asMap();
    }

    /* methods from Cooldown */

    @Override
    public boolean test(@Nonnull T key) {
        return get(key).test();
    }

    @Override
    public boolean testSilently(@Nonnull T key) {
        return get(key).testSilently();
    }

    @Override
    public long elapsed(@Nonnull T key) {
        return get(key).elapsed();
    }

    @Override
    public void reset(@Nonnull T key) {
        get(key).reset();
    }

    @Override
    public long remainingMillis(@Nonnull T key) {
        return get(key).remainingMillis();
    }

    @Override
    public long remainingTime(@Nonnull T key, @Nonnull TimeUnit unit) {
        return get(key).remainingTime(unit);
    }

    @Nonnull
    @Override
    public OptionalLong getLastTested(@Nonnull T key) {
        return get(key).getLastTested();
    }
}
