/*
 * This file is part of helper, licensed under the MIT License.
 *
 *  Copyright (c) lucko (Luck) <luck@lucko.me>
 *  Copyright (c) contributors
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a copy
 *  of this software and associated documentation files (the "Software"), to deal
 *  in the Software without restriction, including without limitation the rights
 *  to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 *  copies of the Software, and to permit persons to whom the Software is
 *  furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in all
 *  copies or substantial portions of the Software.
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 *  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 *  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 *  SOFTWARE.
 */

package me.lucko.helper.cache;

import com.google.common.base.Preconditions;

import java.util.function.Supplier;

/**
 * A lazy supplier extension.
 *
 * <p>The delegate supplier is only called on the first execution of {@link #get()}.
 * The result is then cached and returned for all subsequent calls.</p>
 *
 * @param <T> the supplied type
 */
public final class Lazy<T> implements Supplier<T> {

    public static <T> Lazy<T> suppliedBy(Supplier<T> supplier) {
        return new Lazy<>(Preconditions.checkNotNull(supplier, "supplier"));
    }

    public static <T> Lazy<T> of(T value) {
        return new Lazy<>(Preconditions.checkNotNull(value, "value"));
    }

    private volatile Supplier<T> supplier;
    private volatile boolean initialized = false;
    private T value;

    private Lazy(Supplier<T> supplier) {
        this.supplier = supplier;
    }

    private Lazy(T value) {
        this.value = value;
        this.initialized = true;
    }

    @Override
    public T get() {
        if (!initialized) {
            synchronized (this) {
                if (!initialized) {
                    // compute the value using the delegate
                    T t = supplier.get();

                    value = t;
                    initialized = true;

                    // release the delegate supplier to the gc
                    supplier = null;
                    return t;
                }
            }
        }
        return value;
    }
}
