/*
 * Decompiled with CFR 0.152.
 */
package org.springframework.data.mongodb.core.index;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import org.bson.Document;
import org.springframework.data.domain.Sort;
import org.springframework.data.mongodb.core.index.IndexField;
import org.springframework.data.mongodb.util.BsonUtils;
import org.springframework.lang.Nullable;
import org.springframework.util.Assert;
import org.springframework.util.NumberUtils;
import org.springframework.util.ObjectUtils;

public class IndexInfo {
    private static final Double ONE = 1.0;
    private static final Double MINUS_ONE = -1.0;
    private static final Collection<String> TWO_D_IDENTIFIERS = Arrays.asList("2d", "2dsphere");
    private final List<IndexField> indexFields;
    private final String name;
    private final boolean unique;
    private final boolean sparse;
    private final String language;
    @Nullable
    private Duration expireAfter;
    @Nullable
    private String partialFilterExpression;
    @Nullable
    private Document collation;
    @Nullable
    private Document wildcardProjection;

    public IndexInfo(List<IndexField> indexFields, String name, boolean unique, boolean sparse, String language) {
        this.indexFields = Collections.unmodifiableList(indexFields);
        this.name = name;
        this.unique = unique;
        this.sparse = sparse;
        this.language = language;
    }

    public static IndexInfo indexInfoOf(Document sourceDocument) {
        Document keyDbObject = (Document)sourceDocument.get("key");
        int numberOfElements = keyDbObject.keySet().size();
        ArrayList<IndexField> indexFields = new ArrayList<IndexField>(numberOfElements);
        for (String key : keyDbObject.keySet()) {
            Object value = keyDbObject.get(key);
            if (TWO_D_IDENTIFIERS.contains(value)) {
                indexFields.add(IndexField.geo(key));
                continue;
            }
            if ("text".equals(value)) {
                Document weights = (Document)sourceDocument.get("weights");
                for (String fieldName : weights.keySet()) {
                    indexFields.add(IndexField.text(fieldName, Float.valueOf(weights.get(fieldName).toString())));
                }
                continue;
            }
            if (ObjectUtils.nullSafeEquals("hashed", value)) {
                indexFields.add(IndexField.hashed(key));
                continue;
            }
            if (key.endsWith("$**")) {
                indexFields.add(IndexField.wildcard(key));
                continue;
            }
            Double keyValue = new Double(value.toString());
            if (ONE.equals(keyValue)) {
                indexFields.add(IndexField.create(key, Sort.Direction.ASC));
                continue;
            }
            if (!MINUS_ONE.equals(keyValue)) continue;
            indexFields.add(IndexField.create(key, Sort.Direction.DESC));
        }
        String name = sourceDocument.get("name").toString();
        boolean unique = sourceDocument.containsKey("unique") ? (Boolean)sourceDocument.get("unique") : false;
        boolean sparse = sourceDocument.containsKey("sparse") ? (Boolean)sourceDocument.get("sparse") : false;
        String language = sourceDocument.containsKey("default_language") ? (String)sourceDocument.get("default_language") : "";
        String partialFilter = IndexInfo.extractPartialFilterString(sourceDocument);
        IndexInfo info = new IndexInfo(indexFields, name, unique, sparse, language);
        info.partialFilterExpression = partialFilter;
        info.collation = (Document)((Object)sourceDocument.get((Object)"collation", Document.class));
        if (sourceDocument.containsKey("expireAfterSeconds")) {
            Number expireAfterSeconds = (Number)((Object)sourceDocument.get((Object)"expireAfterSeconds", Number.class));
            info.expireAfter = Duration.ofSeconds(NumberUtils.convertNumberToTargetClass(expireAfterSeconds, Long.class));
        }
        if (sourceDocument.containsKey("wildcardProjection")) {
            info.wildcardProjection = (Document)((Object)sourceDocument.get((Object)"wildcardProjection", Document.class));
        }
        return info;
    }

    @Nullable
    private static String extractPartialFilterString(Document sourceDocument) {
        if (!sourceDocument.containsKey("partialFilterExpression")) {
            return null;
        }
        return BsonUtils.toJson((Document)((Object)sourceDocument.get((Object)"partialFilterExpression", Document.class)));
    }

    public List<IndexField> getIndexFields() {
        return this.indexFields;
    }

    public boolean isIndexForFields(Collection<String> keys) {
        Assert.notNull(keys, "Collection of keys must not be null!");
        ArrayList<String> indexKeys = new ArrayList<String>(this.indexFields.size());
        for (IndexField field : this.indexFields) {
            indexKeys.add(field.getKey());
        }
        return indexKeys.containsAll(keys);
    }

    public String getName() {
        return this.name;
    }

    public boolean isUnique() {
        return this.unique;
    }

    public boolean isSparse() {
        return this.sparse;
    }

    public String getLanguage() {
        return this.language;
    }

    @Nullable
    public String getPartialFilterExpression() {
        return this.partialFilterExpression;
    }

    public Optional<Document> getCollation() {
        return Optional.ofNullable(this.collation);
    }

    public Optional<Document> getWildcardProjection() {
        return Optional.ofNullable(this.wildcardProjection);
    }

    public Optional<Duration> getExpireAfter() {
        return Optional.ofNullable(this.expireAfter);
    }

    public boolean isHashed() {
        return this.getIndexFields().stream().anyMatch(IndexField::isHashed);
    }

    public boolean isWildcard() {
        return this.getIndexFields().stream().anyMatch(IndexField::isWildcard);
    }

    public String toString() {
        return "IndexInfo [indexFields=" + this.indexFields + ", name=" + this.name + ", unique=" + this.unique + ", sparse=" + this.sparse + ", language=" + this.language + ", partialFilterExpression=" + this.partialFilterExpression + ", collation=" + this.collation + ", expireAfterSeconds=" + ObjectUtils.nullSafeToString(this.expireAfter) + "]";
    }

    public int hashCode() {
        int result = 17;
        result += 31 * ObjectUtils.nullSafeHashCode(this.indexFields);
        result += 31 * ObjectUtils.nullSafeHashCode(this.name);
        result += 31 * ObjectUtils.nullSafeHashCode(this.unique);
        result += 31 * ObjectUtils.nullSafeHashCode(this.sparse);
        result += 31 * ObjectUtils.nullSafeHashCode(this.language);
        result += 31 * ObjectUtils.nullSafeHashCode(this.partialFilterExpression);
        result += 31 * ObjectUtils.nullSafeHashCode(this.collation);
        return result += 31 * ObjectUtils.nullSafeHashCode(this.expireAfter);
    }

    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (this.getClass() != obj.getClass()) {
            return false;
        }
        IndexInfo other = (IndexInfo)obj;
        if (this.indexFields == null ? other.indexFields != null : !this.indexFields.equals(other.indexFields)) {
            return false;
        }
        if (this.name == null ? other.name != null : !this.name.equals(other.name)) {
            return false;
        }
        if (this.sparse != other.sparse) {
            return false;
        }
        if (this.unique != other.unique) {
            return false;
        }
        if (!ObjectUtils.nullSafeEquals(this.language, other.language)) {
            return false;
        }
        if (!ObjectUtils.nullSafeEquals(this.partialFilterExpression, other.partialFilterExpression)) {
            return false;
        }
        if (!ObjectUtils.nullSafeEquals(this.collation, other.collation)) {
            return false;
        }
        return ObjectUtils.nullSafeEquals(this.expireAfter, other.expireAfter);
    }
}

