/*
 * Decompiled with CFR 0.152.
 */
package io.ebean.redis;

import io.ebean.cache.ServerCache;
import io.ebean.cache.ServerCacheConfig;
import io.ebean.cache.ServerCacheOptions;
import io.ebean.cache.ServerCacheStatistics;
import io.ebean.meta.MetricVisitor;
import io.ebean.metric.CountMetric;
import io.ebean.metric.MetricFactory;
import io.ebean.metric.TimedMetric;
import io.ebean.metric.TimedMetricStats;
import io.ebean.redis.encode.Encode;
import io.ebean.redis.encode.EncodePrefixKey;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;
import redis.clients.jedis.ScanParams;
import redis.clients.jedis.ScanResult;
import redis.clients.jedis.Transaction;
import redis.clients.jedis.params.SetParams;
import redis.clients.jedis.util.SafeEncoder;

final class RedisCache
implements ServerCache {
    private static final Logger log = LoggerFactory.getLogger(RedisCache.class);
    private static final String CURSOR_0 = "0";
    private static final byte[] CURSOR_0_BYTES = SafeEncoder.encode((String)"0");
    private final JedisPool jedisPool;
    private final String cacheKey;
    private final EncodePrefixKey keyEncode;
    private final Encode valueEncode;
    private final SetParams expiration;
    private final TimedMetric metricGet;
    private final TimedMetric metricGetAll;
    private final TimedMetric metricPut;
    private final TimedMetric metricPutAll;
    private final TimedMetric metricRemove;
    private final TimedMetric metricRemoveAll;
    private final TimedMetric metricClear;
    private final CountMetric hitCount;
    private final CountMetric missCount;

    RedisCache(JedisPool jedisPool, ServerCacheConfig config, Encode valueEncode) {
        this.jedisPool = jedisPool;
        this.cacheKey = config.getCacheKey();
        this.keyEncode = new EncodePrefixKey(config.getCacheKey());
        this.valueEncode = valueEncode;
        this.expiration = this.expiration(config);
        String namePrefix = "l2r." + config.getShortName();
        MetricFactory factory = MetricFactory.get();
        this.hitCount = factory.createCountMetric(namePrefix + ".hit");
        this.missCount = factory.createCountMetric(namePrefix + ".miss");
        this.metricGet = factory.createTimedMetric(namePrefix + ".get");
        this.metricGetAll = factory.createTimedMetric(namePrefix + ".getMany");
        this.metricPut = factory.createTimedMetric(namePrefix + ".put");
        this.metricPutAll = factory.createTimedMetric(namePrefix + ".putMany");
        this.metricRemove = factory.createTimedMetric(namePrefix + ".remove");
        this.metricRemoveAll = factory.createTimedMetric(namePrefix + ".removeMany");
        this.metricClear = factory.createTimedMetric(namePrefix + ".clear");
    }

    private SetParams expiration(ServerCacheConfig config) {
        int maxSecsToLive;
        ServerCacheOptions cacheOptions = config.getCacheOptions();
        if (cacheOptions != null && (maxSecsToLive = cacheOptions.getMaxSecsToLive()) > 0) {
            return new SetParams().ex((long)maxSecsToLive);
        }
        return null;
    }

    public void visit(MetricVisitor visitor) {
        this.hitCount.visit(visitor);
        this.missCount.visit(visitor);
        this.metricGet.visit(visitor);
        this.metricGetAll.visit(visitor);
        this.metricPut.visit(visitor);
        this.metricPutAll.visit(visitor);
        this.metricRemove.visit(visitor);
        this.metricRemoveAll.visit(visitor);
        this.metricClear.visit(visitor);
    }

    private byte[] key(Object id) {
        return this.keyEncode.encode(id);
    }

    private byte[] value(Object data) {
        if (data == null) {
            return null;
        }
        return this.valueEncode.encode(data);
    }

    private Object valueDecode(byte[] data) {
        try {
            if (data == null) {
                return null;
            }
            return this.valueEncode.decode(data);
        }
        catch (Exception e) {
            log.error("Error decoding data, treated as cache miss", (Throwable)e);
            return null;
        }
    }

    private void errorOnRead(Exception e) {
        log.warn("Error when reading redis cache", (Throwable)e);
    }

    private void errorOnWrite(Exception e) {
        log.warn("Error when writing redis cache", (Throwable)e);
    }

    public Map<Object, Object> getAll(Set<Object> keys) {
        LinkedHashMap<Object, Object> linkedHashMap;
        block12: {
            if (keys.isEmpty()) {
                return Collections.emptyMap();
            }
            long start = System.nanoTime();
            LinkedHashMap<Object, Object> map = new LinkedHashMap<Object, Object>();
            ArrayList<Object> keyList = new ArrayList<Object>(keys);
            Jedis resource = this.jedisPool.getResource();
            try {
                List valsAsBytes = resource.mget(this.keysAsBytes(keyList));
                for (int i = 0; i < keyList.size(); ++i) {
                    Object val = this.valueDecode((byte[])valsAsBytes.get(i));
                    if (val == null) continue;
                    map.put(keyList.get(i), val);
                }
                int hits = map.size();
                int miss = keys.size() - hits;
                if (hits > 0) {
                    this.hitCount.add((long)hits);
                }
                if (miss > 0) {
                    this.missCount.add((long)miss);
                }
                this.metricGetAll.addSinceNanos(start);
                linkedHashMap = map;
                if (resource == null) break block12;
            }
            catch (Throwable throwable) {
                try {
                    if (resource != null) {
                        try {
                            resource.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (Exception e) {
                    this.errorOnRead(e);
                    return Collections.emptyMap();
                }
            }
            resource.close();
        }
        return linkedHashMap;
    }

    public Object get(Object id) {
        Object object;
        block10: {
            long start = System.nanoTime();
            Jedis resource = this.jedisPool.getResource();
            try {
                Object val = this.valueDecode(resource.get(this.key(id)));
                if (val != null) {
                    this.hitCount.increment();
                } else {
                    this.missCount.increment();
                }
                this.metricGet.addSinceNanos(start);
                object = val;
                if (resource == null) break block10;
            }
            catch (Throwable throwable) {
                try {
                    if (resource != null) {
                        try {
                            resource.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (Exception e) {
                    this.errorOnRead(e);
                    return null;
                }
            }
            resource.close();
        }
        return object;
    }

    public void put(Object id, Object value) {
        long start = System.nanoTime();
        try (Jedis resource = this.jedisPool.getResource();){
            if (this.expiration == null) {
                resource.set(this.key(id), this.value(value));
            } else {
                resource.set(this.key(id), this.value(value), this.expiration);
            }
            this.metricPut.addSinceNanos(start);
        }
        catch (Exception e) {
            this.errorOnWrite(e);
        }
    }

    public void putAll(Map<Object, Object> keyValues) {
        long start = System.nanoTime();
        try (Jedis resource = this.jedisPool.getResource();){
            try (Transaction multi = resource.multi();){
                for (Map.Entry<Object, Object> entry : keyValues.entrySet()) {
                    if (this.expiration == null) {
                        multi.set(this.key(entry.getKey()), this.value(entry.getValue()));
                        continue;
                    }
                    multi.set(this.key(entry.getKey()), this.value(entry.getValue()), this.expiration);
                }
                multi.exec();
            }
            this.metricPutAll.addSinceNanos(start);
        }
        catch (Exception e) {
            this.errorOnWrite(e);
        }
    }

    public void remove(Object id) {
        long start = System.nanoTime();
        try (Jedis resource = this.jedisPool.getResource();){
            resource.del(this.key(id));
            this.metricRemove.addSinceNanos(start);
        }
        catch (Exception e) {
            this.errorOnWrite(e);
        }
    }

    public void removeAll(Set<Object> keys) {
        long start = System.nanoTime();
        try (Jedis resource = this.jedisPool.getResource();){
            resource.del(this.keysAsBytes(keys));
            this.metricRemoveAll.addSinceNanos(start);
        }
        catch (Exception e) {
            this.errorOnWrite(e);
        }
    }

    public void clear() {
        long start = System.nanoTime();
        try (Jedis resource = this.jedisPool.getResource();){
            String next;
            ScanParams params = new ScanParams();
            params.match(this.cacheKey + ":*");
            byte[] nextCursor = CURSOR_0_BYTES;
            do {
                ScanResult scanResult = resource.scan(nextCursor, params);
                List keys = scanResult.getResult();
                nextCursor = scanResult.getCursorAsBytes();
                if (keys.isEmpty()) continue;
                byte[][] raw = new byte[keys.size()][];
                for (int i = 0; i < keys.size(); ++i) {
                    raw[i] = (byte[])keys.get(i);
                }
                resource.del((byte[][])raw);
            } while (!(next = SafeEncoder.encode((byte[])nextCursor)).equals(CURSOR_0));
            this.metricClear.addSinceNanos(start);
        }
        catch (Exception e) {
            this.errorOnWrite(e);
        }
    }

    private byte[][] keysAsBytes(Collection<Object> keys) {
        byte[][] raw = new byte[keys.size()][];
        int pos = 0;
        for (Object id : keys) {
            raw[pos++] = this.key(id);
        }
        return raw;
    }

    public long getHitCount() {
        return this.hitCount.get(false);
    }

    public long getMissCount() {
        return this.missCount.get(false);
    }

    public int size() {
        return 0;
    }

    public int getHitRatio() {
        return 0;
    }

    public ServerCacheStatistics getStatistics(boolean reset) {
        ServerCacheStatistics cacheStats = new ServerCacheStatistics();
        cacheStats.setCacheName(this.cacheKey);
        cacheStats.setHitCount(this.hitCount.get(reset));
        cacheStats.setMissCount(this.missCount.get(reset));
        cacheStats.setPutCount(this.count(this.metricPut.collect(reset)));
        cacheStats.setRemoveCount(this.count(this.metricRemove.collect(reset)));
        cacheStats.setClearCount(this.count(this.metricClear.collect(reset)));
        return cacheStats;
    }

    private long count(TimedMetricStats stats) {
        return stats == null ? 0L : stats.count();
    }
}

