//$Id: SimpleValueBinder.java,v 1.5 2005/09/06 09:42:44 turin42 Exp $
package org.hibernate.cfg.annotations;

import java.io.Serializable;
import java.lang.reflect.AnnotatedElement;
import java.util.Properties;
import javax.persistence.Basic;
import javax.persistence.Lob;
import javax.persistence.LobType;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.hibernate.AssertionFailure;
import org.hibernate.annotations.Parameter;
import org.hibernate.annotations.Type;
import org.hibernate.cfg.AnnotationBinder;
import org.hibernate.cfg.Ejb3Column;
import org.hibernate.cfg.Mappings;
import org.hibernate.mapping.SimpleValue;
import org.hibernate.mapping.Table;
import org.hibernate.type.ByteArrayBlobType;
import org.hibernate.type.CharacterArrayClobType;
import org.hibernate.type.EnumType;
import org.hibernate.type.PrimitiveByteArrayBlobType;
import org.hibernate.type.PrimitiveCharacterArrayClobType;
import org.hibernate.type.SerializableToBlobType;
import org.hibernate.type.StringClobType;

/**
 * @author Emmanuel Bernard
 */
public class SimpleValueBinder {
	private static Log log = LogFactory.getLog( SimpleValueBinder.class );
	private String propertyName;
	private String returnedClassName;
	private Ejb3Column[] columns;
	private String persistentClassName;
	private String explicitType = "";
	private Properties typeParameters = new Properties();
	private Mappings mappings;

	public void setPropertyName(String propertyName) {
		this.propertyName = propertyName;
	}

	public void setReturnedClassName(String returnedClassName) {
		this.returnedClassName = returnedClassName;
	}

	public void setColumns(Ejb3Column[] columns) {
		this.columns = columns;
	}


	public void setPersistentClassName(String persistentClassName) {
		this.persistentClassName = persistentClassName;
	}

	//TODO execute it lazily to be order safe
	public void setType(AnnotatedElement annotatedElt, Class returnedClass) {
		if ( returnedClass == null ) return; //we cannot guess anything
		Class returnedClassOrElement = returnedClass;
		boolean isArray = false;
		if ( returnedClass.isArray() ) {
			returnedClassOrElement = returnedClassOrElement.getComponentType();
			isArray = true;
		}
		Properties typeParameters = this.typeParameters;
		typeParameters.clear();
		String type = AnnotationBinder.ANNOTATION_STRING_DEFAULT;
		if ( annotatedElt.isAnnotationPresent( Basic.class ) ) {
			Basic ann = (Basic) annotatedElt.getAnnotation( Basic.class );
			switch ( ann.temporalType() ) {
				case DATE:
					type = "date";
					break;
				case TIME:
					type = "time";
					break;
				case TIMESTAMP:
					type = "timestamp";
					break;
				case NONE:
					break;
				default:
					throw new AssertionFailure( "Unknown temporal type: " + ann.temporalType() );
			}
		}
		else if ( annotatedElt.isAnnotationPresent( Lob.class ) ) {
			Lob lob = annotatedElt.getAnnotation( Lob.class );
			LobType lobType = lob.type();
			if ( LobType.CLOB.equals( lobType ) ) {
				if ( String.class.equals( returnedClassOrElement ) ) {
					type = StringClobType.class.getName();
				}
				else if ( Character.class.equals( returnedClassOrElement ) && isArray ) {
					type = CharacterArrayClobType.class.getName();
				}
				else if ( char.class.equals( returnedClassOrElement ) && isArray ) {
					type = PrimitiveCharacterArrayClobType.class.getName();
				}
				else {
					type = "clob";
				}
			}
			if ( LobType.BLOB.equals( lobType ) ) {
				if ( Byte.class.equals( returnedClassOrElement ) && isArray ) {
					type = ByteArrayBlobType.class.getName();
				}
				else if ( byte.class.equals( returnedClassOrElement ) && isArray ) {
					type = PrimitiveByteArrayBlobType.class.getName();
				}
				else if ( Serializable.class.isAssignableFrom( returnedClassOrElement ) ) {
					type = SerializableToBlobType.class.getName();
					//typeParameters = new Properties();
					typeParameters.setProperty(
							SerializableToBlobType.CLASS_NAME,
							returnedClassOrElement.getName()
					);
				}
				else {
					type = "blob";
				}
			}
		}
		else if ( java.sql.Clob.class.equals( returnedClassOrElement ) ) {
			type = "clob";
		}
		else if ( java.sql.Blob.class.equals( returnedClassOrElement ) ) {
			type = "blob";
		}
		//implicit type will check basic types and Serializable classes
		if ( columns == null ) {
			throw new AssertionFailure( "SimpleValueBinder.setColumns should be set before SimpleValueBinder.setType" );
		}
		if ( AnnotationBinder.ANNOTATION_STRING_DEFAULT.equals( type ) ) {
			if ( returnedClassOrElement.isEnum() ) {
				type = EnumType.class.getName();
				typeParameters = new Properties();
				typeParameters.setProperty( EnumType.ENUM, returnedClassOrElement.getName() );
				String schema = columns[0].getTable().getSchema();
				schema = schema == null ? "" : schema;
				String catalog = columns[0].getTable().getCatalog();
				catalog = catalog == null ? "" : catalog;
				typeParameters.setProperty( EnumType.SCHEMA, schema );
				typeParameters.setProperty( EnumType.CATALOG, catalog );
				typeParameters.setProperty( EnumType.TABLE, columns[0].getTable().getName() );
				typeParameters.setProperty( EnumType.COLUMN, columns[0].getName() );
			}
		}
		explicitType = type;
		this.typeParameters = typeParameters;
		Type annType = (Type) annotatedElt.getAnnotation( Type.class );
		setExplicitType( annType );
	}

	public void setExplicitType(String explicitType) {
		this.explicitType = explicitType;
	}

	//FIXME raise an assertion failure  if setExplicitType(String) and setExplicitType(Type) are use at the same time
	public void setExplicitType(Type typeAnn) {
		if ( typeAnn != null ) {
			explicitType = typeAnn.type();
			typeParameters.clear();
			for ( Parameter param : typeAnn.parameters() ) {
				typeParameters.setProperty( param.name(), param.value() );
			}
		}
	}

	public void setMappings(Mappings mappings) {
		this.mappings = mappings;
	}

	private void validate() {
		//TODO check necessary params
		Ejb3Column.checkPropertyConsistency( columns, propertyName );
	}

	public SimpleValue make() {
		validate();
		log.debug( "building SimpleValue for " + propertyName );
		Table table = columns[0].getTable();
		SimpleValue simpleValue = new SimpleValue( table );
		return fillSimpleValue( simpleValue );
	}

	private SimpleValue fillSimpleValue(SimpleValue simpleValue) {
		String type = AnnotationBinder.isDefault( explicitType ) ? returnedClassName : explicitType;
		org.hibernate.mapping.TypeDef typeDef = mappings.getTypeDef( type );
		if ( typeDef != null ) {
			type = typeDef.getTypeClass();
			simpleValue.setTypeParameters( typeDef.getParameters() );
		}
		if ( typeParameters != null && typeParameters.size() != 0 ) {
			//explicit type params takes precedence over type def params
			simpleValue.setTypeParameters( typeParameters );
		}
		simpleValue.setTypeName( type );
		if ( persistentClassName != null ) {
			simpleValue.setTypeUsingReflection( persistentClassName, propertyName );
		}
		for ( Ejb3Column column : columns ) {
			column.linkWithValue( simpleValue );
		}
		return simpleValue;
	}
}
