//$Id: Ejb3JoinColumn.java,v 1.27 2005/09/06 22:13:36 epbernard Exp $
package org.hibernate.cfg;

import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import javax.persistence.JoinColumn;
import javax.persistence.PrimaryKeyJoinColumn;

import org.hibernate.AnnotationException;
import org.hibernate.mapping.Column;
import org.hibernate.mapping.Join;
import org.hibernate.mapping.PersistentClass;
import org.hibernate.mapping.SimpleValue;
import org.hibernate.mapping.Value;
import org.hibernate.util.StringHelper;

/**
 * Wrap state of an EJB3 @JoinColumn annotation
 * and build the Hibernate column mapping element
 *
 * @author Emmanuel Bernard
 */
public class Ejb3JoinColumn extends Ejb3Column {
	private String propertyName;
	/**
	 * property name repated to this column
	 */
	private String referencedColumn;
	private String mappedBy;
	private String defaultColumnHeader;

	//FIXME hacky solution to get the information at proeprty ref resolution
	public String getManyToManyOwnerSideEntityName() {
		return manyToManyOwnerSideEntityName;
	}

	public void setManyToManyOwnerSideEntityName(String manyToManyOwnerSideEntityName) {
		this.manyToManyOwnerSideEntityName = manyToManyOwnerSideEntityName;
	}

	private String manyToManyOwnerSideEntityName;

	public void setPropertyName(String propertyName) {
		this.propertyName = propertyName;
	}

	public void setReferencedColumn(String referencedColumn) {
		this.referencedColumn = referencedColumn;
	}

	public void setDefaultColumnHeader(String defaultColumnHeader) {
		this.defaultColumnHeader = defaultColumnHeader;
	}

	public String getDefaultColumnHeader() {
		return defaultColumnHeader;
	}

	public String getPropertyName() {
		return propertyName;
	}

	public String getMappedBy() {
		return mappedBy;
	}

	public void setMappedBy(String mappedBy) {
		this.mappedBy = mappedBy;
	}

	public Ejb3JoinColumn() {
		setMappedBy( AnnotationBinder.ANNOTATION_STRING_DEFAULT );
	}

	public Ejb3JoinColumn(
			String sqlType,
			String name,
			boolean nullable,
			boolean unique,
			boolean insertable,
			boolean updatable,
			String referencedColumn,
			String secondaryTable,
			Map<String, Join> joins,
			PropertyHolder propertyHolder,
			String propertyName,
			String mappedBy,
			boolean isImplicit,
			Mappings mappings
	) {
		super();
		setImplicit( isImplicit );
		setSqlType( sqlType );
		setColumnName( name );
		setNullable( nullable );
		setUnique( unique );
		setInsertable( insertable );
		setUpdatable( updatable );
		setSecondaryTableName( secondaryTable );
		setPropertyHolder( propertyHolder );
		setJoins( joins );
		setMappings( mappings );
		bind();
		this.referencedColumn = referencedColumn;
		this.propertyName = propertyName;
		this.mappedBy = mappedBy;
	}

	public String getReferencedColumn() {
		return referencedColumn;
	}

	public static Ejb3JoinColumn buildJoinColumn(
			JoinColumn ann,
			Map<String, Join> joins,
			PropertyHolder propertyHolder,
			String propertyName,
			ExtendedMappings mappings
	) {
		if ( ann != null ) {
			Ejb3JoinColumn joinColumn = new Ejb3JoinColumn();
			joinColumn.setJoinAnnotation( ann, null );
			joinColumn.setJoins( joins );
			joinColumn.setPropertyHolder( propertyHolder );
			joinColumn.setPropertyName( propertyName );
			joinColumn.setImplicit( false );
			joinColumn.setMappings( mappings );
			joinColumn.bind();
			return joinColumn;
		}
		else {
			Ejb3JoinColumn joinColumn = new Ejb3JoinColumn();
			joinColumn.setJoins( joins );
			joinColumn.setPropertyHolder( propertyHolder );
			joinColumn.setPropertyName( propertyName );
			joinColumn.setImplicit( true );
			joinColumn.setMappings( mappings );
			joinColumn.bind();
			return joinColumn;
			//return buildImplicitJoinColumn( AnnotationBinder.ANNOTATION_STRING_DEFAULT, joins, propertyHolder, propertyName, mappings);
		}
	}


	//FIXME default name still useful in association table
	public void setJoinAnnotation(JoinColumn annJoin, String defaultName) {
		if ( annJoin == null ) {
			setImplicit( true );
		}
		else {
			setImplicit( false );
			if ( ! AnnotationBinder.isDefault( annJoin.columnDefinition() ) ) setSqlType( annJoin.columnDefinition() );
			if ( ! AnnotationBinder.isDefault( annJoin.name() ) ) setColumnName( annJoin.name() );
			setNullable( annJoin.nullable() );
			setUnique( annJoin.unique() );
			setInsertable( annJoin.insertable() );
			setUpdatable( annJoin.updatable() );
			setReferencedColumn( annJoin.referencedColumnName() );
			setSecondaryTableName( annJoin.secondaryTable() );
		}
	}

	public static Ejb3JoinColumn buildImplicitJoinColumn(
			String mappedBy,
			Map<String, Join> joins,
			PropertyHolder propertyHolder,
			String propertyName,
			ExtendedMappings mappings
	) {
		return new Ejb3JoinColumn(
				(String) null, null,
				true, false, true, true, null, (String) null, joins,
				propertyHolder, propertyName, mappedBy, true, mappings
		);
	}

	/**
	 * Build an Ejb3JoinColumn from a potential annotation and
	 * the identifier it refers to
	 *
	 * @param ann			JoinColumn annotation
	 * @param identifier	 value explaining the identifier
	 * @param propertyHolder
	 * @param mappings	   mappings
	 * @return Ejb3JoinColumn
	 */
	public static Ejb3JoinColumn buildJoinColumn(
			JoinColumn ann,
			Value identifier,
			Map<String, Join> joins,
			PropertyHolder propertyHolder, ExtendedMappings mappings
	) {

		Column col = (Column) identifier.getColumnIterator().next();
		String defaultName = col.getName();
		if ( ann != null ) {
			String sqlType = ann.columnDefinition().equals( "" ) ? null : ann.columnDefinition();
			String name = ann.name().equals( "" ) ? defaultName : ann.name();
			return new Ejb3JoinColumn(
					sqlType,
					name, ann.nullable(), ann.unique(),
					ann.insertable(), ann.updatable(),
					ann.referencedColumnName(),
					ann.secondaryTable(),
					joins,
					propertyHolder, null, null, false, mappings
			);
		}
		else {
			return new Ejb3JoinColumn(
					(String) null, defaultName,
					true, false, true, true, null, (String) null,
					joins, propertyHolder, null, null, false, mappings
			);
		}
	}

	public static Ejb3JoinColumn buildJoinColumn(
			PrimaryKeyJoinColumn ann,
			Value identifier,
			Map<String, Join> joins,
			PropertyHolder propertyHolder, ExtendedMappings mappings
	) {

		Column col = (Column) identifier.getColumnIterator().next();
		String defaultName = col.getName();
		if ( ann != null ) {
			String sqlType = ann.columnDefinition().equals( "" ) ? null : ann.columnDefinition();
			String name = ann.name().equals( "" ) ? defaultName : ann.name();
			return new Ejb3JoinColumn(
					sqlType,
					name, false, false,
					true, true,
					ann.referencedColumnName(),
					null, joins,
					propertyHolder, null, null, false, mappings
			);
		}
		else {
			return new Ejb3JoinColumn(
					(String) null, defaultName,
					false, false, true, true, null, (String) null,
					joins, propertyHolder, null, null, true, mappings
			);
		}
	}

	/**
	 * Override persistent class on oneToMany Cases for late settings
	 */
	public void setPersistentClass(PersistentClass persistentClass, Map<String, Join> joins) {
		this.propertyHolder = PropertyHolderBuilder.buildPropertyHolder( persistentClass, joins );
	}

	public static void checkIfJoinColumn(Object columns, PropertyHolder holder, PropertyInferredData property) {
		if ( ! ( columns instanceof Ejb3JoinColumn[] ) ) {
			throw new AnnotationException(
					"@Column cannot be used on an association property: "
							+ holder.getEntityName()
							+ "."
							+ property.getPropertyName()
			);
		}
	}

	public void linkValueUsingDefaultColumnNaming(Column column, SimpleValue value) {
		String columnName;
		final boolean isAnAssociation = defaultColumnHeader != null || propertyName != null;
		if ( isAnAssociation ) {
			columnName = ( defaultColumnHeader == null ? propertyName : defaultColumnHeader ) + "_" + column.getName();
		}
		else {
			//is an intra-entity hierarchy table join so copy the name by default
			columnName = column.getName();
		}
		initMappingColumn(
				columnName,
				column.getLength(),
				column.getPrecision(),
				column.getScale(),
				getMappingColumn().isNullable(),
				column.getSqlType(),
				getMappingColumn().isUnique()
		);
		linkWithValue( value );
	}

	public void linkValueUsingAColumnCopy(Column column, SimpleValue value) {
		initMappingColumn(
				column.getName(),
				column.getLength(),
				column.getPrecision(),
				column.getScale(),
				getMappingColumn().isNullable(),
				column.getSqlType(),
				getMappingColumn().isUnique()
		);
		linkWithValue( value );
	}

	//keep it JDK 1.4 compliant
	//implicit way
	public static final int NO_REFERENCE = 0;
	//reference to the pk in an explicit order
	public static final int PK_REFERENCE = 1;
	//reference to non pk columns
	public static final int NON_PK_REFERENCE = 2;

	public static int checkReferencedColumnsType(Ejb3JoinColumn[] columns, PersistentClass referencedEntity) {
		//convinient container to find whether a column is an id one or not
		Set<Column> idColumns = new HashSet<Column>();
		Iterator idColumnsIt = referencedEntity.getIdentifier().getColumnIterator();
		while ( idColumnsIt.hasNext() ) {
			idColumns.add( (Column) idColumnsIt.next() );
		}

		boolean isFkReferencedColumnName = false;
		boolean noReferencedColumn = true;

		//check each referenced column
		for ( Ejb3JoinColumn ejb3Column : columns ) {
			String referencedColumnName = ejb3Column.getReferencedColumn();
			if ( StringHelper.isNotEmpty( referencedColumnName ) ) {
				noReferencedColumn = false;
				Column refCol = new Column( referencedColumnName );
				boolean contains = idColumns.contains( refCol );
				if ( ! contains ) {
					isFkReferencedColumnName = true;
				}
			}
		}
		if ( isFkReferencedColumnName ) {
			return NON_PK_REFERENCE;
		}
		else if ( noReferencedColumn ) {
			return NO_REFERENCE;
		}
		else {
			return PK_REFERENCE;
		}
	}

	public void overrideSqlTypeIfNecessary(org.hibernate.mapping.Column column) {
		if ( StringHelper.isEmpty( sqlType ) ) {
			sqlType = column.getSqlType();
			if ( getMappingColumn() != null) getMappingColumn().setSqlType( sqlType );
		}
	}

}