//$Id: AnnotationBinder.java,v 1.142 2005/09/13 16:00:36 epbernard Exp $
package org.hibernate.cfg;

import java.lang.reflect.AnnotatedElement;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.ResourceBundle;
import java.util.Set;
import javax.persistence.AccessType;
import javax.persistence.Basic;
import javax.persistence.Column;
import javax.persistence.DiscriminatorType;
import javax.persistence.Embeddable;
import javax.persistence.EmbeddableSuperclass;
import javax.persistence.Embedded;
import javax.persistence.EmbeddedId;
import javax.persistence.Entity;
import javax.persistence.FetchType;
import javax.persistence.GeneratedIdTable;
import javax.persistence.GeneratorType;
import javax.persistence.Id;
import javax.persistence.IdClass;
import javax.persistence.InheritanceType;
import javax.persistence.JoinColumn;
import javax.persistence.JoinColumns;
import javax.persistence.JoinTable;
import javax.persistence.Lob;
import javax.persistence.ManyToMany;
import javax.persistence.ManyToOne;
import javax.persistence.MapKey;
import javax.persistence.NamedNativeQueries;
import javax.persistence.NamedNativeQuery;
import javax.persistence.NamedQueries;
import javax.persistence.NamedQuery;
import javax.persistence.OneToMany;
import javax.persistence.OneToOne;
import javax.persistence.PrimaryKeyJoinColumn;
import javax.persistence.PrimaryKeyJoinColumns;
import javax.persistence.SequenceGenerator;
import javax.persistence.SqlResultSetMapping;
import javax.persistence.TableGenerator;
import javax.persistence.Transient;
import javax.persistence.Version;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.hibernate.AnnotationException;
import org.hibernate.AssertionFailure;
import org.hibernate.FetchMode;
import org.hibernate.MappingException;
import org.hibernate.annotations.BatchSize;
import org.hibernate.annotations.Cache;
import org.hibernate.annotations.Cascade;
import org.hibernate.annotations.CascadeType;
import org.hibernate.annotations.Check;
import org.hibernate.annotations.Columns;
import org.hibernate.annotations.Filter;
import org.hibernate.annotations.FilterDef;
import org.hibernate.annotations.FilterDefs;
import org.hibernate.annotations.Filters;
import org.hibernate.annotations.Formula;
import org.hibernate.annotations.GenericGenerator;
import org.hibernate.annotations.Index;
import org.hibernate.annotations.NotFound;
import org.hibernate.annotations.NotFoundAction;
import org.hibernate.annotations.OnDelete;
import org.hibernate.annotations.OnDeleteAction;
import org.hibernate.annotations.OrderBy;
import org.hibernate.annotations.ParamDef;
import org.hibernate.annotations.Parameter;
import org.hibernate.annotations.Proxy;
import org.hibernate.annotations.Sort;
import org.hibernate.annotations.Type;
import org.hibernate.annotations.TypeDef;
import org.hibernate.annotations.TypeDefs;
import org.hibernate.annotations.Where;
import org.hibernate.cfg.annotations.CollectionBinder;
import org.hibernate.cfg.annotations.EntityBinder;
import org.hibernate.cfg.annotations.Nullability;
import org.hibernate.cfg.annotations.PropertyBinder;
import org.hibernate.cfg.annotations.QueryBinder;
import org.hibernate.cfg.annotations.SimpleValueBinder;
import org.hibernate.cfg.annotations.TableBinder;
import org.hibernate.engine.FilterDefinition;
import org.hibernate.engine.Versioning;
import org.hibernate.id.MultipleHiLoPerTableGenerator;
import org.hibernate.id.PersistentIdentifierGenerator;
import org.hibernate.id.TableHiLoGenerator;
import org.hibernate.mapping.Component;
import org.hibernate.mapping.DependantValue;
import org.hibernate.mapping.IdGenerator;
import org.hibernate.mapping.Join;
import org.hibernate.mapping.JoinedSubclass;
import org.hibernate.mapping.PersistentClass;
import org.hibernate.mapping.Property;
import org.hibernate.mapping.RootClass;
import org.hibernate.mapping.SimpleValue;
import org.hibernate.mapping.Subclass;
import org.hibernate.mapping.Table;
import org.hibernate.mapping.UnionSubclass;
import org.hibernate.persister.entity.JoinedSubclassEntityPersister;
import org.hibernate.persister.entity.SingleTableEntityPersister;
import org.hibernate.persister.entity.UnionSubclassEntityPersister;
import org.hibernate.type.ForeignKeyDirection;
import org.hibernate.type.TypeFactory;
import org.hibernate.util.ReflectHelper;
import org.hibernate.util.StringHelper;
import org.hibernate.validator.ClassValidator;

/**
 * JSR 175 annotation binder
 * Will read the annotation from classes, apply the
 * principles of the EJB3 spec and produces the Hibernate
 * configuration-time metamodel (the classes in the <tt>mapping</tt>
 * package)
 *
 * @author Emmanuel Bernard
 */
public final class AnnotationBinder {
	private static final String GENERATOR_TABLE_NAME_PARAM = "generatorTableName";
	public static final String ANNOTATION_STRING_DEFAULT = "";

	/*
	 * Some design description
	 * I tried to remove any link to annotation except from the 2 first level of 
	 * method call.
	 * It'll enable to:
	 *   - facilitate annotation overriding
	 *   - mutualize one day xml and annotation binder (probably a dream though)
	 *   - split this huge class in smaller mapping oriented classes
	 *
	 * bindSomething usually create the mapping container and is accessed by one of the 2 first level method
	 * makeSomething usually create the mapping container and is accessed by bindSomething[else]
	 * fillSomething take the container into parameter and fill it.
	 * 
	 *
	 */
	private AnnotationBinder() {
	}

	private static final Log log = LogFactory.getLog( AnnotationBinder.class );

	public static void bindPackage(String packageName, ExtendedMappings mappings) {
		Package pckg = null;
		try {
			pckg = ReflectHelper.classForName( packageName + ".package-info" ).getPackage();
		}
		catch (ClassNotFoundException cnf) {
			log.warn( "Package not found or wo package-info.java: " + packageName );
			return;
		}
		if ( pckg.isAnnotationPresent( SequenceGenerator.class ) ) {
			SequenceGenerator ann = pckg.getAnnotation( SequenceGenerator.class );
			IdGenerator idGen = buildIdGenerator( ann, mappings );
			mappings.addGenerator( idGen );
			log.debug( "Add sequence generator with name: " + idGen.getName() );
		}
		if ( pckg.isAnnotationPresent( TableGenerator.class ) ) {
			TableGenerator ann = pckg.getAnnotation( TableGenerator.class );
			IdGenerator idGen = buildIdGenerator( ann, mappings );
			mappings.addGenerator( idGen );

		}
		if ( pckg.isAnnotationPresent( GeneratedIdTable.class ) ) {
			GeneratedIdTable ann = pckg.getAnnotation( GeneratedIdTable.class );
			Properties params = buildPropertiesFromGeneratorTable( ann );
			mappings.addGeneratorTable( ann.name(), params );
		}
		if ( pckg.isAnnotationPresent( GenericGenerator.class ) ) {
			GenericGenerator ann = pckg.getAnnotation( GenericGenerator.class );
			IdGenerator idGen = buildIdGenerator( ann, mappings );
			mappings.addGenerator( idGen );
		}
		bindQueries( pckg, mappings );
		bindFilterDefs( pckg, mappings );
		bindTypeDefs( pckg, mappings );
	}

	private static void bindQueries(AnnotatedElement annotatedElement, ExtendedMappings mappings) {
		{
			SqlResultSetMapping ann = annotatedElement.getAnnotation( SqlResultSetMapping.class );
			QueryBinder.bindSqlResultsetMapping( ann, mappings );
		}
		{
			NamedQuery ann = annotatedElement.getAnnotation( NamedQuery.class );
			QueryBinder.bindQuery( ann, mappings );
		}
		{
			org.hibernate.annotations.NamedQuery ann = annotatedElement.getAnnotation(
					org.hibernate.annotations.NamedQuery.class
			);
			QueryBinder.bindQuery( ann, mappings );
		}
		{
			NamedQueries ann = annotatedElement.getAnnotation( NamedQueries.class );
			QueryBinder.bindQueries( ann, mappings );
		}
		{
			org.hibernate.annotations.NamedQueries ann = annotatedElement.getAnnotation(
					org.hibernate.annotations.NamedQueries.class
			);
			QueryBinder.bindQueries( ann, mappings );
		}
		{
			NamedNativeQuery ann = annotatedElement.getAnnotation( NamedNativeQuery.class );
			QueryBinder.bindNativeQuery( ann, mappings );
		}
		{
			org.hibernate.annotations.NamedNativeQuery ann = annotatedElement.getAnnotation(
					org.hibernate.annotations.NamedNativeQuery.class
			);
			QueryBinder.bindNativeQuery( ann, mappings );
		}
		{
			NamedNativeQueries ann = annotatedElement.getAnnotation( NamedNativeQueries.class );
			QueryBinder.bindNativeQueries( ann, mappings );
		}
		{
			org.hibernate.annotations.NamedNativeQueries ann = annotatedElement.getAnnotation(
					org.hibernate.annotations.NamedNativeQueries.class
			);
			QueryBinder.bindNativeQueries( ann, mappings );
		}
	}

	private static Properties buildPropertiesFromGeneratorTable(GeneratedIdTable ann) {
		Properties params = new Properties();
		if ( ! isDefault( ann.pkColumnName() ) ) {
			params.setProperty( MultipleHiLoPerTableGenerator.PK_COLUMN_NAME, ann.pkColumnName() );
		}
		if ( ! isDefault( ann.valueColumnName() ) ) {
			params.setProperty( MultipleHiLoPerTableGenerator.VALUE_COLUMN_NAME, ann.valueColumnName() );
		}
		if ( ann.table().specified() ) {
			javax.persistence.Table table = ann.table();
			if ( ! isDefault( table.name() ) ) {
				params.setProperty( MultipleHiLoPerTableGenerator.ID_TABLE, table.name() );
			}
			if ( ! isDefault( table.catalog() ) ) {
				params.setProperty( MultipleHiLoPerTableGenerator.CATALOG, table.catalog() );
			}
			if ( ! isDefault( table.schema() ) ) {
				params.setProperty( MultipleHiLoPerTableGenerator.SCHEMA, table.schema() );
			}
			//FIXME implements uniqueconstrains
		}
		else {
			params.setProperty( MultipleHiLoPerTableGenerator.ID_TABLE, ann.name() );
		}
		return params;
	}

	private static IdGenerator buildIdGenerator(java.lang.annotation.Annotation ann, Mappings mappings) {
		IdGenerator idGen = new IdGenerator();
		if ( mappings.getSchemaName() != null ) {
			idGen.addParam( PersistentIdentifierGenerator.SCHEMA, mappings.getSchemaName() );
		}
		if ( mappings.getCatalogName() != null ) {
			idGen.addParam( PersistentIdentifierGenerator.CATALOG, mappings.getCatalogName() );
		}
		if ( ann == null ) {
			idGen = null;
		}
		else if ( ann instanceof TableGenerator ) {
			TableGenerator tabGen = (TableGenerator) ann;
			idGen.setName( tabGen.name() );
			idGen.setIdentifierGeneratorStrategy( MultipleHiLoPerTableGenerator.class.getName() );
			if ( !isDefault( tabGen.tableName() ) ) {
				idGen.addParam( GENERATOR_TABLE_NAME_PARAM, tabGen.tableName() );
			}
			if ( !isDefault( tabGen.pkColumnValue() ) ) {
				idGen.addParam( MultipleHiLoPerTableGenerator.PK_VALUE_NAME, tabGen.pkColumnValue() );
			}
			idGen.addParam( TableHiLoGenerator.MAX_LO, String.valueOf( tabGen.allocationSize() ) );
			log.debug( "Add table generator with name: " + idGen.getName() );
		}
		else if ( ann instanceof SequenceGenerator ) {
			SequenceGenerator seqGen = (SequenceGenerator) ann;
			idGen.setName( seqGen.name() );
			idGen.setIdentifierGeneratorStrategy( "sequence" );

			if ( ! isDefault( seqGen.sequenceName() ) ) {
				idGen.addParam( org.hibernate.id.SequenceGenerator.SEQUENCE, seqGen.sequenceName() );
			}
			//FIXME: work on initialValue() and allocationSize() through SequenceGenerator.PARAMETERS
			if ( seqGen.initialValue() != 0 || seqGen.allocationSize() != 50 ) {
				log.warn(
						"Hibernate does not support SequenceGenerator.initialValue() nor SequenceGenerator.allocationSize()"
				);
			}
			log.debug( "Add sequence generator with name: " + idGen.getName() );
		}
		else if ( ann instanceof GenericGenerator ) {
			GenericGenerator genGen = (GenericGenerator) ann;
			idGen.setName( genGen.name() );
			idGen.setIdentifierGeneratorStrategy( genGen.strategy() );
			Parameter[] params = genGen.parameters();
			for ( Parameter parameter : params ) {
				idGen.addParam( parameter.name(), parameter.value() );
			}
			log.debug( "Add generic generator with name: " + idGen.getName() );
		}
		else {
			throw new AssertionFailure( "Unknown Generator annotation: " + ann );
		}
		return idGen;
	}

	/**
	 * Bind a class having JSR175 annotations
	 * The subclasses <b>have to</b> be binded after its mother class
	 */
	public static void bindClass(
			Class clazzToProcess, Map<Class, InheritanceState> inheritanceStatePerClass, ExtendedMappings mappings
	) throws MappingException {
		//TODO: be more strict with secondarytable allowance (not for ids, not for secondary table join columns etc)
		InheritanceState inheritanceState = inheritanceStatePerClass.get( clazzToProcess );
		AnnotatedClassType classType = mappings.getClassType( clazzToProcess );
		if ( AnnotatedClassType.EMBEDDABLE_SUPERCLASS
				.equals( classType ) ) {
			return; //will be processed when the top level entity is parsed
		}
		if ( AnnotatedClassType.EMBEDDABLE.equals( classType ) ) return; //allow embeded element declaration
		//FIXME: think of removing from the inheritanceState all non EmbeddedSuperclass and Entity elements
		if ( ! classType.equals( AnnotatedClassType.ENTITY ) ) {
			throw new AnnotationException(
					"Annotated class should have a @javax.persistence.Entity, @javax.persistence.Embeddable or @javax.persistence.EmbeddedSuperclass annotation: " + clazzToProcess
							.getName()
			);
		}
		AnnotatedElement annotatedClass = clazzToProcess;
		bindQueries( annotatedClass, mappings );
		bindFilterDefs( annotatedClass, mappings );
		bindTypeDefs( annotatedClass, mappings );
		Class superClass = clazzToProcess.getSuperclass();
		PersistentClass superEntity = mappings.getClass( superClass.getName() );
		if ( superEntity == null ) {
			//check if superclass is not a potential persistent class
			if ( inheritanceState.hasParents ) {
				throw new AnnotationException(
						"Subclass has to be binded after it's mother class: "
								+ superClass.getName()
				);
			}
		}
		String schema = "";
		String table = ""; //might be no @Table annotation on the annotated class
		String catalog = "";
		String discrimValue = null;
		List<String[]> uniqueConstraints = new ArrayList<String[]>();
		Ejb3DiscriminatorColumn discriminatorColumn = null;
		Ejb3JoinColumn[] inheritanceJoinedColumns = null;

		if ( annotatedClass.isAnnotationPresent( javax.persistence.Table.class ) ) {
			javax.persistence.Table tabAnn = annotatedClass.getAnnotation( javax.persistence.Table.class );
			table = tabAnn.name();
			schema = tabAnn.schema();
			catalog = tabAnn.catalog();
			uniqueConstraints = TableBinder.buildUniqueConstraints( tabAnn.uniqueConstraints() );
		}
		final boolean hasJoinedColumns = InheritanceType.JOINED
				.equals( inheritanceState.type ) && inheritanceState.hasParents;
		if ( hasJoinedColumns ) {
			PrimaryKeyJoinColumns jcsAnn = annotatedClass.getAnnotation( PrimaryKeyJoinColumns.class );
			boolean explicitInheritanceJoinedColumns = jcsAnn != null && jcsAnn.value().length != 0;
			if ( explicitInheritanceJoinedColumns ) {
				int nbrOfInhJoinedColumns = jcsAnn.value().length;
				PrimaryKeyJoinColumn jcAnn;
				inheritanceJoinedColumns = new Ejb3JoinColumn[nbrOfInhJoinedColumns];
				for ( int colIndex = 0; colIndex < nbrOfInhJoinedColumns ; colIndex++ ) {
					jcAnn = jcsAnn.value()[colIndex];
					inheritanceJoinedColumns[colIndex] = Ejb3JoinColumn.buildJoinColumn(
							jcAnn, superEntity.getIdentifier(),
							(Map<String, Join>) null, (PropertyHolder) null, mappings
					);
				}
			}
			else {
				PrimaryKeyJoinColumn jcAnn = annotatedClass.getAnnotation( PrimaryKeyJoinColumn.class );
				inheritanceJoinedColumns = new Ejb3JoinColumn[1];
				inheritanceJoinedColumns[0] = Ejb3JoinColumn.buildJoinColumn(
						jcAnn, superEntity.getIdentifier(),
						(Map<String, Join>) null, (PropertyHolder) null, mappings
				);
			}
			log.debug( "Joined column(s) created" );
		}
		else {
			if ( annotatedClass.isAnnotationPresent( javax.persistence.PrimaryKeyJoinColumns.class )
					|| annotatedClass.isAnnotationPresent( javax.persistence.PrimaryKeyJoinColumn.class ) ) {
				log.warn( "Root entity should not hold an PrimaryKeyJoinColum(s), will be ignored" );
			}
		}

		if ( InheritanceType.SINGLE_TABLE.equals( inheritanceState.type ) ) {
			javax.persistence.Inheritance inhAnn = annotatedClass.getAnnotation( javax.persistence.Inheritance.class );
			DiscriminatorType discriminatorType = inhAnn == null ? DiscriminatorType.STRING : inhAnn.discriminatorType();
			javax.persistence.DiscriminatorColumn discAnn = annotatedClass.getAnnotation(
					javax.persistence.DiscriminatorColumn.class
			);
			org.hibernate.annotations.DiscriminatorFormula discFormulaAnn = annotatedClass.getAnnotation(
					org.hibernate.annotations.DiscriminatorFormula.class
			);
			if ( ! inheritanceState.hasParents ) {
				discriminatorColumn = Ejb3DiscriminatorColumn.buildDiscriminatorColumn(
						discriminatorType, discAnn, discFormulaAnn, mappings
				);
			}
			if ( discAnn != null && inheritanceState.hasParents ) {
				log.warn(
						"Discriminator column has to be defined in the root entity, it will be ignored in subclass: "
								+ clazzToProcess.getName()
				);
			}
			discrimValue = inhAnn == null ? null : inhAnn.discriminatorValue();
		}

		//we now know what kind of persistent entity it is
		PersistentClass persistentClass;
		//create persistent class
		if ( ! inheritanceState.hasParents ) {
			persistentClass = new RootClass();
		}
		else if ( InheritanceType.SINGLE_TABLE.equals( inheritanceState.type ) ) {
			persistentClass = new Subclass( superEntity );
		}
		else if ( InheritanceType.JOINED.equals( inheritanceState.type ) ) {
			persistentClass = new JoinedSubclass( superEntity );
		}
		else if ( InheritanceType.TABLE_PER_CLASS.equals( inheritanceState.type ) ) {
			persistentClass = new UnionSubclass( superEntity );
		}
		else {
			throw new AssertionFailure( "Unknown inheritance type: " + inheritanceState.type );
		}
		Proxy proxyAnn = annotatedClass.getAnnotation( Proxy.class );
		BatchSize sizeAnn = annotatedClass.getAnnotation( BatchSize.class );
		Where whereAnn = annotatedClass.getAnnotation( Where.class );
		Entity entityAnn = annotatedClass.getAnnotation( Entity.class );
		org.hibernate.annotations.Entity hibEntityAnn = annotatedClass.getAnnotation(
				org.hibernate.annotations.Entity.class
		);
		org.hibernate.annotations.Cache cacheAnn = annotatedClass.getAnnotation(
				org.hibernate.annotations.Cache.class
		);
		EntityBinder entityBinder = new EntityBinder(
				entityAnn, hibEntityAnn, clazzToProcess, persistentClass, mappings
		);
		entityBinder.setDiscriminatorValue( discrimValue );
		entityBinder.setBatchSize( sizeAnn );
		entityBinder.setProxy( proxyAnn );
		entityBinder.setWhere( whereAnn );
		entityBinder.setCache( cacheAnn );
		entityBinder.setInheritanceState( inheritanceState );
		Filter filterAnn = annotatedClass.getAnnotation( Filter.class );
		if ( filterAnn != null ) {
			entityBinder.addFilter( filterAnn.name(), filterAnn.condition() );
		}
		Filters filtersAnn = annotatedClass.getAnnotation( Filters.class );
		if ( filtersAnn != null ) {
			for ( Filter filter : filtersAnn.value() ) {
				entityBinder.addFilter( filter.name(), filter.condition() );
			}
		}
		entityBinder.bindEntity();

		if ( inheritanceState.hasTable() ) {
			Check checkAnn = annotatedClass.getAnnotation( Check.class );
			String constraints = checkAnn == null ? null : checkAnn.constraints();
			entityBinder.bindTable(
					schema, catalog, table, uniqueConstraints,
					constraints, inheritanceState.hasDenormalizedTable() ? superEntity.getTable() : null
			);
		}
		Map<String, Column[]> columnOverride = PropertyHolderBuilder.buildColumnOverride(
				annotatedClass,
				persistentClass.getClassName()
		);
		PropertyHolder propertyHolder = PropertyHolderBuilder.buildPropertyHolder( 
				persistentClass,
				columnOverride,
				entityBinder.getSecondaryTables()
		);

		javax.persistence.SecondaryTable secTabAnn = annotatedClass.getAnnotation(
				javax.persistence.SecondaryTable.class
		);
		javax.persistence.SecondaryTables secTabsAnn = annotatedClass.getAnnotation(
				javax.persistence.SecondaryTables.class
		);
		JoinColumn joinColAnn = annotatedClass.getAnnotation( JoinColumn.class );
		JoinColumns joinColsAnn = annotatedClass.getAnnotation( JoinColumns.class );
		entityBinder.firstLevelSecondaryTablesBinding( secTabAnn, secTabsAnn, joinColAnn, joinColsAnn );

		OnDelete onDeleteAnn = annotatedClass.getAnnotation( OnDelete.class );
		boolean onDeleteAppropriate = false;
		if ( InheritanceType.JOINED.equals( inheritanceState.type ) && inheritanceState.hasParents ) {
			onDeleteAppropriate = true;
			JoinedSubclass jsc = (JoinedSubclass) persistentClass;
			if ( persistentClass.getEntityPersisterClass() == null ) {
				persistentClass.getRootClass().setEntityPersisterClass( JoinedSubclassEntityPersister.class );
			}
			SimpleValue key = new DependantValue( jsc.getTable(), jsc.getIdentifier() );
			jsc.setKey( key );
			if ( onDeleteAnn != null ) {
				key.setCascadeDeleteEnabled( OnDeleteAction.CASCADE.equals( onDeleteAnn.action() ) );
			}
			else {
				key.setCascadeDeleteEnabled( false );
			}
			TableBinder.bindFk( jsc, null, inheritanceJoinedColumns, key, false );
			jsc.createPrimaryKey();
			jsc.createForeignKey();

		}
		else if ( InheritanceType.SINGLE_TABLE.equals( inheritanceState.type ) ) {
			if ( inheritanceState.hasParents ) {
				if ( persistentClass.getEntityPersisterClass() == null ) {
					persistentClass.getRootClass().setEntityPersisterClass( SingleTableEntityPersister.class );
				}
			}
			else {
				if ( inheritanceState.hasSons || ! discriminatorColumn.isImplicit() ) {
					//need a discriminator column
					bindDiscriminatorToPersistentClass(
							(RootClass) persistentClass,
							discriminatorColumn,
							entityBinder.getSecondaryTables(),
							propertyHolder
					);
				}
			}
		}
		else if ( InheritanceType.TABLE_PER_CLASS.equals( inheritanceState.type ) ) {
			if ( inheritanceState.hasParents ) {
				if ( persistentClass.getEntityPersisterClass() == null ) {
					persistentClass.getRootClass().setEntityPersisterClass( UnionSubclassEntityPersister.class );
				}
			}
		}
		if ( onDeleteAnn != null && onDeleteAppropriate == false ) {
			log.warn(
					"Inapropriate use of @OnDelete on entity, annotation ignored: " + propertyHolder.getEntityName()
			);
		}

		//try to find class level generators
		HashMap<String, IdGenerator> classGenerators = buildLocalGenerators( annotatedClass, mappings );
		HashMap<String, Properties> classGeneratorTables = buildLocalGeneratorTable( annotatedClass );
		Set<String> idProperties = new HashSet<String>();
		if ( annotatedClass.isAnnotationPresent( IdClass.class ) ) {
			IdClass idClass = annotatedClass.getAnnotation( IdClass.class );
			Class compositeClass = idClass.value();
			Embeddable embeddableAnn = (Embeddable) compositeClass.getAnnotation( Embeddable.class );
			boolean isComponent = true;
			boolean propertyAccess = true;
			if ( embeddableAnn != null && embeddableAnn.access() == AccessType.FIELD ) propertyAccess = false;
			String generatorType = "assigned";
			String generator = ANNOTATION_STRING_DEFAULT;
			PropertyInferredData inferredData = new PropertyInferredData(
					propertyAccess == true ? "property" : "field", "id", compositeClass
			);
			HashMap<String, IdGenerator> localGenerators = new HashMap<String, IdGenerator>();
			HashMap<String, Properties> localGeneratorTables = new HashMap<String, Properties>();
			bindId(
					generatorType,
					generator,
					inferredData,
					null,
					propertyHolder,
					localGenerators,
					localGeneratorTables,
					isComponent,
					columnOverride,
					propertyAccess,
					entityBinder,
					null,
					true,
					mappings
			);
			inferredData = new PropertyInferredData(
					propertyAccess == true ? "property" : "field", "_identifierMapper", compositeClass
			);
			Component mapper = fillComponent(
					propertyHolder,
					inferredData,
					propertyAccess,
					false,
					entityBinder,
					true,
					columnOverride,
					mappings
			);
			mapper.setEmbedded( true );
			persistentClass.setIdentifierMapper( mapper );
			Property property = new Property();
			property.setName( "_identifierMapper" );
			property.setNodeName( "id" );
			property.setUpdateable( false );
			property.setInsertable( false );
			property.setValue( mapper );
			property.setPropertyAccessorName( "embedded" );
			persistentClass.addProperty( property );
			entityBinder.setIgnoreIdAnnotations( true );

			Iterator properties = mapper.getPropertyIterator();
			while ( properties.hasNext() ) {
				idProperties.add( ( (Property) properties.next() ).getName() );
			}
		}

		// check properties
		List<Class> classesToProcess = orderClassesToBeProcessed(
				clazzToProcess, inheritanceStatePerClass, inheritanceState
		);
		int deep = classesToProcess.size();
		List<PropertyAnnotatedElement> elements = new ArrayList<PropertyAnnotatedElement>();
		boolean hasIdentifier = false;
		for ( int index = 0; index < deep ; index++ ) {
			Class clazz = classesToProcess.get( index );
			InheritanceState state = inheritanceStatePerClass.get( clazz );
			boolean isPropertyAccess;
			//yuk
			if ( state.isEmbeddableSuperclass ) {
				isPropertyAccess = state.accessType == AccessType.PROPERTY;
			}
			else {
				isPropertyAccess = entityBinder.isPropertyAccess();
			}
			boolean currentHasIdentifier = addElementsOfAClass( elements, propertyHolder, isPropertyAccess, clazz );
			hasIdentifier = hasIdentifier || currentHasIdentifier;
		}
		final boolean rootEntityWoId = !entityBinder.isIgnoreIdAnnotations() && !hasIdentifier && !inheritanceState.hasParents;
		final boolean subclassAndSingleTableStrategy = inheritanceState.type == InheritanceType.SINGLE_TABLE
				&& inheritanceState.hasParents;
		if ( rootEntityWoId ) {
			throw new AnnotationException( "No identifier specified for entity: " + propertyHolder.getEntityName() );
		}
		Set<String> missingIdProperties = new HashSet<String>( idProperties );
		for ( PropertyAnnotatedElement propertyAnnotatedElement : elements ) {
			String propertyName = propertyAnnotatedElement.inferredData.getPropertyName();
			if ( ! idProperties.contains( propertyName ) ) {
				processElementAnnotations(
						propertyHolder,
						subclassAndSingleTableStrategy ? Nullability.FORCED_NULL : Nullability.NO_CONSTRAINT,
						propertyAnnotatedElement.element,
						propertyAnnotatedElement.inferredData, classGenerators, classGeneratorTables, entityBinder,
						false, mappings
				);
			}
			else {
				missingIdProperties.remove( propertyName );
			}
		}

		if ( missingIdProperties.size() != 0 ) {
			StringBuilder missings = new StringBuilder();
			for ( String property : missingIdProperties ) {
				missings.append( property ).append( ", " );
			}
			throw new AnnotationException(
					"Unable to find properties ("
							+ missings.substring( 0, missings.length() - 2 )
							+ ") in entity annotated with @IdClass:" + persistentClass.getEntityName()
			);
		}

		if ( ! inheritanceState.hasParents ) {
			( (RootClass) persistentClass ).createPrimaryKey();
		}
		else {
			superEntity.addSubclass( (Subclass) persistentClass );
		}

		mappings.addClass( persistentClass );
		entityBinder.finalSecondaryTableBinding( propertyHolder );

		//add indexes
		entityBinder.addIndexes( annotatedClass.getAnnotation( org.hibernate.annotations.Table.class ) );
		entityBinder.addIndexes( annotatedClass.getAnnotation( org.hibernate.annotations.Tables.class ) );

		//integrate the validate framework
		ResourceBundle rb = null;
		//TODO integrate a proper resource bundle
		//ResourceBundle rb = ResourceBundle.getBundle("messages", Locale.ENGLISH);

		new ClassValidator( clazzToProcess, rb ).apply( persistentClass );
	}

	private static List<Class> orderClassesToBeProcessed(
			Class annotatedClass, Map<Class, InheritanceState> inheritanceStatePerClass,
			InheritanceState inheritanceState
	) {
		//ordered to allow proper messages on properties subclassing
		List<Class> classesToProcess = new ArrayList<Class>();
		Class currentClassInHierarchy = annotatedClass;
		InheritanceState superclassState;
		do {
			classesToProcess.add( 0, currentClassInHierarchy );
			currentClassInHierarchy = currentClassInHierarchy.getSuperclass();
			superclassState = inheritanceStatePerClass.get( currentClassInHierarchy );
		}
		while ( ! inheritanceState.hasParents && superclassState != null );
		return classesToProcess;
	}

	private static void bindFilterDefs(AnnotatedElement annotatedElement, ExtendedMappings mappings) {
		FilterDef defAnn = annotatedElement.getAnnotation( FilterDef.class );
		FilterDefs defsAnn = annotatedElement.getAnnotation( FilterDefs.class );
		if ( defAnn != null ) {
			bindFilterDef( defAnn, mappings );
		}
		if ( defsAnn != null ) {
			for ( FilterDef def : defsAnn.value() ) {
				bindFilterDef( def, mappings );
			}
		}
	}

	private static void bindFilterDef(FilterDef defAnn, ExtendedMappings mappings) {
		FilterDefinition def = new FilterDefinition( defAnn.name() );
		for ( ParamDef param : defAnn.parameters() ) {
			//TypeFactory.heuristicType( param.type() );
			def.addParameterType( param.name(), TypeFactory.heuristicType( param.type() ) );
		}
		mappings.addFilterDefinition( def );
	}

	private static void bindTypeDefs(AnnotatedElement annotatedElement, ExtendedMappings mappings) {
		TypeDef defAnn = annotatedElement.getAnnotation( TypeDef.class );
		TypeDefs defsAnn = annotatedElement.getAnnotation( TypeDefs.class );
		if ( defAnn != null ) {
			bindTypeDef( defAnn, mappings );
		}
		if ( defsAnn != null ) {
			for ( TypeDef def : defsAnn.value() ) {
				bindTypeDef( def, mappings );
			}
		}
	}

	private static void bindTypeDef(TypeDef defAnn, ExtendedMappings mappings) {
		Properties params = new Properties();
		for ( Parameter param : defAnn.parameters() ) {
			params.setProperty( param.name(), param.value() );
		}
		mappings.addTypeDef( defAnn.name(), defAnn.typeClass().getName(), params );
	}

	private static HashMap<String, Properties> buildLocalGeneratorTable(AnnotatedElement annotatedClass) {
		HashMap<String, Properties> result = new HashMap<String, Properties>();
		GeneratedIdTable ann = (GeneratedIdTable) annotatedClass.getAnnotation( GeneratedIdTable.class );
		if ( ann != null ) {
			result.put( ann.name(), buildPropertiesFromGeneratorTable( ann ) );
		}
		return result;
	}

	private static void bindDiscriminatorToPersistentClass(
			RootClass rootClass,
			Ejb3DiscriminatorColumn discriminatorColumn, Map<String, Join> secondaryTables,
			PropertyHolder propertyHolder
	) {
		if ( rootClass.getDiscriminator() == null ) {
			if ( discriminatorColumn == null ) {
				throw new AssertionFailure( "discriminator column should have been built" );
			}
			discriminatorColumn.setJoins( secondaryTables );
			discriminatorColumn.setPropertyHolder( propertyHolder );
			SimpleValue discrim = new SimpleValue( rootClass.getTable() );
			rootClass.setDiscriminator( discrim );
			discriminatorColumn.linkWithValue( discrim );
			discrim.setTypeName( discriminatorColumn.getDiscriminatorTypeName() );
			rootClass.setPolymorphic( true );
			log.debug( "Setting discriminator for entity " + rootClass.getEntityName() );
		}
	}

	/**
	 * Add elements of a class
	 */
	private static boolean addElementsOfAClass(
			List<PropertyAnnotatedElement> elements, PropertyHolder propertyHolder, boolean propertyAccess,
			final Class annotatedClass
	) {
		boolean hasIdentifier = false;
		if ( propertyAccess ) {
			log.debug( "Processing " + propertyHolder.getEntityName() + " per property access" );
			Method[] methods = annotatedClass.getDeclaredMethods();
			AnnotatedElement currentElt;
			int index = 0;
			while ( index < methods.length ) {
				currentElt = (AnnotatedElement) methods[index];
				final boolean currentHasIdentifier = addAnnotatedElement( currentElt, elements );
				hasIdentifier = hasIdentifier || currentHasIdentifier;
				index++;
			}
		}
		else {
			log.debug( "Processing " + propertyHolder.getEntityName() + " per field access" );
			Field[] fields = annotatedClass.getDeclaredFields();
			AnnotatedElement currentElt;
			int index = 0;
			while ( index < fields.length ) {
				currentElt = (AnnotatedElement) fields[index];
				final boolean currentHasIdentifier = addAnnotatedElement( currentElt, elements );
				hasIdentifier = hasIdentifier || currentHasIdentifier;
				index++;
			}
		}
		return hasIdentifier;
	}

	private static boolean addAnnotatedElement(AnnotatedElement elt, List<PropertyAnnotatedElement> annElts) {
		boolean hasIdentifier = false;
		PropertyAnnotatedElement propertyAnnotatedElement = new PropertyAnnotatedElement( elt );
		if ( ! propertyAnnotatedElement.inferredData.skip() ) {
			/* 
			 * put element annotated by @Id in front
			 * since it has to be parsed before any assoctation by Hibernate 
			 */
			final AnnotatedElement element = propertyAnnotatedElement.element;
			if ( element.isAnnotationPresent( Id.class ) || element.isAnnotationPresent( EmbeddedId.class ) ) {
				annElts.add( 0, propertyAnnotatedElement );
				hasIdentifier = true;
			}
			else {
				annElts.add( propertyAnnotatedElement );
				hasIdentifier = false;
			}
		}
		return hasIdentifier;
	}

	/**
	 * Process annotation of a particular element
	 */
	private static void processElementAnnotations(
			PropertyHolder propertyHolder, Nullability nullability, AnnotatedElement annotatedElt,
			PropertyInferredData inferredData, HashMap<String, IdGenerator> classGenerators,
			HashMap<String, Properties> classGeneratorTables, EntityBinder entityBinder, boolean isIdentifierMapper,
			ExtendedMappings mappings
	)
			throws MappingException {
		Ejb3Column[] columns = null;
		Ejb3JoinColumn[] joinColumns = null;
		if ( annotatedElt.isAnnotationPresent( Transient.class ) ) {
			return;
		}
		if ( log.isDebugEnabled() ) {
			log.debug(
					"Processing annotations of " + propertyHolder.getEntityName() + "." + inferredData.getPropertyName()
			);
		}

		//process @JoinColumn(s) before @Column(s) to handle collection of elements properly
		if ( annotatedElt.isAnnotationPresent( JoinColumn.class ) ) {
			JoinColumn ann = (JoinColumn) annotatedElt.getAnnotation( JoinColumn.class );
			joinColumns = new Ejb3JoinColumn[1];
			joinColumns[0] = Ejb3JoinColumn.buildJoinColumn(
					ann,
					entityBinder.getSecondaryTables(), propertyHolder, inferredData.getPropertyName(), mappings
			);
		}
		else if ( annotatedElt.isAnnotationPresent( JoinColumns.class ) ) {
			JoinColumns ann = annotatedElt.getAnnotation( JoinColumns.class );
			JoinColumn[] annColumns = ann.value();
			int length = annColumns.length;
			if ( length == 0 ) {
				throw new AnnotationException( "Cannot bind an empty @JoinColumns" );
			}
			joinColumns = new Ejb3JoinColumn[length];
			for ( int index = 0; index < length ; index++ ) {
				joinColumns[index] = Ejb3JoinColumn.buildJoinColumn(
						annColumns[index],
						entityBinder.getSecondaryTables(), propertyHolder, inferredData.getPropertyName(), mappings
				);
			}
		}
		if ( annotatedElt.isAnnotationPresent( Column.class ) || annotatedElt.isAnnotationPresent( Formula.class ) ) {
			Column ann = (Column) annotatedElt.getAnnotation( Column.class );
			Formula formulaAnn = (Formula) annotatedElt.getAnnotation( Formula.class );
			columns = Ejb3Column.buildColumnFromAnnotation(
					new Column[]{ann}, formulaAnn, nullability, propertyHolder, inferredData,
					entityBinder.getSecondaryTables(), mappings
			);
		}
		else if ( annotatedElt.isAnnotationPresent( Columns.class ) ) {
			Columns anns = annotatedElt.getAnnotation( Columns.class );
			columns = Ejb3Column.buildColumnFromAnnotation(
					anns.columns(), null, nullability, propertyHolder, inferredData, entityBinder.getSecondaryTables(),
					mappings
			);
		}

		//set default values
		if ( joinColumns == null &&
				( annotatedElt.isAnnotationPresent( ManyToOne.class )
						|| annotatedElt.isAnnotationPresent( OneToOne.class ) )
				) {
			joinColumns = new Ejb3JoinColumn[1];
			if ( annotatedElt.isAnnotationPresent( JoinTable.class ) ) {
				JoinTable assocTable = annotatedElt.getAnnotation( JoinTable.class );
				//entityBinder.firstLevelSecondaryTablesBinding(assocTable);
				throw new NotYetImplementedException(
						"association table on a single ended association is not yet supported"
				);
			}
			else {
				OneToOne oneToOneAnn = annotatedElt.getAnnotation( OneToOne.class );
				String mappedBy = oneToOneAnn != null ?
						oneToOneAnn.mappedBy() :
						null;
				joinColumns[0] = Ejb3JoinColumn.buildImplicitJoinColumn(
						mappedBy, entityBinder.getSecondaryTables(),
						propertyHolder, inferredData.getPropertyName(), mappings
				);
			}
		}
		else if ( joinColumns == null && annotatedElt.isAnnotationPresent( OneToMany.class ) ) {
			joinColumns = new Ejb3JoinColumn[1];
			String mappedBy = ( (OneToMany) annotatedElt.getAnnotation( OneToMany.class ) ).mappedBy();
			joinColumns[0] = Ejb3JoinColumn.buildImplicitJoinColumn(
					mappedBy, entityBinder.getSecondaryTables(),
					propertyHolder, inferredData.getPropertyName(), mappings
			);
		}
		if ( columns == null ) {
			columns = Ejb3Column.buildColumnFromAnnotation(
					null, null, nullability, propertyHolder, inferredData, entityBinder.getSecondaryTables(), mappings
			);
		}

		//init index
		Index index = annotatedElt.getAnnotation( Index.class );
		if ( index != null ) {
			for ( Ejb3Column column : columns ) {
				column.addIndex( index );
			}
		}

		if ( nullability == Nullability.FORCED_NOT_NULL ) {
			//force columns to not null
			for ( Ejb3Column col : columns ) {
				col.forceNotNull();
			}
		}

		final Class returnedClass = inferredData.getReturnedClassOrElement();
		if ( !entityBinder.isIgnoreIdAnnotations() &&
				( annotatedElt.isAnnotationPresent( Id.class ) || annotatedElt.isAnnotationPresent(
						EmbeddedId.class
				) ) ) {
			if ( isIdentifierMapper ) {
				throw new AnnotationException(
						"@IdClass class should not have @Id nor @EmbeddedId proeperties"
				);
			}
			log.debug( inferredData.getPropertyName() + " is an id" );
			Id idAnn = (Id) annotatedElt.getAnnotation( Id.class );
			EmbeddedId embeddedId = (EmbeddedId) annotatedElt.getAnnotation( EmbeddedId.class );

			//clone classGenerator and override with local values
			HashMap<String, IdGenerator> localGenerators = (HashMap<String, IdGenerator>) classGenerators.clone();
			HashMap<String, Properties> localGeneratorTables = (HashMap<String, Properties>) classGeneratorTables.clone();
			localGenerators.putAll( buildLocalGenerators( annotatedElt, mappings ) );
			localGeneratorTables.putAll( buildLocalGeneratorTable( annotatedElt ) );

			//manage composite related metadata
			Embeddable embeddableAnn = (Embeddable) returnedClass.getAnnotation( Embeddable.class );
			Map<String, Column[]> columnOverride = PropertyHolderBuilder.buildColumnOverride(
					annotatedElt, propertyHolder.getPath() + '.' + inferredData.getPropertyName()
			);
			//guess if its a component and find id data access (property, field etc)
			final boolean isComponent = embeddableAnn != null || embeddedId != null;
			boolean propertyAccess = true;
			if ( isComponent && embeddableAnn != null && embeddableAnn.access() == AccessType.FIELD ) propertyAccess = false;
			//for now columns.length > 1 is impossible for @Id
			String generatorType = idAnn != null ? generatorType( idAnn.generate() ) : "assigned";
			String generator = idAnn != null ? idAnn.generator() : ANNOTATION_STRING_DEFAULT;
			Type typeAnn = annotatedElt.getAnnotation( Type.class );
			bindId(
					generatorType,
					generator,
					inferredData,
					columns[0],
					propertyHolder,
					localGenerators,
					localGeneratorTables,
					isComponent,
					columnOverride,
					propertyAccess,
					entityBinder,
					typeAnn,
					false,
					mappings
			);
			if ( log.isDebugEnabled() ) {
				log.debug(
						"Bind " + ( isComponent ? "@Id" : "@EmbeddedId" ) + " on " + inferredData.getPropertyName()
				);
			}
		}
		else if ( annotatedElt.isAnnotationPresent( Version.class ) ) {
			if ( isIdentifierMapper ) {
				throw new AnnotationException(
						"@IdClass class should not have @Version proeperty"
				);
			}
			if ( ! ( propertyHolder.getPersistentClass() instanceof RootClass ) ) {
				throw new AnnotationException(
						"Unable to define/override @Version on a subclass: "
								+ propertyHolder.getEntityName()
				);
			}
			log.debug( inferredData.getPropertyName() + " is a version property" );
			RootClass rootClass = (RootClass) propertyHolder.getPersistentClass();
			boolean lazy = false;
			//for now columns.length > 1 is impossible for @Version
			Type annType = (Type) annotatedElt.getAnnotation( Type.class );
			PropertyBinder propBinder = new PropertyBinder();
			propBinder.setName( inferredData.getPropertyName() );
			propBinder.setReturnedClassName( inferredData.getReturnedClassName() );
			propBinder.setLazy( lazy );
			propBinder.setPropertyAccessorName( inferredData.getDefaultAccess() );
			propBinder.setColumns( columns );
			propBinder.setHolder( propertyHolder ); //PropertyHolderBuilder.buildPropertyHolder(rootClass)
			propBinder.setAnnotatedElement( annotatedElt );
			propBinder.setReturnedClass( inferredData.getReturnedClass() );
			propBinder.setMappings( mappings );
			Property prop = propBinder.bind();
			rootClass.setVersion( prop );
			SimpleValue simpleValue = (SimpleValue) prop.getValue();
			if ( !simpleValue.isTypeSpecified() ) simpleValue.setTypeName( "integer" );
			simpleValue.setNullValue( "undefined" );
			rootClass.setOptimisticLockMode( Versioning.OPTIMISTIC_LOCK_VERSION );
			log.debug(
					"Version name: " + rootClass.getVersion().getName() + ", unsavedValue: " + ( (SimpleValue) rootClass
							.getVersion()
							.getValue() ).getNullValue()
			);
		}
		else if ( annotatedElt.isAnnotationPresent( ManyToOne.class ) ) {
			ManyToOne ann = annotatedElt.getAnnotation( ManyToOne.class );
			Cascade hibernateCascade = annotatedElt.getAnnotation( Cascade.class );
			NotFound notFound = annotatedElt.getAnnotation( NotFound.class );
			boolean ignoreNotFound = notFound != null && notFound.action().equals( NotFoundAction.IGNORE );
			bindManyToOne(
					getCascadeStrategy( ann.cascade(), hibernateCascade ),
					(Ejb3JoinColumn[]) joinColumns,
					ann.optional(),
					getFetchMode( ann.fetch() ),
					ignoreNotFound, inferredData.getPropertyName(),
					inferredData.getReturnedClassOrElementName(),
					ann.targetEntity(),
					inferredData.getDefaultAccess(),
					propertyHolder,
					false, isIdentifierMapper, mappings
			);
		}
		else if ( annotatedElt.isAnnotationPresent( OneToOne.class ) ) {
			OneToOne ann = annotatedElt.getAnnotation( OneToOne.class );
			boolean trueOneToOne = annotatedElt.isAnnotationPresent( PrimaryKeyJoinColumn.class );
			Cascade hibernateCascade = annotatedElt.getAnnotation( Cascade.class );
			NotFound notFound = annotatedElt.getAnnotation( NotFound.class );
			boolean ignoreNotFound = notFound != null && notFound.action().equals( NotFoundAction.IGNORE );
			bindOneToOne(
					getCascadeStrategy( ann.cascade(), hibernateCascade ),
					(Ejb3JoinColumn[]) joinColumns,
					ann.optional(),
					getFetchMode( ann.fetch() ),
					ignoreNotFound, inferredData.getPropertyName(),
					inferredData.getReturnedClassOrElementName(),
					ann.targetEntity(),
					inferredData.getDefaultAccess(),
					propertyHolder,
					ann.mappedBy(), trueOneToOne, isIdentifierMapper, mappings
			);
		}
		else if ( annotatedElt.isAnnotationPresent( OneToMany.class )
				|| annotatedElt.isAnnotationPresent( ManyToMany.class ) ) {
			OneToMany oneToManyAnn = annotatedElt.getAnnotation( OneToMany.class );
			ManyToMany manyToManyAnn = annotatedElt.getAnnotation( ManyToMany.class );
			org.hibernate.annotations.IndexColumn indexAnn = annotatedElt.getAnnotation(
					org.hibernate.annotations.IndexColumn.class
			);
			JoinTable assocTable = annotatedElt.getAnnotation( JoinTable.class );

			IndexColumn indexColumn = IndexColumn.buildColumnFromAnnotation(
					indexAnn, propertyHolder, inferredData, mappings
			);
			CollectionBinder collectionBinder = CollectionBinder.getCollectionBinder(
					propertyHolder.getEntityName(),
					inferredData,
					! indexColumn.isImplicit()
			);
			collectionBinder.setIndexColumn( indexColumn );
			MapKey mapKeyAnn = annotatedElt.getAnnotation( MapKey.class );
			collectionBinder.setMapKey( mapKeyAnn );
			collectionBinder.setPropertyName( inferredData.getPropertyName() );
			BatchSize batchAnn = annotatedElt.getAnnotation( BatchSize.class );
			collectionBinder.setBatchSize( batchAnn );
			javax.persistence.OrderBy ejb3OrderByAnn = annotatedElt.getAnnotation( javax.persistence.OrderBy.class );
			OrderBy orderByAnn = annotatedElt.getAnnotation( OrderBy.class );
			collectionBinder.setEjb3OrderBy( ejb3OrderByAnn );
			collectionBinder.setSqlOrderBy( orderByAnn );
			Sort sortAnn = annotatedElt.getAnnotation( Sort.class );
			collectionBinder.setSort( sortAnn );
			Cache cachAnn = annotatedElt.getAnnotation( Cache.class );
			collectionBinder.setCache( cachAnn );
			Filter filterAnn = annotatedElt.getAnnotation( Filter.class );
			if ( filterAnn != null ) {
				collectionBinder.addFilter( filterAnn.name(), filterAnn.condition() );
			}
			Filters filtersAnn = annotatedElt.getAnnotation( Filters.class );
			if ( filtersAnn != null ) {
				for ( Filter filter : filtersAnn.value() ) {
					collectionBinder.addFilter( filter.name(), filter.condition() );
				}
			}
			collectionBinder.setPropertyHolder( propertyHolder );
			Where whereAnn = annotatedElt.getAnnotation( Where.class );
			collectionBinder.setWhere( whereAnn );
			Cascade hibernateCascade = annotatedElt.getAnnotation( Cascade.class );
			NotFound notFound = annotatedElt.getAnnotation( NotFound.class );
			boolean ignoreNotFound = notFound != null && notFound.action().equals( NotFoundAction.IGNORE );
			collectionBinder.setIgnoreNotFound(ignoreNotFound);
			collectionBinder.setCollectionType( inferredData.getCollectionType() );
			collectionBinder.setMappings( mappings );
			collectionBinder.setPropertyAccessorName( inferredData.getDefaultAccess() );

			Ejb3Column[] elementColumns = null;
			if ( annotatedElt.isAnnotationPresent( Column.class ) || annotatedElt.isAnnotationPresent(
					Formula.class
			) ) {
				Column ann = (Column) annotatedElt.getAnnotation( Column.class );
				Formula formulaAnn = (Formula) annotatedElt.getAnnotation( Formula.class );
				elementColumns = Ejb3Column.buildColumnFromAnnotation(
						new Column[]{ann},
						formulaAnn,
						nullability,
						propertyHolder,
						inferredData,
						entityBinder.getSecondaryTables(),
						mappings
				);
			}
			else if ( annotatedElt.isAnnotationPresent( Columns.class ) ) {
				Columns anns = annotatedElt.getAnnotation( Columns.class );
				elementColumns = Ejb3Column.buildColumnFromAnnotation(
						anns.columns(), null, nullability, propertyHolder, inferredData,
						entityBinder.getSecondaryTables(), mappings
				);
			}

			//potential element
			collectionBinder.setEmbedded( annotatedElt.isAnnotationPresent( Embedded.class ) );
			collectionBinder.setElementColumns( elementColumns );
			collectionBinder.setAnnotatedElement( annotatedElt );

			if ( oneToManyAnn != null && manyToManyAnn != null ) {
				throw new AnnotationException(
						"@OneToMany and @ManyToMany on the same property is not allowed: "
								+ propertyHolder.getEntityName() + "." + inferredData.getPropertyName()
				);
			}
			String mappedBy = null;
			if ( oneToManyAnn != null ) {
				for ( Ejb3JoinColumn column : joinColumns ) {
					if ( column.isSecondary() ) {
						throw new NotYetImplementedException( "Collections having FK in secondary table" );
					}
				}
				collectionBinder.setFkJoinColumns( joinColumns );
				mappedBy = oneToManyAnn.mappedBy();
				collectionBinder.setTargetEntity( oneToManyAnn.targetEntity() );
				collectionBinder.setFetchType( oneToManyAnn.fetch() );
				collectionBinder.setCascadeStrategy( getCascadeStrategy( oneToManyAnn.cascade(), hibernateCascade ) );
				collectionBinder.setOneToMany( true );
			}
			else if ( manyToManyAnn != null ) {
				mappedBy = manyToManyAnn.mappedBy();
				collectionBinder.setTargetEntity( manyToManyAnn.targetEntity() );
				collectionBinder.setFetchType( manyToManyAnn.fetch() );
				collectionBinder.setCascadeStrategy( getCascadeStrategy( manyToManyAnn.cascade(), hibernateCascade ) );
				collectionBinder.setOneToMany( false );
			}
			collectionBinder.setMappedBy( mappedBy );
			bindJoinedTableAssociation(
					assocTable, mappings, entityBinder, collectionBinder, propertyHolder, inferredData, mappedBy
			);

			OnDelete onDeleteAnn = annotatedElt.getAnnotation( OnDelete.class );
			boolean onDeleteCascade = onDeleteAnn != null && OnDeleteAction.CASCADE.equals( onDeleteAnn.action() );
			collectionBinder.setCascadeDeleteEnabled( onDeleteCascade );
			if ( isIdentifierMapper ) {
				collectionBinder.setInsertable( false );
				collectionBinder.setUpdatable( false );
			}
			collectionBinder.bind();
		}
		else {
			//define whether the type is a component or not
			boolean isComponent = false;
			Embeddable embeddableAnn = (Embeddable) returnedClass.getAnnotation( Embeddable.class );
			Embedded embeddedAnn = (Embedded) annotatedElt.getAnnotation( Embedded.class );
			if ( embeddedAnn != null || embeddableAnn != null ) isComponent = true;

			if ( isComponent ) {
				//process component object
				boolean propertyAccess = true;
				if ( embeddableAnn != null && embeddableAnn.access() == AccessType.FIELD ) propertyAccess = false;
				Map<String, Column[]> columnOverride = PropertyHolderBuilder.buildColumnOverride(
						annotatedElt,
						StringHelper.qualify( propertyHolder.getPath(), inferredData.getPropertyName() )
				);
				bindComponent(
						inferredData, propertyHolder, propertyAccess, entityBinder, isIdentifierMapper, columnOverride,
						mappings
				);
			}
			else {
				//provide the basic property mapping
				boolean optional = true;
				boolean lazy = false;
				if ( annotatedElt.isAnnotationPresent( Basic.class ) ) {
					Basic ann = (Basic) annotatedElt.getAnnotation( Basic.class );
					optional = ann.optional();
					lazy = ann.fetch() == FetchType.LAZY;
				}
				else if ( annotatedElt.isAnnotationPresent( Lob.class ) ) {
					Lob lob = annotatedElt.getAnnotation( Lob.class );
					optional = lob.optional();
					lazy = lob.fetch() == FetchType.LAZY;
				}
				//implicit type will check basic types and Serializable classes
				if ( !optional && nullability != Nullability.FORCED_NULL ) {
					//force columns to not null
					for ( Ejb3Column col : columns ) {
						col.forceNotNull();
					}
				}

				PropertyBinder propBinder = new PropertyBinder();
				propBinder.setName( inferredData.getPropertyName() );
				propBinder.setReturnedClassName( inferredData.getReturnedClassName() );
				propBinder.setLazy( lazy );
				propBinder.setPropertyAccessorName( inferredData.getDefaultAccess() );
				propBinder.setColumns( columns );
				propBinder.setHolder( propertyHolder );
				propBinder.setAnnotatedElement( annotatedElt );
				propBinder.setReturnedClass( inferredData.getReturnedClass() );
				propBinder.setMappings( mappings );
				if ( isIdentifierMapper ) {
					propBinder.setInsertable( false );
					propBinder.setUpdatable( false );
				}
				propBinder.bind();
			}
		}
	}

	private static void bindJoinedTableAssociation(
			JoinTable joinTableAnn, ExtendedMappings mappings, EntityBinder entityBinder,
			CollectionBinder collectionBinder, PropertyHolder propertyHolder, PropertyInferredData inferredData,
			String mappedBy
	) {
		Table assocTable;
		JoinColumn[] annJoins;
		JoinColumn[] annInverseJoins;
		if ( joinTableAnn != null ) {
			collectionBinder.setExplicitAssociationTable( true );
			if ( ! joinTableAnn.table().specified() ) {
				assocTable = null;
			}
			else {
				javax.persistence.Table table = joinTableAnn.table();
				assocTable = TableBinder.fillTable(
						table.schema(),
						table.catalog(),
						mappings.getNamingStrategy().tableName( table.name() ),
						false, TableBinder.buildUniqueConstraints( table.uniqueConstraints() ),
						null, null, mappings
				);
			}
			//set check constaint in the second pass

			JoinColumn[] joins = joinTableAnn.joinColumns();

			if ( joins.length == 0 ) {
				annJoins = null;
			}
			else {
				annJoins = joins;
			}

			JoinColumn[] inverseJoins = joinTableAnn.inverseJoinColumns();

			if ( inverseJoins.length == 0 ) {
				annInverseJoins = null;
			}
			else {
				annInverseJoins = inverseJoins;
			}
		}
		else {
			assocTable = null;
			annJoins = null;
			annInverseJoins = null;
		}
		Ejb3JoinColumn[] joinColumns = buildJoinTableJoinColumns(
				annJoins, entityBinder.getSecondaryTables(), propertyHolder, inferredData.getPropertyName(), mappedBy,
				mappings
		);
		Ejb3JoinColumn[] inverseJoinColumns = buildJoinTableJoinColumns(
				annInverseJoins, entityBinder.getSecondaryTables(), propertyHolder, inferredData.getPropertyName(),
				mappedBy, mappings
		);
		collectionBinder.setTable( assocTable );
		collectionBinder.setJoinColumns( joinColumns );
		collectionBinder.setInverseJoinColumns( inverseJoinColumns );
	}

	private static Ejb3JoinColumn[] buildJoinTableJoinColumns(
			JoinColumn[] annJoins, Map<String, Join> secondaryTables,
			PropertyHolder propertyHolder, String propertyName, String mappedBy, ExtendedMappings mappings
	) {
		Ejb3JoinColumn[] joinColumns;
		if ( annJoins == null ) {
			Ejb3JoinColumn currentJoinColumn = new Ejb3JoinColumn();
			currentJoinColumn.setImplicit( true );
			currentJoinColumn.setNullable( false ); //I break the spec, but it's for good
			currentJoinColumn.setPropertyHolder( propertyHolder );
			currentJoinColumn.setJoins( secondaryTables );
			currentJoinColumn.setMappings( mappings );
			currentJoinColumn.setPropertyName( propertyName );
			currentJoinColumn.setMappedBy( mappedBy );
			currentJoinColumn.bind();

			joinColumns = new Ejb3JoinColumn[]{
					currentJoinColumn

			};
		}
		else {
			joinColumns = new Ejb3JoinColumn[annJoins.length];
			JoinColumn annJoin;
			int length = annJoins.length;
			for ( int index = 0; index < length ; index++ ) {
				annJoin = annJoins[index];
				Ejb3JoinColumn currentJoinColumn = new Ejb3JoinColumn();
				currentJoinColumn.setImplicit( true );
				currentJoinColumn.setPropertyHolder( propertyHolder );
				currentJoinColumn.setJoins( secondaryTables );
				currentJoinColumn.setMappings( mappings );
				currentJoinColumn.setPropertyName( propertyName );
				currentJoinColumn.setMappedBy( mappedBy );
				currentJoinColumn.setJoinAnnotation( annJoin, propertyName );
				currentJoinColumn.setNullable( false ); //I break the spec, but it's for good
				//done after the annotation to override it
				currentJoinColumn.bind();
				joinColumns[index] = currentJoinColumn;
			}
		}
		return joinColumns;
	}

	private static void bindComponent(
			PropertyInferredData inferredData,
			PropertyHolder propertyHolder,
			boolean propertyAccess,
			EntityBinder entityBinder,
			boolean isIdentifierMapper, Map<String, Column[]> columnOverride,
			ExtendedMappings mappings
	) {
		Component comp = fillComponent(
				propertyHolder, inferredData, propertyAccess, true, entityBinder, isIdentifierMapper, columnOverride,
				mappings
		);

		PropertyBinder binder = new PropertyBinder();
		binder.setName( inferredData.getPropertyName() );
		binder.setValue( comp );
		binder.setPropertyAccessorName( inferredData.getDefaultAccess() );
		Property prop = binder.make();
		propertyHolder.addProperty( prop );
	}

	public static Component fillComponent(
			PropertyHolder propertyHolder, PropertyInferredData inferredData,
			boolean propertyAccess, boolean isNullable, EntityBinder entityBinder,
			boolean isIdentifierMapper, Map<String, Column[]> columnOverride, ExtendedMappings mappings
	) {
		Component comp = new Component( propertyHolder.getPersistentClass() );
		//yuk
		comp.setTable( propertyHolder.getTable() );
		if ( !isIdentifierMapper ) {
			comp.setComponentClassName( inferredData.getReturnedClassOrElementName() );
		}
		else {
			comp.setComponentClassName( comp.getOwner().getClassName() );
		}
		String subpath = StringHelper.qualify( propertyHolder.getPath(), inferredData.getPropertyName() );
		log.debug( "Binding component with path: " + subpath );
		Map<String, Column[]> localColumnOverride = propertyHolder.mergeOverridenColumns( columnOverride );
		PropertyHolder subHolder = PropertyHolderBuilder.buildPropertyHolder( comp, subpath, localColumnOverride );
		List<PropertyAnnotatedElement> classElements = new ArrayList<PropertyAnnotatedElement>();
		addElementsOfAClass(
				classElements,
				subHolder,
				propertyAccess,
				inferredData.getReturnedClassOrElement()
		);
		//add elements of the embeddable superclass
		Class superClass = inferredData.getReturnedClass().getSuperclass();
		while ( superClass != null && superClass.isAnnotationPresent( EmbeddableSuperclass.class ) ) {
			addElementsOfAClass(
					classElements,
					subHolder,
					AccessType.PROPERTY == ( (EmbeddableSuperclass) superClass.getAnnotation(
							EmbeddableSuperclass.class
					) ).access(),
					superClass
			);
			superClass = superClass.getSuperclass();
		}
		for ( PropertyAnnotatedElement propertyAnnotatedElement : classElements ) {
			processElementAnnotations(
					subHolder, isNullable ? Nullability.NO_CONSTRAINT : Nullability.FORCED_NOT_NULL,
					propertyAnnotatedElement.element, propertyAnnotatedElement.inferredData,
					new HashMap<String, IdGenerator>(),
					new HashMap<String, Properties>(), entityBinder, isIdentifierMapper, mappings
			);
		}
		return comp;
	}

	private static void bindId(
			String generatorType, String generatorName,
			PropertyInferredData inferredData, Ejb3Column column, PropertyHolder propertyHolder,
			Map<String, IdGenerator> localGenerators, HashMap<String, Properties> localGeneratorTables,
			boolean isComposite, Map<String, Column[]> columnOverride, boolean isPropertyAccess,
			EntityBinder entityBinder, Type typeAnn, boolean isEmbedded, ExtendedMappings mappings
	) {
		/*
		 * Fill simple value and property since and Id is a property
		 */
		PersistentClass persistentClass = propertyHolder.getPersistentClass();
		if ( ! ( persistentClass instanceof RootClass ) ) {
			throw new AnnotationException(
					"Unable to define/override @Id(s) on a subclass: "
							+ propertyHolder.getEntityName()
			);
		}
		RootClass rootClass = (RootClass) persistentClass;
		String persistentClassName = rootClass == null ? null : rootClass.getClassName();
		SimpleValue id;
		if ( isComposite ) {
			id = fillComponent(
					propertyHolder, inferredData, isPropertyAccess, false, entityBinder, false, columnOverride, mappings
			);
			( (Component) id ).setKey( true );
		}
		else {
			column.forceNotNull(); //this is an id
			SimpleValueBinder value = new SimpleValueBinder();
			value.setPropertyName( inferredData.getPropertyName() );
			value.setReturnedClassName( inferredData.getReturnedClassName() );
			value.setColumns( new Ejb3Column[]{column} );
			value.setPersistentClassName( persistentClassName );
			value.setMappings( mappings );
			value.setExplicitType( typeAnn );
			id = value.make();
		}
		rootClass.setIdentifier( id );
		Table table = id.getTable();
		table.setIdentifierValue( id );
		//generator settings
		id.setIdentifierGeneratorStrategy( generatorType );
		Properties params = new Properties();
		//always settable
		params.setProperty(
				PersistentIdentifierGenerator.TABLE, table.getName()
		);
		params.setProperty(
				PersistentIdentifierGenerator.PK,
				( (org.hibernate.mapping.Column) id.getColumnIterator().next() ).getName()
		);
		if ( ! isDefault( generatorName ) ) {
			//we have a named generator
			IdGenerator gen = mappings.getGenerator( generatorName, localGenerators );
			if ( gen == null ) {
				throw new AnnotationException( "Unknown Id.generator: " + generatorName );
			}
			//This is quite vague in the spec but a generator could override the generate choice
			final boolean overrideType = ! generatorType( GeneratorType.AUTO ).equals( generatorType );
			if ( overrideType ) {
				id.setIdentifierGeneratorStrategy( gen.getIdentifierGeneratorStrategy() );
			}
			//checkIfMatchingGenerator(gen, generatorType, generatorName);
			Iterator genParams = gen.getParams().entrySet().iterator();
			while ( genParams.hasNext() ) {
				Map.Entry elt = (Map.Entry) genParams.next();
				params.setProperty( (String) elt.getKey(), (String) elt.getValue() );
			}
			if ( MultipleHiLoPerTableGenerator.class.getName().equals( generatorType ) ) {
				//try and find the associated Generator Table
				fillGeneratorWithGeneratorTableParams( params, localGeneratorTables, generatorName, mappings );
			}
		}
		if ( generatorType == "assigned" ) id.setNullValue( "undefined" );
		id.setIdentifierGeneratorProperties( params );
		if ( isEmbedded ) {
			rootClass.setEmbeddedIdentifier( inferredData.getReturnedClass() == null );
		}
		else {
			PropertyBinder binder = new PropertyBinder();
			binder.setName( inferredData.getPropertyName() );
			binder.setValue( id );
			binder.setPropertyAccessorName( inferredData.getDefaultAccess() );
			Property prop = binder.make();
			rootClass.setIdentifierProperty( prop );
		}
	}

	/**
	 * @param gen
	 * @param generatorType
	 * @param generatorName
	 */
	private static void checkIfMatchingGenerator(IdGenerator gen, String generatorType, String generatorName) {
		//this means: the generators are of the same type or
		//Id annotation's generatorType == AUTO
		//the default generator can be either Sequence, Table or Id, so we should let it go wo restriction
		boolean matchingGenerator = gen.getIdentifierGeneratorStrategy().equals( generatorType );
		boolean defaultGenerator = generatorType( GeneratorType.AUTO ).equals( generatorType );

		if ( ! ( matchingGenerator || defaultGenerator ) ) {
			//named generator and id one should be compatible
			throw new AnnotationException(
					"Incompatible generator between Id.generate and its named generator: "
							+ generatorType + "!=" + generatorName
			);
		}
	}

	private static void fillGeneratorWithGeneratorTableParams(
			Properties params, HashMap<String, Properties> localGeneratorTables, String generatorName,
			ExtendedMappings mappings
	) {
		final String generatorTableName = params.getProperty( GENERATOR_TABLE_NAME_PARAM );
		Properties props = mappings.getGeneratorTableProperties( generatorTableName, localGeneratorTables );
		if ( props == null ) {
			if ( MultipleHiLoPerTableGenerator.DEFAULT_TABLE.equals( generatorTableName ) ) {
				//default value
				return;
			}
			else {
				throw new AnnotationException(
						"Unable to find a @GeneratedIdTable for table name in " + generatorName + ": " + generatorTableName
				);
			}
		}
		else {
			Iterator properties = props.entrySet().iterator();
			java.util.Map.Entry<String, String> property;
			while ( properties.hasNext() ) {
				property = (java.util.Map.Entry<String, String>) properties.next();
				params.setProperty( property.getKey(), property.getValue() );
			}
		}
	}

	private static void bindManyToOne(
			String cascadeStrategy, Ejb3JoinColumn[] columns, boolean optional, FetchMode fetchMode,
			boolean ignoreNotFound, String propertyName,
			String returnedClassName, Class targetEntity, String propertyAccessorName, PropertyHolder propertyHolder,
			boolean unique, boolean isIdentifierMapper, ExtendedMappings mappings
	) {
		//All FK columns should be in the same table
		org.hibernate.mapping.ManyToOne value = new org.hibernate.mapping.ManyToOne( columns[0].getTable() );
		if ( isDefault( targetEntity ) ) {
			value.setReferencedEntityName( returnedClassName );
		}
		else {
			value.setReferencedEntityName( targetEntity.getName() );
		}
		value.setFetchMode( fetchMode );
		value.setIgnoreNotFound( ignoreNotFound );
		value.setLazy( fetchMode != FetchMode.JOIN );
		if ( !optional ) {
			for ( Ejb3JoinColumn column : columns ) {
				column.setNullable( false );
			}
		}
		value.setTypeName( returnedClassName );
		value.setTypeUsingReflection( propertyHolder.getClassName(), propertyName );

		//value.createForeignKey();
		mappings.addSecondPass( new FkSecondPass( value, columns, unique, mappings ) );

		Ejb3Column.checkPropertyConsistency( columns, propertyHolder.getEntityName() + propertyName );
		PropertyBinder binder = new PropertyBinder();
		binder.setName( propertyName );
		binder.setValue( value );
		//binder.setCascade(cascadeStrategy);
		if ( isIdentifierMapper ) {
			binder.setInsertable( false );
			binder.setInsertable( false );
		}
		else {
			binder.setInsertable( columns[0].isInsertable() );
			binder.setUpdatable( columns[0].isUpdatable() );
		}
		binder.setPropertyAccessorName( propertyAccessorName );
		binder.setCascade( cascadeStrategy );
		Property prop = binder.make();
		//composite FK columns are in the same table so its OK
		propertyHolder.addProperty( prop, columns );
	}

	public static void bindFkSecondPass(
			org.hibernate.mapping.ManyToOne manyToOne, Ejb3JoinColumn[] columns, Map persistentClasses, boolean unique,
			ExtendedMappings mappings
	) {
		PersistentClass ref = (PersistentClass) persistentClasses.get( manyToOne.getReferencedEntityName() );
		if ( ref == null ) {
			throw new AnnotationException(
					"Unable to find entity " + manyToOne.getReferencedEntityName()
			);
		}
		BinderHelper.createSyntheticPropertyReference( columns, ref, null, manyToOne, false, mappings );
		TableBinder.bindFk( ref, null, columns, manyToOne, unique );
	}

	private static void bindOneToOne(
			String cascadeStrategy,
			Ejb3JoinColumn[] columns,
			boolean optional,
			FetchMode fetchMode,
			boolean ignoreNotFound, String propertyName,
			String returnedClassName,
			Class targetEntity,
			String propertyAccessorName,
			PropertyHolder propertyHolder,
			String mappedBy,
			boolean trueOneToOne,
			boolean isIdentifierMapper, ExtendedMappings mappings
	) {
		//column.getTable() => persistentClass.getTable()
		log.debug( "Fetching " + propertyName + " with " + fetchMode );
		boolean mapToPK = true;
		if ( ! trueOneToOne ) {
			//try to find a hidden true one to one (FK == PK columns)
			Iterator idColumns = propertyHolder.getIdentifier().getColumnIterator();
			List<String> idColumnNames = new ArrayList<String>();
			org.hibernate.mapping.Column currentColumn;
			while ( idColumns.hasNext() ) {
				currentColumn = (org.hibernate.mapping.Column) idColumns.next();
				idColumnNames.add( currentColumn.getName() );
			}
			for ( Ejb3JoinColumn col : columns ) {
				if ( ! idColumnNames.contains( col.getMappingColumn().getName() ) ) {
					mapToPK = false;
					break;
				}
			}
		}
		if ( trueOneToOne || mapToPK || ! isDefault( mappedBy ) ) {
			//is a true one-to-one
			//FIXME referencedColumnName ignored => ordering may fail.
			org.hibernate.mapping.OneToOne value = new org.hibernate.mapping.OneToOne(
					propertyHolder.getTable(), propertyHolder.getPersistentClass()
			);
			value.setPropertyName( propertyName );
			if ( isDefault( targetEntity ) ) {
				value.setReferencedEntityName( returnedClassName );
			}
			else {
				value.setReferencedEntityName( targetEntity.getName() );
			}
			value.setFetchMode( fetchMode );
			value.setLazy( fetchMode != FetchMode.JOIN );

			if ( !optional ) value.setConstrained( true );
			value.setForeignKeyType(
					value.isConstrained() ?
							ForeignKeyDirection.FOREIGN_KEY_FROM_PARENT :
							ForeignKeyDirection.FOREIGN_KEY_TO_PARENT
			);

			if ( ! isDefault( mappedBy) ) {
				mappings.addSecondPass( new ToOneMappedBySecondPass(
						mappedBy,
						value,
						propertyHolder.getEntityName(),
						propertyName,
						mappings)
				);
			}
//			if ( ! isDefault( mappedBy ) ) value.setReferencedPropertyName( mappedBy );
//
//			String propertyRef = value.getReferencedPropertyName();
//			if ( propertyRef != null ) {
//				mappings.addUniquePropertyReference(
//						value.getReferencedEntityName(),
//						propertyRef
//				);
//			}
			//value.createForeignKey();
			PropertyBinder binder = new PropertyBinder();
			binder.setName( propertyName );
			binder.setValue( value );
			binder.setCascade( cascadeStrategy );
			binder.setPropertyAccessorName( propertyAccessorName );
			Property prop = binder.make();
			prop.setCascade( cascadeStrategy );
			//no column associated since its a one to one
			propertyHolder.addProperty( prop );
		}
		else {
			//has a FK on the table
			bindManyToOne(
					cascadeStrategy, columns, optional, fetchMode, ignoreNotFound, propertyName, returnedClassName,
					targetEntity,
					propertyAccessorName, propertyHolder, true, isIdentifierMapper, mappings
			);
		}
	}

	private static String generatorType(GeneratorType generatorEnum) {
		switch ( generatorEnum ) {
			case NONE :
				return "assigned";
			case IDENTITY:
				return "identity";
			case AUTO:
				return "native";
			case TABLE:
				return MultipleHiLoPerTableGenerator.class.getName();
			case SEQUENCE:
				return "sequence";
		}
		throw new AssertionFailure( "Unknown GeneratorType: " + generatorEnum );
	}

	private static EnumSet<CascadeType> convertToHibernateCascadeType(javax.persistence.CascadeType[] ejbCascades) {
		EnumSet<CascadeType> hibernateCascadeSet = EnumSet.noneOf( CascadeType.class );
		if ( ejbCascades != null && ejbCascades.length > 0 ) {
			for ( javax.persistence.CascadeType cascade : ejbCascades ) {
				switch ( cascade ) {
					case ALL:
						hibernateCascadeSet.add( CascadeType.ALL );
						break;
					case PERSIST:
						hibernateCascadeSet.add( CascadeType.PERSIST );
						break;
					case MERGE:
						hibernateCascadeSet.add( CascadeType.MERGE );
						break;
					case REMOVE:
						hibernateCascadeSet.add( CascadeType.REMOVE );
						break;
					case REFRESH:
						hibernateCascadeSet.add( CascadeType.REFRESH );
						break;
				}
			}
		}

		return hibernateCascadeSet;
	}

	private static String getCascadeStrategy(
			javax.persistence.CascadeType[] ejbCascades, Cascade hibernateCascadeAnnotation
	) {
		EnumSet<CascadeType> hibernateCascadeSet = convertToHibernateCascadeType( ejbCascades );
		CascadeType[] hibernateCascades = hibernateCascadeAnnotation == null ? null : hibernateCascadeAnnotation.value();

		if ( hibernateCascades != null && hibernateCascades.length > 0 ) {
			for ( CascadeType cascadeType : hibernateCascades ) {
				hibernateCascadeSet.add( cascadeType );
			}
		}

		StringBuilder cascade = new StringBuilder();
		Iterator<CascadeType> cascadeType = hibernateCascadeSet.iterator();
		while ( cascadeType.hasNext() ) {
			switch ( cascadeType.next() ) {
				case ALL:
					cascade.append( "," ).append( "all" );
					break;
				case SAVE_UPDATE:
					cascade.append( "," ).append( "save-update" );
					break;
				case PERSIST:
					cascade.append( "," ).append( "persist" );
					break;
				case MERGE:
					cascade.append( "," ).append( "merge" );
					break;
				case LOCK:
					cascade.append( "," ).append( "lock" );
					break;
				case REFRESH:
					cascade.append( "," ).append( "refresh" );
					break;
				case REPLICATE:
					cascade.append( "," ).append( "replicate" );
					break;
				case EVICT:
					cascade.append( "," ).append( "evict" );
					break;
				case DELETE:
					cascade.append( "," ).append( "delete" );
					break;
				case DELETE_ORPHAN:
					cascade.append( "," ).append( "delete-orphan" );
					break;
				case REMOVE:
					cascade.append( "," ).append( "delete" );
					break;
			}
		}
		return cascade.length() > 0 ? cascade.substring( 1 ) : "none";
	}

	private static FetchMode getFetchMode(FetchType fetch) {
		if ( fetch == FetchType.EAGER ) {
			return FetchMode.JOIN;
		}
		else {
			return FetchMode.SELECT;
		}
	}

	private static HashMap<String, IdGenerator> buildLocalGenerators(AnnotatedElement annElt, Mappings mappings) {
		HashMap<String, IdGenerator> generators = new HashMap<String, IdGenerator>();
		TableGenerator tabGen = annElt.getAnnotation( TableGenerator.class );
		SequenceGenerator seqGen = annElt.getAnnotation( SequenceGenerator.class );
		GenericGenerator genGen = annElt.getAnnotation( GenericGenerator.class );
		if ( tabGen != null ) {
			IdGenerator idGen = buildIdGenerator( tabGen, mappings );
			generators.put( idGen.getName(), idGen );
		}
		if ( seqGen != null ) {
			IdGenerator idGen = buildIdGenerator( seqGen, mappings );
			generators.put( idGen.getName(), idGen );
		}
		if ( genGen != null ) {
			IdGenerator idGen = buildIdGenerator( genGen, mappings );
			generators.put( idGen.getName(), idGen );
		}
		return generators;
	}

	public static boolean isDefault(String annotationString) {
		return ANNOTATION_STRING_DEFAULT.equals( annotationString );
	}

	public static boolean isDefault(Class clazz) {
		return void.class.equals( clazz );
	}

	public static Map<Class, InheritanceState> buildInheritanceStates(List<Class> orderedClasses) {
		Map<Class, InheritanceState> inheritanceStatePerClass = new HashMap<Class, InheritanceState>(
				orderedClasses.size()
		);
		for ( Class clazz : orderedClasses ) {
			Class superClazz = clazz.getSuperclass();
			InheritanceState state = new InheritanceState();
			state.setInheritanceType( clazz );
			if ( orderedClasses.contains( superClazz ) ) {
				//the classes are ordered thus preventing an NPE
				InheritanceState superState = inheritanceStatePerClass.get( superClazz );
				superState.hasSons = true;
				if ( superState.isEmbeddableSuperclass ) {
					state.hasParents = false;
					state.hasEmbeddedSuperclass = true;
				}
				else {
					state.hasParents = true;
					state.hasEmbeddedSuperclass = false;
					if ( state.isEmbeddableSuperclass ) {
						throw new AnnotationException( "@EmbeddableSuperclass cannot be a subclass of @Entity" );
					}
				}
				final boolean nonDefault = state.type != null && ! InheritanceType.SINGLE_TABLE.equals( state.type );
				if ( superState.type != null ) {
					final boolean mixingStrategy = ! state.type.equals( superState.type );
					if ( nonDefault && mixingStrategy ) {
						log.warn(
								"Mixing inheritance strategy in a entity hierarchy is not allowed, ignoring sub strategy in: " + clazz
										.getName()
						);
					}
					state.type = superState.type;
				}
			}
			inheritanceStatePerClass.put( clazz, state );
		}
		return inheritanceStatePerClass;
	}

	private static class PropertyAnnotatedElement {
		public PropertyAnnotatedElement(AnnotatedElement elt) {
			element = elt;
			inferredData = new PropertyInferredData( element );
		}

		public AnnotatedElement element;
		public PropertyInferredData inferredData;
	}
}
