//$Id: PropertyInferredData.java,v 1.3 2005/07/11 16:23:29 epbernard Exp $
package org.hibernate.cfg;

import java.beans.Introspector;
import java.lang.reflect.AnnotatedElement;
import java.lang.reflect.Field;
import java.lang.reflect.Member;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;

import org.hibernate.MappingException;

/**
 * Retrieve all inferred data from an annnoted element
 * Currently support field and getter processing.
 * An exception is thrown when the annoted element does not fit.
 * The calculation is lazied until the first access to a public getter.
 * This prevent useless reflection.
 * 
 * @author Emmanuel Bernard
 */
public class PropertyInferredData {
	private String defaultAccess;
	private String propertyName;
	private Class returnedClassOrElement;
	private Class returnedClass;
	private Class collectionType;
	private String returnedClassOrElementName;
	private String returnedClassName;
	private final AnnotatedElement annotedElt;
	private boolean processed;
	private boolean annotable;
	private boolean skip;
	private boolean isArray;

	public PropertyInferredData(String defaultAccess, String propertyName, Class returnedClass) {
		this.defaultAccess = defaultAccess;
		this.processed = true;
		this.propertyName = propertyName;
		this.returnedClass = returnedClass;
		this.returnedClassOrElement = returnedClass;
		this.returnedClassName = returnedClass == null ? null : returnedClass.getName();
		this.returnedClassOrElementName = returnedClassName;
		annotedElt = null;
		annotable = true;
		skip = false;
		isArray = false;
	}

	public boolean skip() {
		execute(false);
		return skip;
	}

	private static final String EXCEPTION_MESSAGE =
		"The annoted inferred element should be a field or a getter";

	/**
	 * Take the annoted element for lazy process
	 * 
	 * @param annotedElt element to process
	 */
	public PropertyInferredData(AnnotatedElement annotedElt) {
		this.annotedElt = annotedElt;
	}
	
	/**
	 * Process an annoted element and give a final
	 * structure.
	 * The annoted element must be either field or getter
	 * 
	 * @throws MappingException No getter or field found or wrong JavaBean spec usage
	 */
	private void execute(boolean throwException) throws MappingException {
		if (! processed) {
			/*
			 * guess access from annoted element type
			 * retrieve property name.
			 */
			skip = false;
			annotable = true;
			if ( ! (annotedElt instanceof Member) ) {
				throw new MappingException(EXCEPTION_MESSAGE);
			}
			propertyName = retrievePropertyNameFromMember( (Member) annotedElt, throwException );
			if (annotedElt instanceof Field) {
				Field field = (Field) annotedElt;
				defaultAccess = "field";
				returnedClass = field.getType();
				collectionType = findCollectionType(field);
				//TODO check whether synthetic shoud be excluded or not?
				if ( field.isSynthetic() ||
					 Modifier.isStatic( field.getModifiers() )
				) {
					skip = true;
				}
			}
			else if (annotedElt instanceof Method) {
				Method method = (Method) annotedElt;
				defaultAccess = "property";
				returnedClass = method.getReturnType();
				collectionType = findCollectionType(method);
				//TODO check whether synthetic shoud be excluded or not?
				if ( method.isSynthetic() ||
					 method.isBridge() ||
					 Modifier.isStatic( method.getModifiers() ) ||
					 method.getParameterTypes().length != 0 || //not a getter
				     void.class.equals( method.getReturnType() ) //not a getter
				) {
					skip = true;
				}
			}
			else {
				throw new MappingException(EXCEPTION_MESSAGE);
			}
			if ( returnedClass.isArray() ) {
				isArray = true;
				returnedClassOrElement = returnedClass.getComponentType();
				collectionType = returnedClassOrElement;
			}
			else {
				returnedClassOrElement = returnedClass;
			}
			returnedClassName = returnedClass.getName();
			returnedClassOrElementName = returnedClassOrElement.getName();
			if (annotable == false) skip = true;
			processed = true;
		}
	}
	
	private Class findCollectionType(Method method) {
		Type t = method.getGenericReturnType();
		return extractType(t);
	}
	
	private Class findCollectionType(Field field) {
		Type t = field.getGenericType();
		return extractType(t);
	}
	
	private Class extractType(Type t) {
		if (t != null && t instanceof ParameterizedType) {
			ParameterizedType pt = (ParameterizedType) t;
			Type[] genTypes = pt.getActualTypeArguments();
			if (genTypes.length == 1 && genTypes[0] instanceof Class) {
				return (Class) genTypes[0];
			}
			else if (genTypes.length == 2 && genTypes[1] instanceof Class) {
				//TODO we might want to store the index type at some point
				return (Class) genTypes[1];
			}
		}
		return null;
	}
	
	/**
	 * Build the property name from the Member one using JavaBean conventions
	 * 
	 * @param member member used to build
	 * @return property name
	 * 
	 * @throws MappingException annoted part not on a getter method
	 */
	private String retrievePropertyNameFromMember(Member member, boolean throwException) throws MappingException {
		
		if (member instanceof Field) {
			return member.getName();
		} 
		else if (member instanceof Method) {
			final String methName = member.getName();
			if ( methName.startsWith( "get" ) ) {

				return Introspector.decapitalize( methName.substring( "get".length() ) );
			}
			else if ( methName.startsWith("is") ) {
				return Introspector.decapitalize( methName.substring( "is".length() ) );
			}
			else {
				annotable = false;
				if (throwException) {
					throw new MappingException("Annotated Method is not a proper getter: " + methName);
				} 
				else {
					return methName;
				}
			}
		} 
		else {
			throw new MappingException(EXCEPTION_MESSAGE);
		}
	}
	
	

	/**
	 * @return default member access (whether field or property)
	 * @throws MappingException No getter or field found or wrong JavaBean spec usage
	 */
	public String getDefaultAccess() throws MappingException {
		execute(true);
		return defaultAccess;
	}

	/**
	 * @return property name
	 * @throws MappingException No getter or field found or wrong JavaBean spec usage
	 */
	public String getPropertyName() throws MappingException {
		execute(true);
		return propertyName;
	}

	/**
	 * Returns the returned class itself or the element type if an array
	 */
	public Class getReturnedClassOrElement() throws MappingException {
		execute(true);
		return returnedClassOrElement;
	}

	/**
	 * Return the class itself
	 */
	public Class getReturnedClass() throws MappingException {
		execute(true);
		return returnedClass;
	}

	/**
	 * Returns the returned class name itself or the element type if an array
	 */
	public String getReturnedClassOrElementName() throws MappingException {
		execute(true);
		return returnedClassOrElementName;
	}

	/**
	 * Returns the returned class name itself
	 */
	public String getReturnedClassName() throws MappingException {
		execute(true);
		return returnedClassName;
	}

	/**
	 * @return collection type if any, or null otherwise
	 */
	public Class getCollectionType() {
		return collectionType;
	}

	public boolean isArray() {
		return isArray;
	}
}
