/*
 * Decompiled with CFR 0.152.
 */
package fr.delthas.skype;

import fr.delthas.skype.ContactRequest;
import fr.delthas.skype.ErrorListener;
import fr.delthas.skype.Group;
import fr.delthas.skype.GroupMessageListener;
import fr.delthas.skype.GroupPropertiesListener;
import fr.delthas.skype.NotifConnector;
import fr.delthas.skype.Pair;
import fr.delthas.skype.Presence;
import fr.delthas.skype.Role;
import fr.delthas.skype.User;
import fr.delthas.skype.UserMessageListener;
import fr.delthas.skype.UserPresenceListener;
import fr.delthas.skype.WebConnector;
import java.io.IOException;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.logging.FileHandler;
import java.util.logging.Handler;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.logging.SimpleFormatter;
import java.util.stream.Collectors;

public final class Skype {
    private static final Logger logger = Logger.getLogger("fr.delthas.skype");
    private final String username;
    private final String password;
    private List<UserMessageListener> userMessageListeners = new LinkedList<UserMessageListener>();
    private List<GroupMessageListener> groupMessageListeners = new LinkedList<GroupMessageListener>();
    private List<UserPresenceListener> userPresenceListeners = new LinkedList<UserPresenceListener>();
    private List<GroupPropertiesListener> groupPropertiesListeners = new LinkedList<GroupPropertiesListener>();
    private ErrorListener errorListener;
    private NotifConnector notifConnector;
    private WebConnector webConnector;
    private Map<String, Group> groups;
    private Set<User> contacts;
    private Map<String, User> users;
    private List<ContactRequest> contactRequests;
    private boolean connected = false;
    private boolean connecting = false;
    private IOException exceptionDuringConnection;

    public Skype(String username, String password) {
        this.username = username;
        this.password = password;
    }

    public static void setDebug(Path path) throws IOException {
        if (path == null) {
            logger.setLevel(Level.OFF);
        } else {
            logger.setLevel(Level.ALL);
            logger.setUseParentHandlers(false);
            for (Handler handler : logger.getHandlers()) {
                logger.removeHandler(handler);
                handler.close();
            }
            FileHandler fh = new FileHandler(path.toString(), false);
            fh.setFormatter(new SimpleFormatter());
            logger.addHandler(fh);
        }
    }

    public void connect() throws IOException, InterruptedException {
        this.connect(Presence.ONLINE);
    }

    public void connect(Presence presence) throws IOException, InterruptedException {
        if (presence == Presence.OFFLINE) {
            throw new IllegalArgumentException("Presence can't be set to offline. Use HIDDEN if you want to connect without being visible.");
        }
        if (this.connecting || this.connected) {
            return;
        }
        this.connected = true;
        this.connecting = true;
        logger.fine("Connecting to Skype");
        this.reset();
        this.webConnector.start();
        this.getSelf().setPresence(presence, false);
        this.notifConnector.connect();
        this.connecting = false;
        if (this.exceptionDuringConnection != null) {
            throw new IOException(this.exceptionDuringConnection);
        }
    }

    public void disconnect() {
        if (this.connecting || !this.connected) {
            return;
        }
        this.connected = false;
        logger.fine("Disconnecting from Skype");
        this.notifConnector.disconnect();
        for (Map.Entry<String, User> user : this.users.entrySet()) {
            user.getValue().setPresence(Presence.OFFLINE, false);
        }
        this.reset();
    }

    public List<ContactRequest> getContactRequests() {
        this.ensureConnected();
        return Collections.unmodifiableList(new ArrayList<ContactRequest>(this.contactRequests));
    }

    public List<Group> getGroups() {
        this.ensureConnected();
        return Collections.unmodifiableList(new ArrayList<Group>(this.groups.values()));
    }

    public User getSelf() {
        this.ensureConnected();
        return this.getUser(this.username);
    }

    public List<User> getContacts() {
        this.ensureConnected();
        return Collections.unmodifiableList(new ArrayList<User>(this.contacts));
    }

    public void changePresence(Presence presence) {
        if (presence == Presence.OFFLINE) {
            throw new IllegalArgumentException("Presence can't be set to offline. Use HIDDEN if you want to connect without being visible.");
        }
        this.ensureConnected();
        try {
            logger.finer("Changing presence to " + (Object)((Object)presence));
            this.notifConnector.changePresence(presence);
        }
        catch (IOException e) {
            this.error(e);
        }
    }

    public boolean isConnected() {
        return this.connected;
    }

    User getUser(String username) {
        User user = this.users.get(username);
        if (user == null) {
            user = new User(this, username);
            this.users.put(username, user);
        }
        return user;
    }

    Group getGroup(String id) {
        Group group = this.groups.get(id);
        if (group == null) {
            group = new Group(this, id);
            this.groups.put(id, group);
        }
        return group;
    }

    void addContact(String username) {
        logger.finest("Adding contact " + username);
        this.contacts.add(this.getUser(username));
    }

    void error(IOException e) {
        logger.log(Level.SEVERE, "Error thrown", e);
        if (this.errorListener != null) {
            this.errorListener.error(e);
        } else {
            logger.severe("No error listener set!!!");
        }
        if (this.connecting) {
            this.exceptionDuringConnection = e;
        } else {
            this.disconnect();
        }
    }

    private void ensureConnected() throws IllegalStateException {
        if (!this.connected) {
            throw new IllegalStateException("Not connected to Skype!");
        }
    }

    private void reset() {
        logger.finest("Resetting the Skype object");
        this.notifConnector = new NotifConnector(this, this.username, this.password);
        this.webConnector = new WebConnector(this, this.username, this.password);
        this.groups = new HashMap<String, Group>();
        this.contacts = new HashSet<User>();
        this.users = new HashMap<String, User>();
        this.contactRequests = new LinkedList<ContactRequest>();
        this.exceptionDuringConnection = null;
    }

    void block(User user) {
        this.ensureConnected();
        try {
            logger.finer("Blocking user: " + user);
            this.webConnector.block(user);
        }
        catch (IOException e) {
            this.error(e);
        }
    }

    void unblock(User user) {
        this.ensureConnected();
        try {
            logger.finer("Unblocking user: " + user);
            this.webConnector.unblock(user);
        }
        catch (IOException e) {
            this.error(e);
        }
    }

    void sendContactRequest(User user, String greeting) {
        this.ensureConnected();
        try {
            logger.finer("Sending user: " + user + " a contact request: greeting:" + greeting);
            this.webConnector.sendContactRequest(user, greeting);
        }
        catch (IOException e) {
            this.error(e);
        }
    }

    void removeFromContacts(User user) {
        this.ensureConnected();
        try {
            logger.finer("Removing user: " + user + " from contacts");
            this.webConnector.removeFromContacts(user);
            this.contacts.remove(user);
        }
        catch (IOException e) {
            this.error(e);
        }
    }

    byte[] getAvatar(User user) {
        this.ensureConnected();
        try {
            return this.webConnector.getAvatar(user);
        }
        catch (IOException e) {
            this.error(e);
            return null;
        }
    }

    void updateUser(User user) {
        if (!this.users.containsKey(user.getUsername())) {
            try {
                logger.finest("Updating user info: " + user);
                this.webConnector.updateUser(user);
            }
            catch (IOException e) {
                this.error(e);
            }
        }
    }

    void acceptContactRequest(ContactRequest contactRequest) {
        this.ensureConnected();
        try {
            logger.finer("Accepting contact request: " + contactRequest);
            this.webConnector.acceptContactRequest(contactRequest);
            this.contactRequests.remove(contactRequest);
        }
        catch (IOException e) {
            this.error(e);
        }
    }

    void declineContactRequest(ContactRequest contactRequest) {
        this.ensureConnected();
        try {
            logger.finer("Declining contact request: " + contactRequest);
            this.webConnector.declineContactRequest(contactRequest);
            this.contactRequests.remove(contactRequest);
        }
        catch (IOException e) {
            this.error(e);
        }
    }

    void sendUserMessage(User user, String message) {
        this.ensureConnected();
        try {
            logger.finer("Sending user: " + user + " message: " + message);
            this.notifConnector.sendUserMessage(user, message);
        }
        catch (IOException e) {
            this.error(e);
        }
    }

    void sendGroupMessage(Group group, String message) {
        this.ensureConnected();
        try {
            logger.finer("Sending group: " + group + " message: " + message);
            this.notifConnector.sendGroupMessage(group, message);
        }
        catch (IOException e) {
            this.error(e);
        }
    }

    void addUserToGroup(User user, Role role, Group group) {
        this.ensureConnected();
        try {
            logger.finer("Adding user: " + user + " to group: " + group + " with role: " + (Object)((Object)role));
            this.notifConnector.addUserToGroup(user, role, group);
        }
        catch (IOException e) {
            this.error(e);
        }
    }

    void removeUserFromGroup(User user, Group group) {
        this.ensureConnected();
        try {
            logger.finer("Removing user: " + user + " from group: " + group);
            this.notifConnector.removeUserFromGroup(user, group);
        }
        catch (IOException e) {
            this.error(e);
        }
    }

    void changeUserRole(User user, Role role, Group group) {
        this.ensureConnected();
        try {
            logger.finer("Changing user: " + user + " from group: " + group + " role to: " + (Object)((Object)role));
            this.notifConnector.changeUserRole(user, role, group);
        }
        catch (IOException e) {
            this.error(e);
        }
    }

    void changeGroupTopic(Group group, String topic) {
        this.ensureConnected();
        try {
            logger.finer("Setting group: " + group + " topic to: " + topic);
            this.notifConnector.changeGroupTopic(group, topic);
        }
        catch (IOException e) {
            this.error(e);
        }
    }

    void userMessageReceived(User sender, String message) {
        this.updateUser(sender);
        logger.finer("Received message: " + message + " from user: " + sender);
        for (UserMessageListener listener : this.userMessageListeners) {
            listener.messageReceived(sender, message);
        }
    }

    void groupMessageReceived(Group group, User sender, String message) {
        logger.finer("Received group message: " + message + " from user: " + sender + " in group: " + group);
        for (GroupMessageListener listener : this.groupMessageListeners) {
            listener.messageReceived(group, sender, message);
        }
    }

    void userPresenceChanged(User user, Presence oldPresence, Presence presence) {
        logger.finer("User: " + user + " changed presence from: " + (Object)((Object)oldPresence) + " to: " + (Object)((Object)presence));
        for (UserPresenceListener listener : this.userPresenceListeners) {
            listener.presenceChanged(user, oldPresence, presence);
        }
    }

    void usersAddedToGroup(List<User> users, Group group) {
        logger.finer("Users: " + users.stream().map(User::getUsername).collect(Collectors.joining(", ")) + " added to group: " + group);
        for (GroupPropertiesListener listener : this.groupPropertiesListeners) {
            listener.usersAdded(group, users);
        }
    }

    void usersRemovedFromGroup(List<User> users, Group group) {
        logger.finer("Users: " + users.stream().map(User::getUsername).collect(Collectors.joining(", ")) + " removed from group: " + group);
        for (GroupPropertiesListener listener : this.groupPropertiesListeners) {
            listener.usersRemoved(group, users);
        }
    }

    void usersRolesChanged(Group group, List<Pair<User, Role>> newRoles) {
        logger.finer("User roles changed: " + newRoles.stream().map(p -> ((User)p.getFirst()).getUsername() + ":" + p.getSecond()).collect(Collectors.joining(", ")));
        for (GroupPropertiesListener listener : this.groupPropertiesListeners) {
            listener.usersRolesChanged(group, newRoles);
        }
    }

    void groupTopicChanged(Group group, String topic) {
        logger.finer("Group: " + group + " topic changed to: " + topic);
        for (GroupPropertiesListener listener : this.groupPropertiesListeners) {
            listener.topicChanged(group, topic);
        }
    }

    public void addUserMessageListener(UserMessageListener userMessageListener) {
        this.userMessageListeners.add(userMessageListener);
    }

    public void removeUserMessageListener(UserMessageListener userMessageListener) {
        this.userMessageListeners.remove(userMessageListener);
    }

    public void addGroupMessageListener(GroupMessageListener groupMessageListener) {
        this.groupMessageListeners.add(groupMessageListener);
    }

    public void removeGroupMessageListener(GroupMessageListener groupMessageListener) {
        this.groupMessageListeners.remove(groupMessageListener);
    }

    public void addUserPresenceListener(UserPresenceListener userPresenceListener) {
        this.userPresenceListeners.add(userPresenceListener);
    }

    public void removeUserPresenceListener(UserPresenceListener userPresenceListener) {
        this.userPresenceListeners.remove(userPresenceListener);
    }

    public void addGroupPropertiesListener(GroupPropertiesListener groupPropertiesListener) {
        this.groupPropertiesListeners.add(groupPropertiesListener);
    }

    public void removeGroupPropertiesListener(GroupPropertiesListener groupPropertiesListener) {
        this.groupPropertiesListeners.remove(groupPropertiesListener);
    }

    public void setErrorListener(ErrorListener errorListener) {
        this.errorListener = errorListener;
    }

    static {
        try {
            Skype.setDebug(null);
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }
}

