/*
 * Decompiled with CFR 0.152.
 */
package com.zaxxer.hikari.pool;

import com.codahale.metrics.MetricRegistry;
import com.codahale.metrics.health.HealthCheckRegistry;
import com.zaxxer.hikari.HikariConfig;
import com.zaxxer.hikari.HikariPoolMXBean;
import com.zaxxer.hikari.metrics.CodahaleHealthChecker;
import com.zaxxer.hikari.metrics.CodahaleMetricsTrackerFactory;
import com.zaxxer.hikari.metrics.MetricsTracker;
import com.zaxxer.hikari.metrics.MetricsTrackerFactory;
import com.zaxxer.hikari.metrics.PoolStats;
import com.zaxxer.hikari.pool.LeakTask;
import com.zaxxer.hikari.pool.PoolBagEntry;
import com.zaxxer.hikari.pool.PoolElf;
import com.zaxxer.hikari.pool.PoolInitializationException;
import com.zaxxer.hikari.pool.SuspendResumeLock;
import com.zaxxer.hikari.proxy.ConnectionProxy;
import com.zaxxer.hikari.proxy.IHikariConnectionProxy;
import com.zaxxer.hikari.proxy.ProxyFactory;
import com.zaxxer.hikari.util.ClockSource;
import com.zaxxer.hikari.util.ConcurrentBag;
import com.zaxxer.hikari.util.DefaultThreadFactory;
import com.zaxxer.hikari.util.PropertyElf;
import com.zaxxer.hikari.util.UtilityElf;
import java.sql.Connection;
import java.sql.SQLException;
import java.sql.SQLTransientConnectionException;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.FutureTask;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import javax.sql.DataSource;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class HikariPool
implements HikariPoolMXBean,
ConcurrentBag.IBagStateListener {
    private static final Logger LOGGER = LoggerFactory.getLogger(HikariPool.class);
    private static final ClockSource clockSource = ClockSource.INSTANCE;
    private final long ALIVE_BYPASS_WINDOW_MS = Long.getLong("com.zaxxer.hikari.aliveBypassWindow", TimeUnit.SECONDS.toMillis(1L));
    private final long HOUSEKEEPING_PERIOD_MS = Long.getLong("com.zaxxer.hikari.housekeeping.periodMs", TimeUnit.SECONDS.toMillis(30L));
    private static final int POOL_NORMAL = 0;
    private static final int POOL_SUSPENDED = 1;
    private static final int POOL_SHUTDOWN = 2;
    final PoolElf poolElf;
    final HikariConfig config;
    final ConcurrentBag<PoolBagEntry> connectionBag;
    final ScheduledThreadPoolExecutor houseKeepingExecutorService;
    private final AtomicInteger totalConnections;
    private final ThreadPoolExecutor addConnectionExecutor;
    private final ThreadPoolExecutor closeConnectionExecutor;
    private volatile int poolState;
    private long connectionTimeout;
    private final String poolName;
    private final LeakTask leakTask;
    private final DataSource dataSource;
    private final SuspendResumeLock suspendResumeLock;
    private final AtomicReference<Throwable> lastConnectionFailure;
    private volatile MetricsTracker metricsTracker;
    private boolean isRecordMetrics;

    public HikariPool(HikariConfig config) {
        this.config = config;
        this.poolElf = new PoolElf(config);
        this.dataSource = this.poolElf.initializeDataSource();
        this.poolName = config.getPoolName();
        this.connectionBag = new ConcurrentBag(this);
        this.totalConnections = new AtomicInteger();
        this.connectionTimeout = config.getConnectionTimeout();
        this.lastConnectionFailure = new AtomicReference();
        this.suspendResumeLock = config.isAllowPoolSuspension() ? new SuspendResumeLock(true) : SuspendResumeLock.FAUX_LOCK;
        this.addConnectionExecutor = UtilityElf.createThreadPoolExecutor(config.getMaximumPoolSize(), "Hikari connection filler (pool " + this.poolName + ")", config.getThreadFactory(), new ThreadPoolExecutor.DiscardPolicy());
        this.closeConnectionExecutor = UtilityElf.createThreadPoolExecutor(4, "Hikari connection closer (pool " + this.poolName + ")", config.getThreadFactory(), new ThreadPoolExecutor.CallerRunsPolicy());
        if (config.getScheduledExecutorService() == null) {
            ThreadFactory threadFactory = config.getThreadFactory() != null ? config.getThreadFactory() : new DefaultThreadFactory("Hikari housekeeper (pool " + this.poolName + ")", true);
            this.houseKeepingExecutorService = new ScheduledThreadPoolExecutor(1, threadFactory, new ThreadPoolExecutor.DiscardPolicy());
            this.houseKeepingExecutorService.setExecuteExistingDelayedTasksAfterShutdownPolicy(false);
            this.houseKeepingExecutorService.setRemoveOnCancelPolicy(true);
        } else {
            this.houseKeepingExecutorService = config.getScheduledExecutorService();
        }
        this.houseKeepingExecutorService.scheduleAtFixedRate(new HouseKeeper(), this.HOUSEKEEPING_PERIOD_MS, this.HOUSEKEEPING_PERIOD_MS, TimeUnit.MILLISECONDS);
        this.leakTask = new LeakTask(config.getLeakDetectionThreshold(), this.houseKeepingExecutorService);
        if (config.getMetricsTrackerFactory() != null) {
            this.setMetricsTrackerFactory(config.getMetricsTrackerFactory());
        } else {
            this.setMetricRegistry(config.getMetricRegistry());
        }
        this.setHealthCheckRegistry(config.getHealthCheckRegistry());
        this.poolElf.registerMBeans(this);
        PropertyElf.flushCaches();
        this.initializeConnections();
    }

    public final Connection getConnection() throws SQLException {
        return this.getConnection(this.connectionTimeout);
    }

    public final Connection getConnection(long hardTimeout) throws SQLException {
        this.suspendResumeLock.acquire();
        long startTime = clockSource.currentTime();
        try {
            MetricsTracker.MetricsContext metricsContext;
            long timeout = hardTimeout;
            MetricsTracker.MetricsContext metricsContext2 = metricsContext = this.isRecordMetrics ? this.metricsTracker.recordConnectionRequest() : MetricsTracker.NO_CONTEXT;
            do {
                PoolBagEntry bagEntry;
                if ((bagEntry = this.connectionBag.borrow(timeout, TimeUnit.MILLISECONDS)) == null) {
                    break;
                }
                long now = clockSource.currentTime();
                if (!bagEntry.evict && (clockSource.elapsedMillis(bagEntry.lastAccess, now) <= this.ALIVE_BYPASS_WINDOW_MS || this.poolElf.isConnectionAlive(bagEntry.connection, this.lastConnectionFailure))) {
                    metricsContext.setConnectionLastOpen(bagEntry, now);
                    metricsContext.stop();
                    IHikariConnectionProxy iHikariConnectionProxy = ProxyFactory.getProxyConnection(bagEntry, this.leakTask.start(bagEntry), now);
                    return iHikariConnectionProxy;
                }
                this.closeConnection(bagEntry, "(connection evicted or dead)");
                timeout = hardTimeout - clockSource.elapsedMillis(startTime, now);
            } while (timeout > 0L);
        }
        catch (InterruptedException e) {
            throw new SQLException(this.poolName + " - Interrupted during connection acquisition", e);
        }
        finally {
            this.suspendResumeLock.release();
        }
        this.logPoolState("Timeout failure\t");
        String sqlState = null;
        Throwable originalException = this.lastConnectionFailure.getAndSet(null);
        if (originalException instanceof SQLException) {
            sqlState = ((SQLException)originalException).getSQLState();
        }
        throw new SQLTransientConnectionException(this.poolName + " - Connection is not available, request timed out after " + clockSource.elapsedMillis(startTime) + "ms.", sqlState, originalException);
    }

    public final void releaseConnection(PoolBagEntry bagEntry) {
        this.metricsTracker.recordConnectionUsage(bagEntry);
        if (bagEntry.evict) {
            this.closeConnection(bagEntry, "(connection broken or evicted)");
        } else {
            this.connectionBag.requite(bagEntry);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public final synchronized void shutdown() throws InterruptedException {
        try {
            this.poolState = 2;
            LOGGER.info("{} - is closing down.", (Object)this.poolName);
            this.logPoolState("Before closing\t");
            this.connectionBag.close();
            this.softEvictConnections();
            this.addConnectionExecutor.shutdown();
            this.addConnectionExecutor.awaitTermination(5L, TimeUnit.SECONDS);
            if (this.config.getScheduledExecutorService() == null) {
                this.houseKeepingExecutorService.shutdown();
                this.houseKeepingExecutorService.awaitTermination(5L, TimeUnit.SECONDS);
            }
            ThreadPoolExecutor assassinExecutor = UtilityElf.createThreadPoolExecutor(this.config.getMaximumPoolSize(), "Hikari connection assassin", this.config.getThreadFactory(), new ThreadPoolExecutor.CallerRunsPolicy());
            try {
                long start = clockSource.currentTime();
                do {
                    this.softEvictConnections();
                    this.abortActiveConnections(assassinExecutor);
                } while (this.getTotalConnections() > 0 && clockSource.elapsedMillis(start) < TimeUnit.SECONDS.toMillis(5L));
            }
            finally {
                assassinExecutor.shutdown();
                assassinExecutor.awaitTermination(5L, TimeUnit.SECONDS);
            }
            this.poolElf.shutdownTimeoutExecutor();
            this.closeConnectionExecutor.shutdown();
            this.closeConnectionExecutor.awaitTermination(5L, TimeUnit.SECONDS);
        }
        catch (Throwable throwable) {
            this.logPoolState("After closing\t");
            this.poolElf.unregisterMBeans();
            this.metricsTracker.close();
            throw throwable;
        }
        this.logPoolState("After closing\t");
        this.poolElf.unregisterMBeans();
        this.metricsTracker.close();
    }

    public final void evictConnection(IHikariConnectionProxy proxyConnection) {
        this.closeConnection(proxyConnection.getPoolBagEntry(), "(connection evicted by user)");
    }

    public final DataSource getDataSource() {
        return this.dataSource;
    }

    public void setMetricRegistry(Object metricRegistry) {
        boolean bl = this.isRecordMetrics = metricRegistry != null;
        if (this.isRecordMetrics) {
            this.setMetricsTrackerFactory(new CodahaleMetricsTrackerFactory((MetricRegistry)metricRegistry));
        } else {
            this.setMetricsTrackerFactory(null);
        }
    }

    public void setMetricsTrackerFactory(MetricsTrackerFactory metricsTrackerFactory) {
        this.isRecordMetrics = metricsTrackerFactory != null;
        this.metricsTracker = this.isRecordMetrics ? metricsTrackerFactory.create(this.config.getPoolName(), this.getPoolStats()) : new MetricsTracker();
    }

    public void setHealthCheckRegistry(Object healthCheckRegistry) {
        if (healthCheckRegistry != null) {
            CodahaleHealthChecker.registerHealthChecks(this, this.config, (HealthCheckRegistry)healthCheckRegistry);
        }
    }

    public final void logPoolState(String ... prefix) {
        if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("{}pool {} stats (total={}, active={}, idle={}, waiting={})", new Object[]{prefix.length > 0 ? prefix[0] : "", this.poolName, this.getTotalConnections(), this.getActiveConnections(), this.getIdleConnections(), this.getThreadsAwaitingConnection()});
        }
    }

    public String toString() {
        return this.poolName;
    }

    @Override
    public Future<Boolean> addBagItem() {
        FutureTask<Boolean> future = new FutureTask<Boolean>(new Runnable(){

            @Override
            public void run() {
                long sleepBackoff = 200L;
                int minimumIdle = HikariPool.this.config.getMinimumIdle();
                int maxPoolSize = HikariPool.this.config.getMaximumPoolSize();
                while (HikariPool.this.poolState == 0 && HikariPool.this.totalConnections.get() < maxPoolSize && HikariPool.this.getIdleConnections() <= minimumIdle && !HikariPool.this.addConnection()) {
                    UtilityElf.quietlySleep(sleepBackoff);
                    sleepBackoff = Math.min(HikariPool.this.connectionTimeout / 2L, (long)((double)sleepBackoff * 1.5));
                }
            }
        }, true);
        this.addConnectionExecutor.execute(future);
        return future;
    }

    @Override
    public final int getActiveConnections() {
        return this.connectionBag.getCount(1);
    }

    @Override
    public final int getIdleConnections() {
        return this.connectionBag.getCount(0);
    }

    @Override
    public final int getTotalConnections() {
        return this.connectionBag.size() - this.connectionBag.getCount(-1);
    }

    @Override
    public final int getThreadsAwaitingConnection() {
        return this.connectionBag.getPendingQueue();
    }

    @Override
    public void softEvictConnections() {
        for (PoolBagEntry bagEntry : this.connectionBag.values()) {
            bagEntry.evict = true;
            if (!this.connectionBag.reserve(bagEntry)) continue;
            this.closeConnection(bagEntry, "(connection evicted by user)");
        }
    }

    @Override
    public final synchronized void suspendPool() {
        if (this.suspendResumeLock == SuspendResumeLock.FAUX_LOCK) {
            throw new IllegalStateException(this.poolName + " - is not suspendable");
        }
        if (this.poolState != 1) {
            this.suspendResumeLock.suspend();
            this.poolState = 1;
        }
    }

    @Override
    public final synchronized void resumePool() {
        if (this.poolState == 1) {
            this.poolState = 0;
            this.fillPool();
            this.suspendResumeLock.resume();
        }
    }

    void closeConnection(PoolBagEntry bagEntry, final String closureReason) {
        final Connection connection = bagEntry.connection;
        bagEntry.close();
        if (this.connectionBag.remove(bagEntry)) {
            int tc = this.totalConnections.decrementAndGet();
            if (tc < 0) {
                LOGGER.warn("{} - Internal accounting inconsistency, totalConnections={}", new Object[]{this.poolName, tc, new Exception()});
            }
            this.closeConnectionExecutor.execute(new Runnable(){

                @Override
                public void run() {
                    HikariPool.this.poolElf.quietlyCloseConnection(connection, closureReason);
                }
            });
        }
    }

    private boolean addConnection() {
        if (this.totalConnections.incrementAndGet() > this.config.getMaximumPoolSize()) {
            this.totalConnections.decrementAndGet();
            this.lastConnectionFailure.set(new SQLException(this.poolName + " - is at maximum capacity"));
            return true;
        }
        Connection connection = null;
        try {
            String username = this.config.getUsername();
            String password = this.config.getPassword();
            connection = username == null ? this.dataSource.getConnection() : this.dataSource.getConnection(username, password);
            this.poolElf.setupConnection(connection, this.connectionTimeout);
            this.connectionBag.add(new PoolBagEntry(connection, this));
            this.lastConnectionFailure.set(null);
            LOGGER.debug("{} - Added connection {}", (Object)this.poolName, (Object)connection);
            return true;
        }
        catch (Exception e) {
            this.totalConnections.decrementAndGet();
            this.lastConnectionFailure.set(e);
            if (this.poolState == 0) {
                LOGGER.debug("{} - Cannot acquire connection from data source", (Object)this.poolName, (Object)e);
            }
            this.poolElf.quietlyCloseConnection(connection, "(exception during connection creation)");
            return false;
        }
    }

    private void fillPool() {
        int connectionsToAdd = Math.min(this.config.getMaximumPoolSize() - this.totalConnections.get(), this.config.getMinimumIdle() - this.getIdleConnections());
        for (int i = 0; i < connectionsToAdd; ++i) {
            this.addBagItem();
        }
        if (connectionsToAdd > 0 && LOGGER.isDebugEnabled()) {
            this.addConnectionExecutor.execute(new Runnable(){

                @Override
                public void run() {
                    HikariPool.this.logPoolState("After fill\t");
                }
            });
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void abortActiveConnections(ExecutorService assassinExecutor) {
        for (PoolBagEntry bagEntry : this.connectionBag.values(1)) {
            try {
                bagEntry.evict = true;
                bagEntry.connection.abort(assassinExecutor);
            }
            catch (Throwable e) {
                this.poolElf.quietlyCloseConnection(bagEntry.connection, "(connection aborted during shutdown)");
            }
            finally {
                bagEntry.close();
                if (!this.connectionBag.remove(bagEntry)) continue;
                this.totalConnections.decrementAndGet();
            }
        }
    }

    private void initializeConnections() {
        if (this.config.isInitializationFailFast()) {
            try {
                if (!this.addConnection()) {
                    throw (Throwable)this.lastConnectionFailure.getAndSet(null);
                }
                ConnectionProxy connection = (ConnectionProxy)this.getConnection();
                connection.getPoolBagEntry().evict = this.config.getMinimumIdle() == 0;
                connection.close();
            }
            catch (Throwable e) {
                try {
                    this.shutdown();
                }
                catch (Throwable ex) {
                    e.addSuppressed(ex);
                }
                throw new PoolInitializationException(e);
            }
        }
        this.fillPool();
    }

    private PoolStats getPoolStats() {
        return new PoolStats(TimeUnit.SECONDS.toMillis(1L)){

            @Override
            protected void update() {
                this.pendingThreads = HikariPool.this.getThreadsAwaitingConnection();
                this.idleConnections = HikariPool.this.getIdleConnections();
                this.totalConnections = HikariPool.this.getTotalConnections();
                this.activeConnections = HikariPool.this.getActiveConnections();
            }
        };
    }

    private class HouseKeeper
    implements Runnable {
        private volatile long previous = HikariPool.access$500().currentTime();

        private HouseKeeper() {
        }

        @Override
        public void run() {
            HikariPool.this.connectionTimeout = HikariPool.this.config.getConnectionTimeout();
            HikariPool.this.poolElf.setValidationTimeout(HikariPool.this.config.getValidationTimeout());
            HikariPool.this.leakTask.updateLeakDetectionThreshold(HikariPool.this.config.getLeakDetectionThreshold());
            long now = clockSource.currentTime();
            long idleTimeout = HikariPool.this.config.getIdleTimeout();
            if (now < this.previous || now > clockSource.plusMillis(this.previous, 2L * HikariPool.this.HOUSEKEEPING_PERIOD_MS)) {
                LOGGER.warn("{} - Unusual system clock change detected, soft-evicting connections from pool.", (Object)HikariPool.this.poolName);
                this.previous = now;
                HikariPool.this.softEvictConnections();
                HikariPool.this.fillPool();
                return;
            }
            this.previous = now;
            if (idleTimeout > 0L) {
                HikariPool.this.logPoolState("Before cleanup\t");
                List<PoolBagEntry> bag = HikariPool.this.connectionBag.values(0);
                int removable = bag.size() - HikariPool.this.config.getMinimumIdle();
                for (PoolBagEntry bagEntry : bag) {
                    if (removable <= 0) break;
                    if (clockSource.elapsedMillis(bagEntry.lastAccess, now) <= idleTimeout || !HikariPool.this.connectionBag.reserve(bagEntry)) continue;
                    HikariPool.this.closeConnection(bagEntry, "(connection passed idleTimeout)");
                    --removable;
                }
                HikariPool.this.logPoolState("After cleanup\t");
            }
            HikariPool.this.fillPool();
        }
    }
}

