/*
 * Decompiled with CFR 0.152.
 */
package com.oracle.graal.pointsto.reports;

import com.oracle.graal.pointsto.BigBang;
import com.oracle.graal.pointsto.flow.InvokeTypeFlow;
import com.oracle.graal.pointsto.meta.AnalysisMethod;
import com.oracle.graal.pointsto.reports.ReportUtils;
import java.io.PrintWriter;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Deque;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;
import jdk.vm.ci.meta.ResolvedJavaMethod;
import org.graalvm.compiler.nodes.CallTargetNode;
import org.graalvm.compiler.nodes.FrameState;
import org.graalvm.compiler.nodes.java.MethodCallTargetNode;

public final class CallTreePrinter {
    private final BigBang bigbang;
    private final Map<AnalysisMethod, MethodNode> methodToNode;
    private static final String METHOD_FORMAT = "%H.%n(%P):%R";

    public static void print(BigBang bigbang, String path, String reportName) {
        CallTreePrinter printer = new CallTreePrinter(bigbang);
        printer.buildCallTree();
        ReportUtils.report("call tree", path + "/reports", "call_tree_" + reportName, "txt", writer -> printer.printMethods((PrintWriter)writer));
        ReportUtils.report("list of used classes", path + "/reports", "used_classes_" + reportName, "txt", writer -> printer.printClasses((PrintWriter)writer, false));
        ReportUtils.report("list of used packages", path + "/reports", "used_packages_" + reportName, "txt", writer -> printer.printClasses((PrintWriter)writer, true));
    }

    public CallTreePrinter(BigBang bigbang) {
        this.bigbang = bigbang;
        this.methodToNode = new LinkedHashMap<AnalysisMethod, MethodNode>();
    }

    public void buildCallTree() {
        this.bigbang.getUniverse().getMethods().stream().filter(m -> m.isRootMethod() && !this.methodToNode.containsKey(m)).sorted(ReportUtils.methodComparator).forEach(method -> this.methodToNode.put((AnalysisMethod)method, new MethodNode((AnalysisMethod)method, true)));
        ArrayDeque<MethodNode> workList = new ArrayDeque<MethodNode>();
        workList.addAll(this.methodToNode.values());
        while (!workList.isEmpty()) {
            MethodNode node = (MethodNode)workList.removeFirst();
            node.method.getTypeFlow().getInvokes().stream().sorted(ReportUtils.invokeComparator).forEach(invoke -> this.processInvoke((InvokeTypeFlow)invoke, node, (Deque<MethodNode>)workList));
        }
    }

    private void processInvoke(InvokeTypeFlow invokeFlow, MethodNode callerNode, Deque<MethodNode> workList) {
        InvokeNode invokeNode = new InvokeNode(invokeFlow.getTargetMethod(), invokeFlow.invoke().getInvokeKind(), CallTreePrinter.sourceReference(invokeFlow));
        callerNode.addInvoke(invokeNode);
        invokeFlow.getCallees().stream().sorted(ReportUtils.methodComparator).forEach(callee -> {
            if (this.methodToNode.containsKey(callee)) {
                MethodNodeReference calleeNode = new MethodNodeReference(this.methodToNode.get(callee));
                invokeNode.addCallee(calleeNode);
            } else {
                MethodNode calleeNode = new MethodNode((AnalysisMethod)callee);
                invokeNode.addCallee(calleeNode);
                this.methodToNode.put((AnalysisMethod)callee, calleeNode);
                workList.add(calleeNode);
            }
        });
    }

    private static SourceReference[] sourceReference(InvokeTypeFlow invoke) {
        ArrayList<SourceReference> sourceReference = new ArrayList<SourceReference>();
        for (FrameState state = ((MethodCallTargetNode)invoke.getSource()).invoke().stateAfter(); state != null; state = state.outerFrameState()) {
            if (state.getCode() != null) {
                sourceReference.add(new SourceReference(state.bci, state.getCode().asStackTraceElement(state.bci)));
                continue;
            }
            sourceReference.add(SourceReference.UNKNOWN_SOURCE_REFERENCE);
        }
        return sourceReference.toArray(new SourceReference[sourceReference.size()]);
    }

    private void printMethods(PrintWriter out) {
        out.println("VM Entry Points");
        Iterator iterator = this.methodToNode.values().stream().filter(n -> ((MethodNode)n).isEntryPoint).iterator();
        while (iterator.hasNext()) {
            MethodNode node = (MethodNode)iterator.next();
            boolean lastEntryPoint = !iterator.hasNext();
            out.format("%s%s %s %n", lastEntryPoint ? "\u2514\u2500\u2500 " : "\u251c\u2500\u2500 ", "entry", node.format());
            CallTreePrinter.printCallTreeNode(out, lastEntryPoint ? "    " : "\u2502   ", node);
        }
        out.println();
    }

    private static void printCallTreeNode(PrintWriter out, String prefix, MethodNode node) {
        for (int invokeIdx = 0; invokeIdx < node.invokes.size(); ++invokeIdx) {
            boolean lastInvoke;
            InvokeNode invoke = (InvokeNode)node.invokes.get(invokeIdx);
            boolean bl = lastInvoke = invokeIdx == node.invokes.size() - 1;
            if (invoke.kind == CallTargetNode.InvokeKind.Static || invoke.kind == CallTargetNode.InvokeKind.Special) {
                if (invoke.callees.size() <= 0) continue;
                Node calleeNode = (Node)invoke.callees.get(0);
                out.format("%s%s%s %s @bci=%s %n", prefix, lastInvoke ? "\u2514\u2500\u2500 " : "\u251c\u2500\u2500 ", "directly calls", calleeNode.format(), invoke.formatLocation());
                if (!(calleeNode instanceof MethodNode)) continue;
                CallTreePrinter.printCallTreeNode(out, prefix + (lastInvoke ? "    " : "\u2502   "), (MethodNode)calleeNode);
                continue;
            }
            if (invoke.kind != CallTargetNode.InvokeKind.Virtual && invoke.kind != CallTargetNode.InvokeKind.Interface) continue;
            out.format("%s%s%s %s @bci=%s%n", prefix, lastInvoke ? "\u2514\u2500\u2500 " : "\u251c\u2500\u2500 ", invoke.kind == CallTargetNode.InvokeKind.Virtual ? "virtually calls" : "interfacially calls", invoke.formatTarget(), invoke.formatLocation());
            for (int calleeIdx = 0; calleeIdx < invoke.callees.size(); ++calleeIdx) {
                boolean lastCallee = calleeIdx == invoke.callees.size() - 1;
                Node calleeNode = (Node)invoke.callees.get(calleeIdx);
                out.format("%s%s%s %s %n", prefix + (lastInvoke ? "    " : "\u2502   "), lastCallee ? "\u2514\u2500\u2500 " : "\u251c\u2500\u2500 ", invoke.kind == CallTargetNode.InvokeKind.Virtual ? "is overridden by" : "is implemented by", calleeNode.format());
                if (!(calleeNode instanceof MethodNode)) continue;
                CallTreePrinter.printCallTreeNode(out, prefix + (lastInvoke ? "    " : "\u2502   ") + (lastCallee ? "    " : "\u2502   "), (MethodNode)calleeNode);
            }
        }
    }

    private void printClasses(PrintWriter out, boolean packageNameOnly) {
        ArrayList<String> classList = new ArrayList<String>(this.classesSet(packageNameOnly));
        classList.sort(null);
        for (String name : classList) {
            out.println(name);
        }
    }

    public Set<String> classesSet(boolean packageNameOnly) {
        HashSet<String> classSet = new HashSet<String>();
        for (ResolvedJavaMethod resolvedJavaMethod : this.methodToNode.keySet()) {
            String name = resolvedJavaMethod.getDeclaringClass().toJavaName(true);
            if (packageNameOnly && (name = CallTreePrinter.packagePrefix(name)).contains("$$Lambda$")) {
                name = CallTreePrinter.packagePrefix(name);
            }
            classSet.add(name);
        }
        return classSet;
    }

    private static String packagePrefix(String name) {
        int lastDot = name.lastIndexOf(46);
        if (lastDot == -1) {
            return name;
        }
        return name.substring(0, lastDot);
    }

    static class SourceReference {
        static final SourceReference UNKNOWN_SOURCE_REFERENCE = new SourceReference(-1, null);
        final int bci;
        final StackTraceElement trace;

        SourceReference(int bci, StackTraceElement trace) {
            this.bci = bci;
            this.trace = trace;
        }
    }

    static class InvokeNode {
        private final AnalysisMethod targetMethod;
        private final List<Node> callees;
        private final CallTargetNode.InvokeKind kind;
        private final SourceReference[] sourceReferences;

        InvokeNode(AnalysisMethod targetMethod, CallTargetNode.InvokeKind kind, SourceReference[] sourceReferences) {
            this.targetMethod = targetMethod;
            this.kind = kind;
            this.sourceReferences = sourceReferences;
            this.callees = new ArrayList<Node>();
        }

        void addCallee(Node callee) {
            this.callees.add(callee);
        }

        String formatLocation() {
            return Arrays.stream(this.sourceReferences).map(s -> String.valueOf(s.bci)).collect(Collectors.joining("->"));
        }

        String formatTarget() {
            return this.targetMethod.format(CallTreePrinter.METHOD_FORMAT);
        }
    }

    static class MethodNode
    implements Node {
        static int methodId = 0;
        private final int id = methodId++;
        private final AnalysisMethod method;
        private final List<InvokeNode> invokes;
        private final boolean isEntryPoint;

        MethodNode(AnalysisMethod method) {
            this(method, false);
        }

        MethodNode(AnalysisMethod method, boolean isEntryPoint) {
            this.method = method;
            this.invokes = new ArrayList<InvokeNode>();
            this.isEntryPoint = isEntryPoint;
        }

        void addInvoke(InvokeNode invoke) {
            this.invokes.add(invoke);
        }

        @Override
        public String format() {
            return this.method.format(CallTreePrinter.METHOD_FORMAT) + " id=" + this.id;
        }
    }

    static class MethodNodeReference
    implements Node {
        private final MethodNode methodNode;

        MethodNodeReference(MethodNode methodNode) {
            this.methodNode = methodNode;
        }

        @Override
        public String format() {
            return this.methodNode.method.format(CallTreePrinter.METHOD_FORMAT) + " id-ref=" + this.methodNode.id;
        }
    }

    static interface Node {
        public String format();
    }
}

