package com.linkare.jboolexpr.test;

import java.applet.Applet;
import java.awt.BorderLayout;
import java.awt.Choice;
import java.awt.Frame;
import java.awt.Label;
import java.awt.Panel;
import java.awt.TextArea;
import java.awt.TextField;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.awt.event.TextEvent;
import java.awt.event.TextListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;

import com.linkare.jboolexpr.BooleanExpression;

/**
 * Class demo to evaluate boolean expressions.
 */
public final class BooleanExpressionTest extends Applet implements TextListener, ItemListener {

    private static final int MY_HEIGHT = 150;

    private static final int MY_WIDTH = 600;

    private static final int FIELD_BOOLEAN_EXPRESSION_COLUMNS = 34;

    /**
     * If read boolean expression from left to right.
     */
    private static final String LR = "Read left to right";

    /**
     * If read boolean expression from right to left.
     */
    private static final String RL = "Read right to left";

    /**
     * The serialVersionUID.
     */
    private static final long serialVersionUID = 1L;

    /**
     * The text field to enter the boolean expression.
     */
    private TextField fieldBooleanExpression = new TextField("!true&&false||true", FIELD_BOOLEAN_EXPRESSION_COLUMNS);

    /**
     * The boolean expression type.
     */
    private Choice booleanExpresisonType = new Choice();

    /**
     * The text area to output the result.
     */
    private TextArea outResult = new TextArea();

    /**
     * Add controls and init applet.
     */
    public void init() {
	BorderLayout borderLayout = new BorderLayout();
	setLayout(borderLayout);
	Panel input = new Panel();
	input.add(new Label("Boolean expression to evaluate:", Label.RIGHT));
	input.add(this.fieldBooleanExpression);
	input.add(this.booleanExpresisonType);
	add(input, BorderLayout.NORTH);
	add(this.outResult, BorderLayout.CENTER);
	setSize(MY_WIDTH, MY_HEIGHT);

	// Boolean expression types
	this.booleanExpresisonType.add(LR);
	this.booleanExpresisonType.add(RL);

	this.booleanExpresisonType.addItemListener(this);

	// Listen to text changes
	this.fieldBooleanExpression.addTextListener(this);

	// Initial UI update
	textValueChanged(null);
    }

    /**
     * Called when boolean expression type choice change.
     * 
     * @param itemEvent
     *            The {@link ItemEvent}.
     */
    public void itemStateChanged(final ItemEvent itemEvent) {
	updateTextArea();
    }

    /**
     * Called when text values change.
     * 
     * @param textEvent
     *            The {@link TextEvent}.
     */
    public void textValueChanged(final TextEvent textEvent) {
	updateTextArea();
    }

    /**
     * 
     */
    private void updateTextArea() {
	String booleanExpression = this.fieldBooleanExpression.getText();
	String selectedItem = this.booleanExpresisonType.getSelectedItem();
	BooleanExpression be = null;
	try {
	    if (LR.equals(selectedItem)) {
		be = BooleanExpression.readLeftToRight(booleanExpression);
	    } else if (RL.equals(selectedItem)) {
		be = BooleanExpression.readRightToLeft(booleanExpression);
	    } else {
		throw new IllegalArgumentException("selectedItem is diferent of '" + LR + "' and '" + RL + "' (selectedItem='" + selectedItem + "')");
	    }
	    this.outResult.setText(be.toString() + " == " + be.booleanValue());
	} catch (Exception e) {
	    StringBuilder exceptionText = new StringBuilder();
	    String exceptionName = e.getClass().getName();
	    exceptionText.append(exceptionName);
	    String exceptionMessage = e.getMessage();
	    if (exceptionMessage != null) {
		exceptionText.append("\n");
		exceptionText.append(exceptionMessage);
	    }
	    StackTraceElement[] stackTraceElements = e.getStackTrace();
	    for (int i = 0; i < stackTraceElements.length; i++) {
		String stackTraceElement = stackTraceElements[i].toString();
		if (stackTraceElement.startsWith(BooleanExpression.class.getPackage().getName())) {
		    exceptionText.append("\n\t");
		    exceptionText.append(stackTraceElements[i]);
		}
	    }
	    this.outResult.setText(exceptionText.toString());
	}
    }

    /**
     * Main application entrypoint.
     * 
     * @param arg
     *            Command line arguments.
     */
    public static void main(String[] arg) {
	Frame frame = new Frame("Boolean Expression Demo");
	frame.addWindowListener(new WindowAdapter() {
	    public void windowClosing(WindowEvent windowEvent) {
		System.exit(0);
	    }
	});
	BooleanExpressionTest test = new BooleanExpressionTest();
	frame.add(test);
	test.init();
	frame.pack();
	frame.setVisible(true);
    }

}