/*
 * Copyright 2009 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


package com.googlecode.guicejunit4.core;

import java.util.ArrayList;
import java.util.Collection;

import org.junit.internal.runners.model.EachTestNotifier;
import org.junit.runner.notification.RunNotifier;
import org.junit.runners.BlockJUnit4ClassRunner;
import org.junit.runners.model.FrameworkMethod;
import org.junit.runners.model.InitializationError;

import com.google.inject.Injector;
import com.google.inject.internal.Preconditions;
import com.googlecode.guicejunit4.core.inner.Annotations;
import com.googlecode.guicejunit4.core.inner.Instances;

/**
 * 
 * @author java2enterprise@gmail.com (James Wang)
 * @since Nov 20, 2009
 * 
 */
public class GuiceJunit4ClassRunner extends BlockJUnit4ClassRunner {
	
	private Injector injector;
	
	private Collection<TestInterceptor> interceptors = new ArrayList<TestInterceptor>();

	private Object testInstance;
	
	public GuiceJunit4ClassRunner(Class<?> clazz) throws InitializationError {
		super(clazz);
		retrieveInjector(clazz); 
		retrieveInterceptors(clazz);
	}

	private void retrieveInjector(Class<?> clazz) {
		InjectWith injectWith = Annotations.findInherited(clazz, InjectWith.class);
		Preconditions.checkNotNull(injectWith, "Annotation " + InjectWith.class.getSimpleName() + " needed to present on " + clazz + " or it's super class or interfaces.");
		injector = Instances.create(injectWith.value()).get();
	}

	private void retrieveInterceptors(Class<?> clazz) {
		/** TODO process cumulative */
		InterceptWith testInterceptors = Annotations.findInherited(clazz, InterceptWith.class);
		if (testInterceptors == null) {
			return;
		}
		
		Class<? extends TestInterceptor>[] interceptorTypes = testInterceptors.value();
		
		for (Class<? extends TestInterceptor> type : interceptorTypes) {
			interceptors.add(Instances.create(type));
		}
	}
	
	@Override
	protected Object createTest() throws Exception {
		Object instance = super.createTest();
		this.injector.injectMembers(instance);
		//TODO is this way safe ?
		this.testInstance = instance;
		return instance;
	}

	@Override
	protected void runChild(FrameworkMethod method, RunNotifier notifier) {
		EachTestNotifier eachNotifier= makeNotifier(method, notifier);
	
		TestContext testContext = new TestContext(injector, testInstance, getTestClass().getJavaClass(), method.getMethod());
		
		for (TestInterceptor interceptor : interceptors) {
			try {
				interceptor.before(testContext);
			} catch (Exception e) {
				eachNotifier.addFailure(e);
			}
		}
		
		try {
			super.runChild(method, notifier);
		} finally {
			for (TestInterceptor interceptor : interceptors) {
				try {
					interceptor.after(testContext);
				} catch (Exception e) {
					eachNotifier.addFailure(e);
				}
			}	
		}
	}
	
	
	
	
	
	
}
