/**********************************************************************
Copyright (c) 2005 Erik Bengtson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
    ...
**********************************************************************/
package org.datanucleus.store.mapped.expression;

/**
 * Interface that allows one datastore adapt operations.
 */
public interface ExpressionMethodAdapter
{
    /**
     * Returns whether this string ends with the specified string.
     * @param leftOperand the source string
     * @param rightOperand The string to compare against.
     * @return Whether it ends with the string.
     **/
    BooleanExpression endsWithMethod(ScalarExpression leftOperand, ScalarExpression rightOperand);

    /**
     * Returns the expression for the JDOQL
     * String.substring(str,begin) method.
     * In SQL, it may compile to:
     * <p>
     * <pre>
     * SUBSTRING(str FROM begin)
     * </pre>
     * Note that the value of <var>begin</var> is base 0(Java-style), while most
     * SQL string functions use base 1.
     *
     * @param str The first argument to the substring() method.
     * @param begin The second argument to the substring() method.
     * @return  The expression.
     */
    StringExpression substringMethod(StringExpression str, NumericExpression begin);

    /**
     * Returns the expression for the JDOQL
     * String.substring(str,begin,end) method.
     * In SQL, it may compile to:
     * <p>
     * <blockquote><pre>
     * SUBSTRING(str FROM begin FOR len)
     * </pre></blockquote>
     * Note that the value of <var>begin</var> is base 0 (Java-style), while most
     * SQL string functions use base 1.
     * Note also that an end position is given, while most SQL substring
     * functions take a length.
     *
     * @param str   The first argument to the substring() method.
     * @param begin The second argument to the substring() method.
     * @param end   The third argument to the substring() method.
     * @return  The expression.
     */
    StringExpression substringMethod(StringExpression str,
                                     NumericExpression begin,
                                     NumericExpression end);

    /**
     * Method to handle the starts with operation.
     * @param source The expression with the searched string
     * @param str The expression for the search string 
     * @return The expression.
     **/
    BooleanExpression startsWithMethod(ScalarExpression source, ScalarExpression str);

    /**
     * Method to handle the indexOf operation.
     * @param source The expression with the searched string
     * @param str The expression for the search string
     * @param from Position to start searching from
     * @return The expression.
     **/
    NumericExpression indexOfMethod(ScalarExpression source, ScalarExpression str, NumericExpression from);

    /**
     * Returns the expression for the JDOQL String.trim(str) method.
     * In SQL, it may compile to:
     * <pre>TRIM(str)</pre>
     * @param str The first argument to the trim() method.
     * @param leading Whether to trim leading spaces
     * @param trailing Whether to trim trailing spaces
     * @return  The expression.
     */
    StringExpression trimMethod(StringExpression str, boolean leading, boolean trailing);

    /**
     * Matches this to the argument expression pattern. Use "." to find any
     * character and ".*" for wildcard matches. A global case-insensitive flag
     * "(?i)" can be set for the pattern. If used, the global case-insensitive
     * flag must prefix the pattern. The pattern passed to matches must be a
     * literal or parameter.
     * @param text   The argument to the length() method.
     * @param pattern The literal expression with the pattern.
     * @return the match expression.
     */
    BooleanExpression matchesMethod(StringExpression text, StringExpression pattern);

    /**
     * Method to translate all chars in this expression to the <code>fromExpr</code> which 
     * corresponds to <code>toExpr</code>.
     * @return The expression.
     **/
    StringExpression translateMethod(ScalarExpression expr, ScalarExpression toExpr, ScalarExpression fromExpr);

    /**
     * Returns the expression for the JDOQL String.toLowerCase(str) method.
     * In SQL, it may compile to:
     * <pre>LOWER(str)</pre>
     * @param str The first argument to the trim() method.
     * @return  The expression.
     */
    StringExpression lowerMethod(StringExpression str);

    /**
     * Returns the expression for the JDOQL String.toUpperCase(str) method.
     * In SQL, it may compile to:
     * <pre>UPPER(str)</pre>
     * @param str The first argument to the trim() method.
     * @return  The expression.
     */
    StringExpression upperMethod(StringExpression str);

    /**
     * Method to return an expression for the current date in the datastore.
     * @param qs QueryExpression
     * @return Current date expression for this datastore
     */
    ScalarExpression getCurrentDateMethod(QueryExpression qs);

    /**
     * Method to return an expression for the current time in the datastore.
     * @param qs QueryExpression
     * @return Current time expression for this datastore
     */
    ScalarExpression getCurrentTimeMethod(QueryExpression qs);

    /**
     * Method to return an expression for the current timestamp in the datastore.
     * @param qs QueryExpression
     * @return Current timestamp expression for this datastore
     */
    ScalarExpression getCurrentTimestampMethod(QueryExpression qs);
}