/**********************************************************************
Copyright (c) 2002 Kelly Grizzle (TJDO) and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
 

Contributors:
2002 Mike Martin (TJDO)
2003 Andy Jefferson - coding standards
    ...
**********************************************************************/
package org.datanucleus.store.mapped.expression;

import org.datanucleus.store.mapped.DatastoreAdapter;
import org.datanucleus.store.mapped.mapping.JavaTypeMapping;

/**
 * Representation of a Boolean expression in a Query.
 *
 * @version $Revision: 1.12 $
 **/
public class BooleanExpression extends ScalarExpression
{
    /**
     * Constructor
     * @param qs the QueryExpression
     */
    protected BooleanExpression(QueryExpression qs)
    {
        super(qs);
    }

    /**
     * 
     * @param qs the QueryExpression
     * @param mapping the mapping associated to this expression
     * @param te the TableExpression where this expression refers to
     */    
    public BooleanExpression(QueryExpression qs, JavaTypeMapping mapping, LogicSetExpression te)
    {
        super(qs, mapping, te);
    }

    /**
     * Perform a boolean operator <code>op</code> on a operand <code>operand</code>
     * @param op the operator
     * @param operand the operand
     */
    public BooleanExpression(MonadicOperator op, ScalarExpression operand)
    {
        super(op, operand);
    }

    /**
     * Perform a boolean operator <code>op</code> between two operands <code>operand1</code> and <code>operand2</code>
     * @param op the operator
     * @param operand1 the operand
     * @param operand2 the operand
     */
    public BooleanExpression(ScalarExpression operand1, DyadicOperator op, ScalarExpression operand2)
    {
        super(operand1, op, operand2);
    }

    public BooleanExpression and(ScalarExpression expr)
    {
        if (expr instanceof BooleanLiteral)
        {
            return expr.and(this);
        }
        else if (expr instanceof ExistsExpression && !(this instanceof ExistsExpression))
        {
            // WHERE ... && fld.contains(...)
            boolean existsIncludesAll = false;
            Object ext = expr.qs.getValueForExtension("datanucleus.rdbms.jdoql.existsIncludesConstraints");
            if (ext != null && ((String)ext).equals("true"))
            {
                existsIncludesAll = true;
            }
            if (existsIncludesAll)
            {
                // e.g 1. SELECT FROM MyClass WHERE fld1.contains(val1) && val1.name == 'some value'
                // with this will result in the val1.name constraint being applied to the EXISTS - correct
                // e.g 2. SELECT FROM MyClass WHERE fld1 == val1 && (fld2.contains(val2) && val2.name == 'some value')
                // with this will result in the "fld1 == val1" being applied to the EXISTS too - wrong
                // This is a hack for some queries
                return expr.and(this);
            }
            else
            {
                // e.g 1. SELECT FROM MyClass WHERE fld1.contains(val1) && val1.name == 'some value'
                // will result in val1.name being outside the EXISTS - wrong
                // e.g 2. SELECT FROM MyClass WHERE fld1 == val1 && (fld2.contains(val2) && val2.name == 'some value')
                // will result in the "fld1 == val1" being outside the EXISTS - correct
            }
        }

        if (expr instanceof BooleanExpression)
        {
            return new BooleanExpression(this, OP_AND, expr);
        }
        else
        {
            return super.and(expr);
        }
    }

    public BooleanExpression eor(ScalarExpression expr)
    {
        if (expr instanceof BooleanLiteral)
        {
            return expr.eor(this);
        }
        else if (expr instanceof ExistsExpression)
        {
            return expr.eor(this);
        }
        else if (expr instanceof BooleanExpression)
        {
            if (qs.getStoreManager().getDatastoreAdapter().supportsOption(DatastoreAdapter.BOOLEAN_COMPARISON))
            {
                return new BooleanExpression(this, OP_NOTEQ, expr);
            }
            else
            {
                return and(expr.not()).ior(not().and(expr));
            }
        }
        else
        {
            return super.eor(expr);
        }
    }

    public BooleanExpression ior(ScalarExpression expr)
    {
        if (expr instanceof BooleanLiteral)
        {
            return expr.ior(this);
        }
        else if (expr instanceof BooleanExpression)
        {
            return new BooleanExpression(this, OP_OR, expr);
        }
        else
        {
            return super.ior(expr);
        }
    }

    public BooleanExpression not()
    {
        return new BooleanExpression(OP_NOT, this);
    }

    public BooleanExpression eq(ScalarExpression expr)
    {
        assertValidTypeForParameterComparison(expr, BooleanExpression.class);

        if (expr instanceof BooleanLiteral || expr instanceof NullLiteral)
        {
            return expr.eq(this);
        }
        else if (expr instanceof BooleanExpression)
        {
            if (qs.getStoreManager().getDatastoreAdapter().supportsOption(DatastoreAdapter.BOOLEAN_COMPARISON))
            {
                return new BooleanExpression(this, OP_EQ, expr);
            }
            else
            {
                return and(expr).ior(not().and(expr.not()));
            }
        }
        else
        {
            return super.eq(expr);
        }
    }

    public BooleanExpression noteq(ScalarExpression expr)
    {
        assertValidTypeForParameterComparison(expr, BooleanExpression.class);

        if (expr instanceof BooleanLiteral || expr instanceof NullLiteral)
        {
            return expr.noteq(this);
        }
        else if (expr instanceof BooleanExpression)
        {
            if (qs.getStoreManager().getDatastoreAdapter().supportsOption(DatastoreAdapter.BOOLEAN_COMPARISON))
            {
                return new BooleanExpression(this, OP_NOTEQ, expr);
            }
            else
            {
                return and(expr.not()).ior(not().and(expr));
            }
        }
        else
        {
            return super.noteq(expr);
        }
    }
    
    public BooleanExpression in(ScalarExpression expr)
    {
        return new BooleanExpression(this, OP_IN, expr);
    }
    
    public ScalarExpression neg()
    {
        return new NumericExpression(OP_NEG, this);
    }
}