/**********************************************************************
Copyright (c) 2002 Mike Martin (TJDO) and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
2003 Andy Jefferson - commented
    ...
**********************************************************************/
package org.datanucleus.store.mapped.expression;

import java.math.BigInteger;
import java.util.ArrayList;
import java.util.List;

import org.datanucleus.exceptions.NucleusUserException;
import org.datanucleus.store.mapped.mapping.JavaTypeMapping;

/**
 * Representation of a Character expression in a Query
 *
 * @version $Revision: 1.16 $
 **/
public class CharacterExpression extends ScalarExpression
{
    /** JOIN expression used in JPQL queries **/
    /*private ScalarExpression joinExpr;*/
    
    /**
     * Constructor
     * @param qs the QueryExpression
     */
    protected CharacterExpression(QueryExpression qs)
    {
        super(qs);
    }

    /**
     * 
     * @param qs the QueryExpression
     * @param mapping the mapping associated to this expression
     * @param te the TableExpression where this expression refers to
     */    
    public CharacterExpression(QueryExpression qs, JavaTypeMapping mapping, LogicSetExpression te)
    {
        super(qs, mapping, te);
    }

    /**
     * Generates statement as e.g. FUNCTION_NAME(arg[,argN]). The function returns a character value
     * @param functionName
     * @param args ScalarExpression list
     */
    public CharacterExpression(String functionName, List args)
    {
        super(functionName, args);
    }

    public BooleanExpression eq(ScalarExpression expr)
    {
        if (expr instanceof NullLiteral)
        {
            return expr.eq(this);
        }
        else if (expr instanceof CharacterLiteral)
        {
            return new BooleanExpression(this, OP_EQ, expr);            
        }
        else if (expr instanceof CharacterExpression)
        {
            return new BooleanExpression(this, OP_EQ, expr);   
        }
        else if (expr instanceof StringLiteral)
        {
            Object value = ((StringLiteral)expr).getValue();
            if (value instanceof String && ((String)value).length() > 1)
            {
                // Can't compare a character with a String of more than 1 character
                throw new NucleusUserException("Can't perform equality comparison between a character and a String of more than 1 character (" + value + ") !");
            }
            return new BooleanExpression(this, OP_EQ, expr);
        }
        else if (expr instanceof StringExpression)
        {
            return new BooleanExpression(this, OP_EQ, expr); 
        }
        else if (expr instanceof NumericExpression)
        {
            return qs.getStoreManager().getDatastoreAdapter().toNumericExpression(this).eq(expr);
        }               
        else
        {
            return super.eq(expr);
        }
    }
    
    public BooleanExpression noteq(ScalarExpression expr)
    {
        if (expr instanceof NullLiteral)
        {
            return expr.noteq(this);
        }
        else if (expr instanceof CharacterLiteral)
        {
            return new BooleanExpression(this, OP_NOTEQ, expr);            
        }
        else if (expr instanceof CharacterExpression)
        {
            return new BooleanExpression(this, OP_NOTEQ, expr);
        }
        else if (expr instanceof StringLiteral)
        {
            return new BooleanExpression(this, OP_NOTEQ, expr);            
        }
        else if (expr instanceof StringExpression)
        {
            return new BooleanExpression(this, OP_NOTEQ, expr); 
        }
        else if (expr instanceof NumericExpression)
        {
            return qs.getStoreManager().getDatastoreAdapter().toNumericExpression(this).noteq(expr);
        }               
        else
        {
            return super.noteq(expr);
        }
    }

    public BooleanExpression lt(ScalarExpression expr)
    {
        if (expr instanceof NullLiteral)
        {
            return expr.lt(this);
        }
        else if (expr instanceof CharacterLiteral)
        {
            return new BooleanExpression(this, OP_LT, expr);            
        }
        else if (expr instanceof CharacterExpression)
        {
            return new BooleanExpression(this, OP_LT, expr);
        }
        else if (expr instanceof StringLiteral)
        {
            return new BooleanExpression(this, OP_LT, expr);
        }
        else if (expr instanceof StringExpression)
        {
            return new BooleanExpression(this, OP_LT, expr);  
        }
        else if (expr instanceof NumericExpression)
        {
            return qs.getStoreManager().getDatastoreAdapter().toNumericExpression(this).lt(expr);
        }
        else
        {
            return super.lt(expr);
        }
    }

    public BooleanExpression lteq(ScalarExpression expr)
    {
        if (expr instanceof NullLiteral)
        {
            return expr.lteq(this);
        }
        else if (expr instanceof CharacterLiteral)
        {
            return new BooleanExpression(this, OP_LTEQ, expr);            
        }
        else if (expr instanceof CharacterExpression)
        {
            return new BooleanExpression(this, OP_LTEQ, expr);   
        }
        else if (expr instanceof StringLiteral)
        {
            return new BooleanExpression(this, OP_LTEQ, expr);            
        }
        else if (expr instanceof StringExpression)
        {
            return new BooleanExpression(this, OP_LTEQ, expr);   
        }
        else if (expr instanceof NumericExpression)
        {
            return qs.getStoreManager().getDatastoreAdapter().toNumericExpression(this).lteq(expr);
        }
        else
        {
            return super.lteq(expr);
        }
    }

    public BooleanExpression gt(ScalarExpression expr)
    {
        if (expr instanceof NullLiteral)
        {
            return expr.gt(this);
        }
        else if (expr instanceof CharacterLiteral)
        {
            return new BooleanExpression(this, OP_GT, expr);            
        }
        else if (expr instanceof CharacterExpression)
        {
            return new BooleanExpression(this, OP_GT, expr);  
        }
        else if (expr instanceof StringLiteral)
        {
            return new BooleanExpression(this, OP_GT, expr);            
        }
        else if (expr instanceof StringExpression)
        {
            return new BooleanExpression(this, OP_GT, expr);  
        }
        else if (expr instanceof NumericExpression)
        {
            return qs.getStoreManager().getDatastoreAdapter().toNumericExpression(this).gt(expr);
        }               
        else
        {
            return super.gt(expr);
        }
    }

    public BooleanExpression gteq(ScalarExpression expr)
    {
        if (expr instanceof NullLiteral)
        {
            return expr.gteq(this);
        }
        else if (expr instanceof CharacterLiteral)
        {
            return new BooleanExpression(this, OP_GTEQ, expr);            
        }
        else if (expr instanceof CharacterExpression)
        {
            return new BooleanExpression(this, OP_GTEQ, expr);   
        }
        else if (expr instanceof StringLiteral)
        {
            return new BooleanExpression(this, OP_GTEQ, expr);            
        }
        else if (expr instanceof StringExpression)
        {
            return new BooleanExpression(this, OP_GTEQ, expr);  
        }
        else if (expr instanceof NumericExpression)
        {
            return qs.getStoreManager().getDatastoreAdapter().toNumericExpression(this).gteq(expr);
        }               
        else
        {
            return super.gteq(expr);
        }
    }

    public ScalarExpression add(ScalarExpression expr)
    {
        if (expr instanceof CharacterExpression)
        {
            return new NumericExpression(qs.getStoreManager().getDatastoreAdapter().toNumericExpression(this), OP_ADD, qs.getStoreManager().getDatastoreAdapter().toNumericExpression((CharacterExpression) expr));
        }
        else if (expr instanceof NumericExpression )
        {
            return new NumericExpression(qs.getStoreManager().getDatastoreAdapter().toNumericExpression(this), OP_ADD, expr);
        }         
        else
        {
            return super.add(expr);
        }
    }

    public ScalarExpression sub(ScalarExpression expr)
    {
        if (expr instanceof CharacterExpression)
        {
            return new NumericExpression(qs.getStoreManager().getDatastoreAdapter().toNumericExpression(this), OP_SUB, qs.getStoreManager().getDatastoreAdapter().toNumericExpression((CharacterExpression) expr));
        }
        else if (expr instanceof NumericExpression )
        {
            return new NumericExpression(qs.getStoreManager().getDatastoreAdapter().toNumericExpression(this), OP_SUB, expr);
        }         
        else
        {
            return super.sub(expr);
        }
    }
    
    public BooleanExpression in(ScalarExpression expr)
    {
        return new BooleanExpression(this, OP_IN, expr);
    }
          
    /**
     * Converts the Character Expression to lower case
     * @return the lowercased Character Expression
     */
    public CharacterExpression toLowerCaseMethod()
    {
        ArrayList args = new ArrayList();
        args.add(this);

        return new CharacterExpression("LOWER", args);
    }

    /**
     * Converts the Character Expression to upper case
     * @return the uppercased Character Expression
     */
    public CharacterExpression toUpperCaseMethod()
    {
        ArrayList args = new ArrayList();
        args.add(this);

        return new CharacterExpression("UPPER", args);
    }
    
    public ScalarExpression mul(ScalarExpression expr)
    {
        if (expr instanceof NumericExpression)
        {
            return new NumericExpression(qs.getStoreManager().getDatastoreAdapter().toNumericExpression(this), OP_MUL, expr);
        }
        else if (expr instanceof CharacterExpression)
        {
            return new NumericExpression(qs.getStoreManager().getDatastoreAdapter().toNumericExpression(this), OP_MUL, qs.getStoreManager().getDatastoreAdapter().toNumericExpression((CharacterExpression) expr));
        }
        else if (expr instanceof NumericExpression)
        {
            return qs.getStoreManager().getDatastoreAdapter().toNumericExpression(this).mul(expr);
        }               
        else
        {
            return super.mul(expr);
        }
    }

    public ScalarExpression div(ScalarExpression expr)
    {
        if (expr instanceof NumericExpression)
        {
            return new NumericExpression(qs.getStoreManager().getDatastoreAdapter().toNumericExpression(this), OP_DIV, expr);
        }
        else if (expr instanceof CharacterExpression)
        {
            return new NumericExpression(qs.getStoreManager().getDatastoreAdapter().toNumericExpression(this), OP_DIV, qs.getStoreManager().getDatastoreAdapter().toNumericExpression((CharacterExpression) expr));
        }
        else if (expr instanceof NumericExpression)
        {
            return qs.getStoreManager().getDatastoreAdapter().toNumericExpression(this).div(expr);
        }               
        else
        {
            return super.div(expr);
        }
    }
       
    /**
     * Method to return a modulus expression.
     * @param expr The expression to modulus against
     * @return The modulus expression
     */
    public ScalarExpression mod(ScalarExpression expr)
    {
        if (expr instanceof CharacterExpression)
        {
            return qs.getStoreManager().getDatastoreAdapter().modOperator(qs.getStoreManager().getDatastoreAdapter().toNumericExpression(this), qs.getStoreManager().getDatastoreAdapter().toNumericExpression((CharacterExpression) expr));
        }        
        else if (expr instanceof NumericExpression)
        {
            return qs.getStoreManager().getDatastoreAdapter().modOperator(qs.getStoreManager().getDatastoreAdapter().toNumericExpression(this), expr);
        }
        else
        {
            return super.mod(expr);
        }
    }    
    
    public ScalarExpression neg()
    {
        return new NumericExpression(OP_NEG, qs.getStoreManager().getDatastoreAdapter().toNumericExpression(this));
    }
    
    public ScalarExpression com()
    {
        return qs.getStoreManager().getDatastoreAdapter().toNumericExpression(this).neg().sub(new IntegerLiteral(qs,mapping,BigInteger.ONE));
    }
    
    public ScalarExpression join(ScalarExpression expr)
    {
        /*this.joinExpr = expr;*/
        return this;
    }
}
