/*
 * Decompiled with CFR 0.152.
 */
package com.github.twitch4j.eventsub.util;

import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.github.twitch4j.common.util.CryptoUtils;
import java.nio.charset.StandardCharsets;
import java.security.InvalidKeyException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.time.Duration;
import java.time.Instant;
import java.time.format.DateTimeParseException;
import javax.crypto.Mac;
import javax.crypto.spec.SecretKeySpec;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class EventSubVerifier {
    private static final Logger log = LoggerFactory.getLogger(EventSubVerifier.class);
    private static final Duration RECENT_EVENT = Duration.ofMinutes(10L);
    private static final Cache<String, Boolean> RECENT_MESSAGE_IDS = Caffeine.newBuilder().expireAfterWrite(RECENT_EVENT).build();
    private static final String SIGNATURE_HASH_PREFIX = "sha256=";
    public static final String JAVA_HMAC_ALGORITHM = "HmacSHA256";
    private static final int HASH_LENGTH = 64;
    private static final ThreadLocal<Mac> HMAC_FUNCTION = ThreadLocal.withInitial(() -> {
        try {
            return Mac.getInstance(JAVA_HMAC_ALGORITHM);
        }
        catch (NoSuchAlgorithmException e) {
            return null;
        }
    });

    public static boolean verifyMessageId(String messageId) {
        return messageId != null && !messageId.isEmpty() && RECENT_MESSAGE_IDS.asMap().putIfAbsent(messageId, Boolean.TRUE) == null;
    }

    public static boolean verifyTimestamp(String messageTimestamp) {
        Instant timestamp;
        if (messageTimestamp == null || messageTimestamp.isEmpty()) {
            return false;
        }
        try {
            timestamp = Instant.parse(messageTimestamp);
        }
        catch (DateTimeParseException e) {
            log.debug("Failed to parse timestamp: " + messageTimestamp, (Throwable)e);
            return false;
        }
        return Duration.between(timestamp, Instant.now()).compareTo(RECENT_EVENT) < 0;
    }

    public static boolean verifySignature(SecretKeySpec secret, String messageId, String messageTimestamp, byte[] requestBody, String expectedSignature) {
        if (secret == null || expectedSignature == null || messageId == null || messageTimestamp == null || requestBody == null) {
            log.warn("Could not verify eventsub signature with null argument");
            return false;
        }
        if (expectedSignature.length() - SIGNATURE_HASH_PREFIX.length() != 64 || !StringUtils.startsWithIgnoreCase((CharSequence)expectedSignature, (CharSequence)SIGNATURE_HASH_PREFIX)) {
            log.debug("Could not verify unknown eventsub signature hash scheme; " + expectedSignature);
            return false;
        }
        Mac mac = HMAC_FUNCTION.get();
        if (mac == null) {
            log.error("Unable to prepare hash function for eventsub signature verification!");
            return false;
        }
        try {
            mac.init(secret);
        }
        catch (InvalidKeyException e) {
            log.error("Unable to initialize secret for eventsub signature verification!", (Throwable)e);
            return false;
        }
        byte[] id = messageId.getBytes(StandardCharsets.UTF_8);
        byte[] timestamp = messageTimestamp.getBytes(StandardCharsets.UTF_8);
        byte[] message = new byte[id.length + timestamp.length + requestBody.length];
        System.arraycopy(id, 0, message, 0, id.length);
        System.arraycopy(timestamp, 0, message, id.length, timestamp.length);
        System.arraycopy(requestBody, 0, message, id.length + timestamp.length, requestBody.length);
        byte[] computedHmac = mac.doFinal(message);
        mac.reset();
        byte[] expectedHmac = CryptoUtils.hexToBytes((String)expectedSignature.substring(SIGNATURE_HASH_PREFIX.length()));
        return MessageDigest.isEqual(computedHmac, expectedHmac);
    }

    public static boolean verifySignature(byte[] secret, String messageId, String messageTimestamp, byte[] requestBody, String expectedSignature) {
        return EventSubVerifier.verifySignature(new SecretKeySpec(secret, JAVA_HMAC_ALGORITHM), messageId, messageTimestamp, requestBody, expectedSignature);
    }

    public static boolean verifySignature(String secret, String messageId, String messageTimestamp, byte[] requestBody, String expectedSignature) {
        return EventSubVerifier.verifySignature(secret.getBytes(StandardCharsets.UTF_8), messageId, messageTimestamp, requestBody, expectedSignature);
    }

    private EventSubVerifier() {
        throw new UnsupportedOperationException("This is a utility class and cannot be instantiated");
    }
}

