package com.github.twitch4j.common.util;

import com.github.twitch4j.common.enums.CommandPermission;
import com.github.twitch4j.common.events.domain.EventUser;
import lombok.NonNull;
import org.apache.commons.lang3.StringUtils;
import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.Nullable;

import java.util.Collection;
import java.util.Collections;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

public class TwitchUtils {

    /**
     * The account used by twitch to signify an anonymous subscription gifter
     *
     * @see <a href="https://discuss.dev.twitch.tv/t/anonymous-sub-gifting-to-launch-11-15-launch-details/18683">Official Announcement</a>
     */
    public static final EventUser ANONYMOUS_GIFTER = new EventUser("274598607", "ananonymousgifter");

    /**
     * The account used by twitch to signify an anonymous cheerer
     */
    public static final EventUser ANONYMOUS_CHEERER = new EventUser("407665396", "ananonymouscheerer");

    @Deprecated // not used by twitch4j
    public static Set<CommandPermission> getPermissionsFromTags(Map<String, Object> tags) {
        return getPermissionsFromTags(tags, new HashMap<>());
    }

    @Deprecated // not used by twitch4j
    public static Set<CommandPermission> getPermissionsFromTags(@NonNull Map<String, Object> tags, @NonNull Map<String, String> badges) {
        return getPermissionsFromTags(tags, badges, null, null);
    }

    @ApiStatus.Internal
    public static Set<CommandPermission> getPermissionsFromTags(@NonNull Map<String, Object> tags, @NonNull Map<String, String> badges, String userId, Collection<String> botOwnerIds) {
        // allows for accurate sub detection when user has the founder badge equipped
        Object subscriber = tags.get("subscriber");
        if (subscriber instanceof CharSequence && !StringUtils.equals("0", (CharSequence) subscriber)) {
            badges.put("subscriber", subscriber.toString());
        }

        // irc parsing branch
        Object inputBadges = tags.get("badges");
        if (inputBadges instanceof CharSequence) {
            return getPermissionsFromTags((CharSequence) inputBadges, userId, botOwnerIds, badges);
        }

        // otherwise: handle pubsub whispers topic
        if (inputBadges instanceof Collection) {
            Collection<?> list = (Collection<?>) inputBadges;
            for (Object badgeObj : list) {
                if (badgeObj instanceof Map) {
                    Map<?, ?> badge = (Map<?, ?>) badgeObj;
                    Object badgeId = badge.get("id");
                    if (badgeId instanceof String) {
                        Object badgeVersion = badge.get("version");
                        badges.put((String) badgeId, badgeVersion instanceof String ? (String) badgeVersion : null);
                    }
                }
            }
        }
        return getPermissionsFromTags(null, userId, botOwnerIds, badges);
    }

    private static Set<CommandPermission> getPermissionsFromTags(@Nullable CharSequence badgesTag, String userId, Collection<String> botOwnerIds, @NonNull Map<String, String> badges) {
        Set<CommandPermission> permissionSet = EnumSet.of(CommandPermission.EVERYONE);

        // Parse badges tag
        if (badgesTag != null) {
            badges.putAll(parseBadges(badgesTag.toString()));
        }

        // Check for Permissions
        if (!badges.isEmpty()) {
            // Broadcaster
            if (badges.containsKey("broadcaster")) {
                permissionSet.add(CommandPermission.BROADCASTER);
                permissionSet.add(CommandPermission.MODERATOR);
            }
            // Twitch Prime
            if (badges.containsKey("premium") || badges.containsKey("turbo")) {
                permissionSet.add(CommandPermission.PRIME_TURBO);
            }
            // Moderator
            if (badges.containsKey("moderator")) {
                permissionSet.add(CommandPermission.MODERATOR);
            }
            // Partner
            if (badges.containsKey("partner") || badges.containsKey("ambassador")) {
                permissionSet.add(CommandPermission.PARTNER);
            }
            // VIP
            if (badges.containsKey("vip")) {
                permissionSet.add(CommandPermission.VIP);
            }
            // Turbo
            if (badges.containsKey("turbo")) {
                permissionSet.add(CommandPermission.PRIME_TURBO);
            }
            // Twitch Staff
            if (badges.containsKey("staff") || badges.containsKey("admin")) {
                permissionSet.add(CommandPermission.TWITCHSTAFF);
            }
            // Subscriber
            if (badges.containsKey("subscriber")) {
                permissionSet.add(CommandPermission.SUBSCRIBER);
            }
            // SubGifter
            if (badges.containsKey("sub-gifter") || badges.containsKey("sub-gift-leader")) {
                permissionSet.add(CommandPermission.SUBGIFTER);
            }
            // Cheerer
            if (badges.containsKey("bits") || badges.containsKey("bits-leader") || badges.containsKey("anonymous-cheerer")) {
                permissionSet.add(CommandPermission.BITS_CHEERER);
            }
            // Founder
            if (badges.containsKey("founder")) {
                permissionSet.add(CommandPermission.FOUNDER);

                // also contains info about the tier if needed
                /*
                if (badges.get("founder").equals("0")) {
                    // Tier 1 Founder
                } else if (badges.get("founder").equals("1")) {
                    // Tier 2 Founder
                } else if (badges.get("founder").equals("2")) {
                    // Tier 3 Founder
                }
                */
            }
            // Hype Train Conductor
            String hypeBadge = badges.get("hype-train");
            if ("1".equals(hypeBadge)) {
                permissionSet.add(CommandPermission.CURRENT_HYPE_TRAIN_CONDUCTOR);
            } else if ("2".equals(hypeBadge)) {
                permissionSet.add(CommandPermission.FORMER_HYPE_TRAIN_CONDUCTOR);
            }
            // Predictions Participation
            String predictionBadge = badges.get("predictions");
            if (StringUtils.isNotEmpty(predictionBadge)) {
                char first = predictionBadge.charAt(0);
                if (first == 'b') {
                    permissionSet.add(CommandPermission.PREDICTIONS_BLUE);
                } else if (first == 'p') {
                    permissionSet.add(CommandPermission.PREDICTIONS_PINK);
                }
            }
            // Accessibility
            if (badges.containsKey("no_audio")) {
                permissionSet.add(CommandPermission.NO_AUDIO);
            }
            if (badges.containsKey("no_video")) {
                permissionSet.add(CommandPermission.NO_VIDEO);
            }
            // Present for Channel Moment
            if (badges.containsKey("moments")) {
                permissionSet.add(CommandPermission.MOMENTS);
            }
            // Channel Emote Artist
            if (badges.containsKey("artist-badge")) {
                permissionSet.add(CommandPermission.ARTIST);
            }
        }

        if (userId != null && botOwnerIds != null && botOwnerIds.contains(userId))
            permissionSet.add(CommandPermission.OWNER);

        return permissionSet;
    }

    /**
     * Parse Badges from raw list
     *
     * @param raw The raw list of tags.
     * @return A key-value map of the tags.
     */
    public static Map<String, String> parseBadges(String raw) {
        Map<String, String> map = new HashMap<>();
        if (StringUtils.isBlank(raw)) return map;

        // Fix Whitespaces
        raw = EscapeUtils.unescapeTagValue(raw);

        for (String tag : raw.split(",")) {
            String[] val = tag.split("/", 2);
            final String key = val[0];
            String value = (val.length > 1) ? val[1] : null;
            map.put(key, value);
        }

        return Collections.unmodifiableMap(map); // formatting to Read-Only Map
    }

}
