package io.github.retrooper.packetevents.factory.netty;

import com.github.retrooper.packetevents.PacketEvents;
import com.github.retrooper.packetevents.PacketEventsAPI;
import com.github.retrooper.packetevents.event.PacketListenerPriority;
import com.github.retrooper.packetevents.injector.ChannelInjector;
import com.github.retrooper.packetevents.injector.InternalPacketListener;
import com.github.retrooper.packetevents.manager.player.PlayerManager;
import com.github.retrooper.packetevents.manager.server.ServerManager;
import com.github.retrooper.packetevents.netty.NettyManager;
import com.github.retrooper.packetevents.protocol.packettype.PacketType;
import com.github.retrooper.packetevents.settings.PacketEventsSettings;
import io.github.retrooper.packetevents.manager.netty.NettyManagerImpl;
import io.github.retrooper.packetevents.manager.player.PlayerManagerImpl;
import io.github.retrooper.packetevents.manager.server.ServerManagerImpl;

public class NettyPacketEventsBuilder {
    private static PacketEventsAPI<BuildData> INSTANCE;

    public static void clearBuildCache() {
        INSTANCE = null;
    }

    public static PacketEventsAPI<BuildData> build(BuildData data, ChannelInjector injector,
                                                   ServerManagerImpl serverManager,
                                                   PlayerManagerImpl playerManager) {
        if (INSTANCE == null) {
            INSTANCE = buildNoCache(data, injector, serverManager, playerManager);
        }
        return INSTANCE;
    }

    public static PacketEventsAPI<BuildData> build(BuildData data, ChannelInjector injector,
                                                   ServerManagerImpl serverManager,
                                                   PlayerManagerImpl playerManager, PacketEventsSettings settings) {
        if (INSTANCE == null) {
            INSTANCE = buildNoCache(data, injector, serverManager, playerManager, settings);
        }
        return INSTANCE;
    }


    public static PacketEventsAPI<BuildData> buildNoCache(BuildData data, ChannelInjector injector,
                                                          ServerManagerImpl serverManager,
                                                          PlayerManagerImpl playerManager) {
        return buildNoCache(data, injector, serverManager, playerManager, new PacketEventsSettings());
    }

    public static PacketEventsAPI<BuildData> buildNoCache(BuildData data, ChannelInjector injector,
                                                          ServerManagerImpl serverManager,
                                                          PlayerManagerImpl playerManager,
                                                          PacketEventsSettings inSettings) {
        return new PacketEventsAPI<BuildData>() {
            private final PacketEventsSettings settings = inSettings;
            private final NettyManager nettyManager = new NettyManagerImpl();
            private boolean loaded;
            private boolean initialized;

            @Override
            public void load() {
                if (!loaded) {
                    //Resolve server version and cache
                    PacketEvents.IDENTIFIER = "pe-" + data.getName().toLowerCase();
                    PacketEvents.ENCODER_NAME = "pe-encoder-" + data.getName().toLowerCase();
                    PacketEvents.DECODER_NAME = "pe-decoder-" + data.getName().toLowerCase();
                    PacketEvents.CONNECTION_NAME = "pe-connection-handler-" + data.getName().toLowerCase();
                    PacketEvents.SERVER_CHANNEL_HANDLER_NAME = "pe-connection-initializer-" + data.getName().toLowerCase();
                    injector.inject();

                    loaded = true;

                    //Register internal packet listener (should be the first listener)
                    //This listener doesn't do any modifications to the packets, just reads data
                    getEventManager().registerListener(new InternalPacketListener(),
                            PacketListenerPriority.LOWEST, true);
                }
            }

            @Override
            public boolean isLoaded() {
                return loaded;
            }

            @Override
            public void init() {
                //Load if we haven't loaded already
                load();
                if (!initialized) {
                    if (settings.shouldCheckForUpdates()) {
                        getUpdateChecker().handleUpdateCheck();
                    }

                    if (settings.isbStatsEnabled()) {
                        //TODO Cross-platform metrics?
                    }

                    PacketType.Play.Client.load();
                    PacketType.Play.Server.load();
                    initialized = true;
                }
            }

            @Override
            public boolean isInitialized() {
                return initialized;
            }

            @Override
            public void terminate() {
                if (initialized) {
                    //Eject the injector if needed(depends on the injector implementation)
                    injector.eject();
                    //Unregister all our listeners
                    getEventManager().unregisterAllListeners();
                    initialized = false;
                }
            }

            @Override
            public BuildData getPlugin() {
                return data;
            }

            @Override
            public ServerManager getServerManager() {
                return serverManager;
            }

            @Override
            public PlayerManager getPlayerManager() {
                return playerManager;
            }

            @Override
            public PacketEventsSettings getSettings() {
                return settings;
            }

            @Override
            public NettyManager getNettyManager() {
                return nettyManager;
            }

            @Override
            public ChannelInjector getInjector() {
                return injector;
            }
        };
    }
}
