package com.github.niupengyu.schedule.mains.util;

import org.quartz.*;
import org.quartz.impl.triggers.SimpleTriggerImpl;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


public class QuartzManager {
	
	private static final Logger logger= LoggerFactory.getLogger(QuartzManager.class);
	
	private QuartzManager() {
	}
	
    /** 
     * @Description: 添加一个定时任务 
     *  
     * @param jobName 任务名 
     * @param jobGroupName  任务组名 
     * @param triggerName 触发器名 
     * @param triggerGroupName 触发器组名 
     * @param jobClass  任务 
     * @param cron   时间设置，参考quartz说明文档  
     */  
    @SuppressWarnings({ "unchecked", "rawtypes" })  
    public static void addJob(String jobName, String jobGroupName, 
        String triggerName, String triggerGroupName, Class jobClass, String cron,Scheduler sched) throws SchedulerException {
        // 任务名，任务组，任务执行类
        JobDetail job= JobBuilder.newJob(jobClass).withIdentity(jobName, jobGroupName).build();
       // 触发器
        TriggerBuilder<Trigger> triggerBuilder = TriggerBuilder.newTrigger();
        // 触发器名,触发器组
        triggerBuilder.withIdentity(triggerName, triggerGroupName);
        triggerBuilder.startNow();
        // 触发器时间设定
        triggerBuilder.withSchedule(CronScheduleBuilder.cronSchedule(cron));
        // 创建Trigger对象
        CronTrigger trigger = (CronTrigger) triggerBuilder.build();
        // 调度容器设置JobDetail和Trigger
        sched.scheduleJob(job, trigger);
        // 启动
        if (!sched.isShutdown()) {
            sched.start();
        }
    }
    
    public static void addJobNow(String jobName, String jobGroupName,
        String triggerName, String triggerGroupName, Class jobClass,Scheduler sched) throws SchedulerException {
        // 任务名，任务组，任务执行类
          JobDetail job= JobBuilder.newJob(jobClass).withIdentity(jobName, jobGroupName).build();
       // 触发器
        TriggerBuilder<Trigger> triggerBuilder = TriggerBuilder.newTrigger();
        // 触发器名,触发器组
        triggerBuilder.withIdentity(triggerName, triggerGroupName);
        triggerBuilder.startNow();
        // 创建Trigger对象
        SimpleTriggerImpl trigger = (SimpleTriggerImpl) triggerBuilder.build();
        // 调度容器设置JobDetail和Trigger
        sched.scheduleJob(job, trigger);
        // 启动
        if (!sched.isShutdown()) {
            sched.start();
        }
    }

    /** 
     * @Description: 修改一个任务的触发时间
     *  
     * @param jobName 
     * @param jobGroupName
     * @param triggerName 触发器名
     * @param triggerGroupName 触发器组名 
     * @param cron   时间设置，参考quartz说明文档   
     */  
    public static void modifyJobTime( 
        String triggerName, String triggerGroupName, String cron,Scheduler sched) throws SchedulerException {
        TriggerKey triggerKey = TriggerKey.triggerKey(triggerName, triggerGroupName);
        CronTrigger trigger = (CronTrigger) sched.getTrigger(triggerKey);
        if (trigger == null) {
            return;
        }

        String oldTime = trigger.getCronExpression();
        if (!oldTime.equalsIgnoreCase(cron)) {
            /** 方式一 ：调用 rescheduleJob 开始 */
            // 触发器
            TriggerBuilder<Trigger> triggerBuilder = TriggerBuilder.newTrigger();
            // 触发器名,触发器组
            triggerBuilder.withIdentity(triggerName, triggerGroupName);
            triggerBuilder.startNow();
            // 触发器时间设定
            triggerBuilder.withSchedule(CronScheduleBuilder.cronSchedule(cron));
            // 创建Trigger对象
            trigger = (CronTrigger) triggerBuilder.build();
            // 方式一 ：修改一个任务的触发时间
            sched.rescheduleJob(triggerKey, trigger);
        }
    }

    /** 
     * @Description: 移除一个任务 
     *  
     * @param jobName 
     * @param jobGroupName 
     * @param triggerName 
     * @param triggerGroupName 
     */  
    public static boolean removeJob(String jobName, String jobGroupName,  
            String triggerName, String triggerGroupName,Scheduler sched)  throws SchedulerException{
        TriggerKey triggerKey = TriggerKey.triggerKey(triggerName, triggerGroupName);
        sched.pauseTrigger(triggerKey);// 停止触发器
        boolean flag1=sched.unscheduleJob(triggerKey);// 移除触发器
        JobKey jobKey=JobKey.jobKey(jobName, jobGroupName);
        boolean flag=sched.deleteJob(jobKey);// 删除任务
        return flag;
    }
    
    public static void pauseJob(String jobName, String jobGroupName,  
            String triggerName, String triggerGroupName,Scheduler sched) throws SchedulerException {
        TriggerKey triggerKey = TriggerKey.triggerKey(triggerName, triggerGroupName);
        sched.pauseTrigger(triggerKey);// 停止触发器
        sched.pauseJob(JobKey.jobKey(jobName, jobGroupName));// 删除任务
    }

    public static void resumeJob(String jobName, String jobGroupName,
                                String triggerName, String triggerGroupName,Scheduler sched) throws SchedulerException {
        TriggerKey triggerKey = TriggerKey.triggerKey(triggerName, triggerGroupName);
        sched.resumeTrigger(triggerKey);// 停止触发器
        sched.resumeJob(JobKey.jobKey(jobName, jobGroupName));// 删除任务
    }

    public static boolean unscheduleJob(String triggerName, String triggerGroupName,Scheduler sched)
            throws SchedulerException{
        TriggerKey triggerKey = TriggerKey.triggerKey(triggerName, triggerGroupName);
        return sched.unscheduleJob(triggerKey);
    }

    /** 
     * @Description:启动所有定时任务 
     */  
    public static void startJobs(Scheduler sched) {  
        try {  
            sched.start();  
        } catch (Exception e) {  
        	logger.error(e.getMessage(), e);
        }  
    }  

    /** 
     * @Description:关闭所有定时任务 
     */  
    public static void shutdownJobs(Scheduler sched) throws SchedulerException {
        if (!sched.isShutdown()) {
            sched.shutdown();
        }
    }

    public static void interrupt(Scheduler sched,String jobName,String jobGroupName) throws UnableToInterruptJobException {
        sched.interrupt(JobKey.jobKey(jobName, jobGroupName));
    }

}