/*
 * Decompiled with CFR 0.152.
 */
package smile.math.matrix;

import java.io.Serializable;
import java.nio.DoubleBuffer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import smile.math.MathEx;
import smile.math.blas.BLAS;
import smile.math.blas.LAPACK;
import smile.math.blas.Layout;
import smile.math.blas.Transpose;
import smile.math.blas.UPLO;
import smile.math.matrix.IMatrix;
import smile.math.matrix.Matrix;

public class SymmMatrix
extends IMatrix {
    private static final long serialVersionUID = 2L;
    private static final Logger logger = LoggerFactory.getLogger(SymmMatrix.class);
    final double[] AP;
    final int n;
    final UPLO uplo;

    public SymmMatrix(UPLO uplo, int n) {
        if (uplo == null) {
            throw new NullPointerException("UPLO is null");
        }
        this.uplo = uplo;
        this.n = n;
        this.AP = new double[n * (n + 1) / 2];
    }

    public SymmMatrix(UPLO uplo, double[][] AP) {
        this(uplo, AP.length);
        if (uplo == UPLO.LOWER) {
            for (int i = 0; i < this.n; ++i) {
                for (int j = 0; j <= i; ++j) {
                    this.AP[i + (2 * this.n - j - 1) * j / 2] = AP[i][j];
                }
            }
        } else {
            for (int i = 0; i < this.n; ++i) {
                for (int j = i; j < this.n; ++j) {
                    this.AP[i + j * (j + 1) / 2] = AP[i][j];
                }
            }
        }
    }

    @Override
    public SymmMatrix copy() {
        SymmMatrix matrix = new SymmMatrix(this.uplo, this.n);
        System.arraycopy(this.AP, 0, matrix.AP, 0, this.AP.length);
        return matrix;
    }

    @Override
    public int nrow() {
        return this.n;
    }

    @Override
    public int ncol() {
        return this.n;
    }

    @Override
    public long size() {
        return this.AP.length;
    }

    public Layout layout() {
        return Layout.COL_MAJOR;
    }

    public UPLO uplo() {
        return this.uplo;
    }

    public boolean equals(Object o) {
        if (!(o instanceof SymmMatrix)) {
            return false;
        }
        return this.equals((SymmMatrix)o, 1.0E-10);
    }

    public boolean equals(SymmMatrix o, double epsilon) {
        if (this.n != o.n) {
            return false;
        }
        for (int j = 0; j < this.n; ++j) {
            for (int i = 0; i < this.n; ++i) {
                if (MathEx.isZero(this.get(i, j) - o.get(i, j), epsilon)) continue;
                return false;
            }
        }
        return true;
    }

    @Override
    public double get(int i, int j) {
        if (this.uplo == UPLO.LOWER) {
            if (j > i) {
                int tmp = i;
                i = j;
                j = tmp;
            }
            return this.AP[i + (2 * this.n - j - 1) * j / 2];
        }
        if (i > j) {
            int tmp = i;
            i = j;
            j = tmp;
        }
        return this.AP[i + j * (j + 1) / 2];
    }

    @Override
    public void set(int i, int j, double x) {
        if (this.uplo == UPLO.LOWER) {
            if (j > i) {
                int tmp = i;
                i = j;
                j = tmp;
            }
            this.AP[i + (2 * this.n - j - 1) * j / 2] = x;
        } else {
            if (i > j) {
                int tmp = i;
                i = j;
                j = tmp;
            }
            this.AP[i + j * (j + 1) / 2] = x;
        }
    }

    @Override
    public void mv(Transpose trans, double alpha, double[] x, double beta, double[] y) {
        BLAS.engine.spmv(this.layout(), this.uplo, this.n, alpha, this.AP, x, 1, beta, y, 1);
    }

    @Override
    public void mv(double[] work, int inputOffset, int outputOffset) {
        DoubleBuffer xb = DoubleBuffer.wrap(work, inputOffset, this.n);
        DoubleBuffer yb = DoubleBuffer.wrap(work, outputOffset, this.n);
        BLAS.engine.spmv(this.layout(), this.uplo, this.n, 1.0, DoubleBuffer.wrap(this.AP), xb, 1, 0.0, yb, 1);
    }

    @Override
    public void tv(double[] work, int inputOffset, int outputOffset) {
        this.mv(work, inputOffset, outputOffset);
    }

    public BunchKaufman bk() {
        SymmMatrix lu = this.copy();
        int[] ipiv = new int[this.n];
        int info = LAPACK.engine.sptrf(lu.layout(), lu.uplo, lu.n, lu.AP, ipiv);
        if (info < 0) {
            logger.error("LAPACK SPTRF error code: {}", (Object)info);
            throw new ArithmeticException("LAPACK SPTRF error code: " + info);
        }
        return new BunchKaufman(lu, ipiv, info);
    }

    public Cholesky cholesky() {
        if (this.uplo == null) {
            throw new IllegalArgumentException("The matrix is not symmetric");
        }
        SymmMatrix lu = this.copy();
        int info = LAPACK.engine.pptrf(lu.layout(), lu.uplo, lu.n, lu.AP);
        if (info != 0) {
            logger.error("LAPACK PPTRF error code: {}", (Object)info);
            throw new ArithmeticException("LAPACK PPTRF error code: " + info);
        }
        return new Cholesky(lu);
    }

    public static class BunchKaufman
    implements Serializable {
        private static final long serialVersionUID = 2L;
        public final SymmMatrix lu;
        public final int[] ipiv;
        public final int info;

        public BunchKaufman(SymmMatrix lu, int[] ipiv, int info) {
            this.lu = lu;
            this.ipiv = ipiv;
            this.info = info;
        }

        public boolean isSingular() {
            return this.info > 0;
        }

        public double det() {
            int j;
            int n = this.lu.n;
            double d = 1.0;
            for (j = 0; j < n; ++j) {
                d *= this.lu.get(j, j);
            }
            for (j = 0; j < n; ++j) {
                if (j + 1 == this.ipiv[j]) continue;
                d = -d;
            }
            return d;
        }

        public Matrix inverse() {
            Matrix inv = Matrix.eye(this.lu.n);
            this.solve(inv);
            return inv;
        }

        public double[] solve(double[] b) {
            Matrix x = Matrix.column(b);
            this.solve(x);
            return x.A;
        }

        public void solve(Matrix B) {
            if (B.m != this.lu.n) {
                throw new IllegalArgumentException(String.format("Row dimensions do not agree: A is %d x %d, but B is %d x %d", this.lu.n, this.lu.n, B.m, B.n));
            }
            if (this.lu.layout() != B.layout()) {
                throw new IllegalArgumentException("The matrix layout is inconsistent.");
            }
            if (this.info > 0) {
                throw new RuntimeException("The matrix is singular.");
            }
            int ret = LAPACK.engine.sptrs(this.lu.layout(), this.lu.uplo, this.lu.n, B.n, this.lu.AP, this.ipiv, B.A, B.ld);
            if (ret != 0) {
                logger.error("LAPACK GETRS error code: {}", (Object)ret);
                throw new ArithmeticException("LAPACK GETRS error code: " + ret);
            }
        }
    }

    public static class Cholesky
    implements Serializable {
        private static final long serialVersionUID = 2L;
        public final SymmMatrix lu;

        public Cholesky(SymmMatrix lu) {
            if (lu.nrow() != lu.ncol()) {
                throw new UnsupportedOperationException("Cholesky constructor on a non-square matrix");
            }
            this.lu = lu;
        }

        public double det() {
            double d = 1.0;
            for (int i = 0; i < this.lu.n; ++i) {
                d *= this.lu.get(i, i);
            }
            return d * d;
        }

        public double logdet() {
            int n = this.lu.n;
            double d = 0.0;
            for (int i = 0; i < n; ++i) {
                d += Math.log(this.lu.get(i, i));
            }
            return 2.0 * d;
        }

        public Matrix inverse() {
            Matrix inv = Matrix.eye(this.lu.n);
            this.solve(inv);
            return inv;
        }

        public double[] solve(double[] b) {
            Matrix x = Matrix.column(b);
            this.solve(x);
            return x.A;
        }

        public void solve(Matrix B) {
            if (B.m != this.lu.n) {
                throw new IllegalArgumentException(String.format("Row dimensions do not agree: A is %d x %d, but B is %d x %d", this.lu.n, this.lu.n, B.m, B.n));
            }
            int info = LAPACK.engine.pptrs(this.lu.layout(), this.lu.uplo, this.lu.n, B.n, this.lu.AP, B.A, B.ld);
            if (info != 0) {
                logger.error("LAPACK POTRS error code: {}", (Object)info);
                throw new ArithmeticException("LAPACK POTRS error code: " + info);
            }
        }
    }
}

