/*
 * Decompiled with CFR 0.152.
 */
package smile.math.matrix;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import org.bytedeco.javacpp.DoublePointer;
import org.bytedeco.javacpp.IntPointer;
import org.bytedeco.javacpp.Pointer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import smile.math.MathEx;
import smile.math.blas.BLAS;
import smile.math.blas.Diag;
import smile.math.blas.EVDJob;
import smile.math.blas.LAPACK;
import smile.math.blas.Layout;
import smile.math.blas.SVDJob;
import smile.math.blas.Side;
import smile.math.blas.Transpose;
import smile.math.blas.UPLO;
import smile.math.matrix.IMatrix;
import smile.sort.QuickSort;
import smile.stat.distribution.Distribution;
import smile.stat.distribution.GaussianDistribution;

public class BigMatrix
extends IMatrix
implements AutoCloseable {
    private static final long serialVersionUID = 3L;
    private static final Logger logger = LoggerFactory.getLogger(BigMatrix.class);
    transient DoublePointer A;
    transient int ld;
    int m;
    int n;
    UPLO uplo;
    Diag diag;

    public BigMatrix(int m, int n) {
        this(m, n, 0.0);
    }

    public BigMatrix(int m, int n, double a) {
        if (m <= 0 || n <= 0) {
            throw new IllegalArgumentException(String.format("Invalid matrix size: %d x %d", m, n));
        }
        this.m = m;
        this.n = n;
        this.ld = BigMatrix.ld(m);
        this.A = new DoublePointer((long)this.ld * (long)n);
        this.fill(a);
    }

    public BigMatrix(int m, int n, int ld, DoublePointer A) {
        if (this.layout() == Layout.COL_MAJOR && ld < m) {
            throw new IllegalArgumentException(String.format("Invalid leading dimension for COL_MAJOR: %d < %d", ld, m));
        }
        if (this.layout() == Layout.ROW_MAJOR && ld < n) {
            throw new IllegalArgumentException(String.format("Invalid leading dimension for ROW_MAJOR: %d < %d", ld, n));
        }
        this.m = m;
        this.n = n;
        this.ld = ld;
        this.A = A;
    }

    @Override
    public void close() {
        this.A.close();
    }

    public static BigMatrix of(double[][] A) {
        int m = A.length;
        int n = A[0].length;
        BigMatrix matrix = new BigMatrix(m, n);
        for (int i = 0; i < m; ++i) {
            for (int j = 0; j < n; ++j) {
                matrix.set(i, j, A[i][j]);
            }
        }
        return matrix;
    }

    public static BigMatrix column(double[] A) {
        return BigMatrix.column(A, 0, A.length);
    }

    public static BigMatrix column(double[] A, int offset, int length) {
        DoublePointer pointer = new DoublePointer((long)length);
        pointer.put(A, offset, length);
        return new BigMatrix(length, 1, length, pointer);
    }

    public static BigMatrix row(double[] A) {
        return BigMatrix.row(A, 0, A.length);
    }

    public static BigMatrix row(double[] A, int offset, int length) {
        DoublePointer pointer = new DoublePointer((long)length);
        pointer.put(A, offset, length);
        return new BigMatrix(1, length, 1, pointer);
    }

    public static BigMatrix rand(int m, int n, Distribution distribution) {
        BigMatrix matrix = new BigMatrix(m, n);
        for (int j = 0; j < n; ++j) {
            for (int i = 0; i < m; ++i) {
                matrix.set(i, j, distribution.rand());
            }
        }
        return matrix;
    }

    public static BigMatrix randn(int m, int n) {
        return BigMatrix.rand(m, n, GaussianDistribution.getInstance());
    }

    public static BigMatrix rand(int m, int n) {
        BigMatrix matrix = new BigMatrix(m, n);
        for (int j = 0; j < n; ++j) {
            for (int i = 0; i < m; ++i) {
                matrix.set(i, j, MathEx.random());
            }
        }
        return matrix;
    }

    public static BigMatrix rand(int m, int n, double lo, double hi) {
        BigMatrix matrix = new BigMatrix(m, n);
        for (int j = 0; j < n; ++j) {
            for (int i = 0; i < m; ++i) {
                matrix.set(i, j, MathEx.random(lo, hi));
            }
        }
        return matrix;
    }

    public static BigMatrix eye(int n) {
        return BigMatrix.diag(n, 1.0);
    }

    public static BigMatrix eye(int m, int n) {
        return BigMatrix.diag(m, n, 1.0);
    }

    public static BigMatrix diag(int n, double diag) {
        return BigMatrix.diag(n, n, diag);
    }

    public static BigMatrix diag(int m, int n, double diag) {
        BigMatrix D = new BigMatrix(m, n);
        int k = Math.min(m, n);
        for (int i = 0; i < k; ++i) {
            D.set(i, i, diag);
        }
        return D;
    }

    public static BigMatrix diag(double[] diag) {
        int n = diag.length;
        BigMatrix D = new BigMatrix(n, n);
        for (int i = 0; i < n; ++i) {
            D.set(i, i, diag[i]);
        }
        return D;
    }

    public static BigMatrix diag(DoublePointer diag) {
        int n = (int)BigMatrix.length(diag);
        BigMatrix D = new BigMatrix(n, n);
        for (int i = 0; i < n; ++i) {
            D.set(i, i, diag.get((long)i));
        }
        return D;
    }

    public static BigMatrix toeplitz(double[] a) {
        int n = a.length;
        BigMatrix toeplitz = new BigMatrix(n, n);
        toeplitz.uplo(UPLO.LOWER);
        for (int i = 0; i < n; ++i) {
            int j;
            for (j = 0; j < i; ++j) {
                toeplitz.set(i, j, a[i - j]);
            }
            for (j = i; j < n; ++j) {
                toeplitz.set(i, j, a[j - i]);
            }
        }
        return toeplitz;
    }

    public static BigMatrix toeplitz(double[] kl, double[] ku) {
        if (kl.length != ku.length - 1) {
            throw new IllegalArgumentException(String.format("Invalid subdiagonals and superdiagonals size: %d != %d - 1", kl.length, ku.length));
        }
        int n = kl.length;
        BigMatrix toeplitz = new BigMatrix(n, n);
        for (int i = 0; i < n; ++i) {
            int j;
            for (j = 0; j < i; ++j) {
                toeplitz.set(i, j, kl[i - j]);
            }
            for (j = i; j < n; ++j) {
                toeplitz.set(i, j, ku[j - i]);
            }
        }
        return toeplitz;
    }

    private void writeObject(ObjectOutputStream out) throws IOException {
        out.defaultWriteObject();
        if (this.layout() == Layout.COL_MAJOR) {
            for (int j = 0; j < this.n; ++j) {
                for (int i = 0; i < this.m; ++i) {
                    out.writeDouble(this.get(i, j));
                }
            }
        } else {
            for (int i = 0; i < this.m; ++i) {
                for (int j = 0; j < this.n; ++j) {
                    out.writeDouble(this.get(i, j));
                }
            }
        }
    }

    private void readObject(ObjectInputStream in) throws IOException, ClassNotFoundException {
        in.defaultReadObject();
        if (this.layout() == Layout.COL_MAJOR) {
            this.ld = BigMatrix.ld(this.m);
            this.A = new DoublePointer((long)this.ld * (long)this.n);
            for (int j = 0; j < this.n; ++j) {
                for (int i = 0; i < this.m; ++i) {
                    this.set(i, j, in.readDouble());
                }
            }
        } else {
            this.ld = BigMatrix.ld(this.n);
            this.A = new DoublePointer((long)this.m * (long)this.ld);
            for (int i = 0; i < this.m; ++i) {
                for (int j = 0; j < this.n; ++j) {
                    this.set(i, j, in.readDouble());
                }
            }
        }
    }

    @Override
    public int nrow() {
        return this.m;
    }

    @Override
    public int ncol() {
        return this.n;
    }

    @Override
    public long size() {
        return (long)this.m * (long)this.n;
    }

    private static long length(DoublePointer A) {
        return A.limit() - A.position();
    }

    private static long bytes(DoublePointer A) {
        return (long)A.sizeof() * (A.limit() - A.position());
    }

    public Layout layout() {
        return Layout.COL_MAJOR;
    }

    public int ld() {
        return this.ld;
    }

    public boolean isSymmetric() {
        return this.uplo != null && this.diag == null;
    }

    public BigMatrix uplo(UPLO uplo) {
        if (this.m != this.n) {
            throw new IllegalArgumentException(String.format("The matrix is not square: %d x %d", this.m, this.n));
        }
        this.uplo = uplo;
        return this;
    }

    public UPLO uplo() {
        return this.uplo;
    }

    public BigMatrix triangular(Diag diag) {
        if (this.m != this.n) {
            throw new IllegalArgumentException(String.format("The matrix is not square: %d x %d", this.m, this.n));
        }
        this.diag = diag;
        return this;
    }

    public Diag triangular() {
        return this.diag;
    }

    @Override
    public BigMatrix copy() {
        BigMatrix matrix;
        if (this.layout() == Layout.COL_MAJOR) {
            DoublePointer pointer = new DoublePointer(BigMatrix.length(this.A));
            DoublePointer.memcpy((Pointer)pointer, (Pointer)this.A, (long)BigMatrix.bytes(this.A));
            matrix = new BigMatrix(this.m, this.n, this.ld, pointer);
        } else {
            matrix = new BigMatrix(this.m, this.n);
            for (int j = 0; j < this.n; ++j) {
                for (int i = 0; i < this.m; ++i) {
                    matrix.set(i, j, this.get(i, j));
                }
            }
        }
        if (this.m == this.n) {
            matrix.uplo(this.uplo);
            matrix.triangular(this.diag);
        }
        return matrix;
    }

    public double[][] toArray() {
        double[][] array = new double[this.m][this.n];
        for (int i = 0; i < this.m; ++i) {
            for (int j = 0; j < this.n; ++j) {
                array[i][j] = this.get(i, j);
            }
        }
        return array;
    }

    public BigMatrix set(BigMatrix b) {
        this.m = b.m;
        this.n = b.n;
        this.diag = b.diag;
        this.uplo = b.uplo;
        if (this.layout() == b.layout()) {
            this.A = b.A;
            this.ld = b.ld;
        } else if (this.layout() == Layout.COL_MAJOR) {
            this.ld = BigMatrix.ld(this.m);
            this.A = new DoublePointer((long)this.ld * (long)this.n);
            for (int j = 0; j < this.n; ++j) {
                for (int i = 0; i < this.m; ++i) {
                    this.set(i, j, this.get(i, j));
                }
            }
        } else {
            this.ld = BigMatrix.ld(this.n);
            this.A = new DoublePointer((long)this.ld * (long)this.m);
            for (int i = 0; i < this.m; ++i) {
                for (int j = 0; j < this.n; ++j) {
                    this.set(i, j, this.get(i, j));
                }
            }
        }
        return this;
    }

    protected long index(int i, int j) {
        return (long)j * (long)this.ld + (long)i;
    }

    @Override
    public double get(int i, int j) {
        return this.A.get(this.index(i, j));
    }

    @Override
    public void set(int i, int j, double x) {
        this.A.put(this.index(i, j), x);
    }

    public BigMatrix get(int[] rows, int[] cols) {
        BigMatrix sub = new BigMatrix(rows.length, cols.length);
        for (int j = 0; j < cols.length; ++j) {
            int col = cols[j];
            if (col < 0) {
                col = this.n + col;
            }
            for (int i = 0; i < rows.length; ++i) {
                int row = rows[i];
                if (row < 0) {
                    row = this.m + row;
                }
                sub.set(i, j, this.get(row, col));
            }
        }
        return sub;
    }

    public double[] row(int i) {
        double[] x = new double[this.n];
        if (i < 0) {
            i = this.m + i;
        }
        for (int j = 0; j < this.n; ++j) {
            x[j] = this.get(i, j);
        }
        return x;
    }

    public double[] col(int j) {
        double[] x = new double[this.m];
        for (int i = 0; i < this.m; ++i) {
            x[i] = this.get(i, j);
        }
        return x;
    }

    public BigMatrix row(int ... rows) {
        BigMatrix x = new BigMatrix(rows.length, this.n);
        for (int i = 0; i < rows.length; ++i) {
            int row = rows[i];
            for (int j = 0; j < this.n; ++j) {
                x.set(i, j, this.get(row, j));
            }
        }
        return x;
    }

    public BigMatrix col(int ... cols) {
        BigMatrix x = new BigMatrix(this.m, cols.length);
        for (int j = 0; j < cols.length; ++j) {
            int col = cols[j];
            for (int i = 0; i < this.m; ++i) {
                x.set(i, j, this.get(i, col));
            }
        }
        return x;
    }

    public BigMatrix submatrix(int i, int j, int k, int l) {
        if (i < 0 || i >= this.m || k < i || k >= this.m || j < 0 || j >= this.n || l < j || l >= this.n) {
            throw new IllegalArgumentException(String.format("Invalid submatrix range (%d:%d, %d:%d) of %d x %d", i, k, j, l, this.m, this.n));
        }
        long offset = this.index(i, j);
        long length = this.index(k, l) - offset + 1L;
        DoublePointer B = this.A.getPointer(offset).limit(length);
        if (this.layout() == Layout.COL_MAJOR) {
            return new BigMatrix(k - i + 1, l - j + 1, this.ld, B);
        }
        return new RowMajor(k - i + 1, l - j + 1, this.ld, B);
    }

    public void fill(double x) {
        if (x == 0.0) {
            DoublePointer.memset((Pointer)this.A, (int)0, (long)BigMatrix.bytes(this.A));
        } else {
            long length = BigMatrix.length(this.A);
            for (long i = 0L; i < length; ++i) {
                this.A.put(i, x);
            }
        }
    }

    public BigMatrix transpose() {
        return this.transpose(true);
    }

    public BigMatrix transpose(boolean share) {
        BigMatrix matrix;
        if (share) {
            matrix = this.layout() == Layout.ROW_MAJOR ? new BigMatrix(this.n, this.m, this.ld, this.A) : new RowMajor(this.n, this.m, this.ld, this.A);
        } else {
            matrix = new BigMatrix(this.n, this.m);
            for (int j = 0; j < this.m; ++j) {
                for (int i = 0; i < this.n; ++i) {
                    matrix.set(i, j, this.get(j, i));
                }
            }
        }
        if (this.m == this.n) {
            matrix.uplo(this.uplo);
            matrix.triangular(this.diag);
        }
        return matrix;
    }

    public boolean equals(Object o) {
        if (!(o instanceof BigMatrix)) {
            return false;
        }
        return this.equals((BigMatrix)o, 1.0E-10);
    }

    public boolean equals(BigMatrix o, double epsilon) {
        if (this.m != o.m || this.n != o.n) {
            return false;
        }
        for (int j = 0; j < this.n; ++j) {
            for (int i = 0; i < this.m; ++i) {
                if (MathEx.isZero(this.get(i, j) - o.get(i, j), epsilon)) continue;
                return false;
            }
        }
        return true;
    }

    public double add(int i, int j, double b) {
        long k = this.index(i, j);
        double y = this.A.get(k) + b;
        this.A.put(k, y);
        return y;
    }

    public double sub(int i, int j, double b) {
        long k = this.index(i, j);
        double y = this.A.get(k) - b;
        this.A.put(k, y);
        return y;
    }

    public double mul(int i, int j, double b) {
        long k = this.index(i, j);
        double y = this.A.get(k) * b;
        this.A.put(k, y);
        return y;
    }

    public double div(int i, int j, double b) {
        long k = this.index(i, j);
        double y = this.A.get(k) / b;
        this.A.put(k, y);
        return y;
    }

    public BigMatrix addDiag(double b) {
        int l = Math.min(this.m, this.n);
        for (int i = 0; i < l; ++i) {
            long k = this.index(i, i);
            this.A.put((long)this.m, this.A.get(k) + b);
        }
        return this;
    }

    public BigMatrix addDiag(double[] b) {
        int l = Math.min(this.m, this.n);
        if (b.length != l) {
            throw new IllegalArgumentException("Invalid diagonal array size: " + b.length);
        }
        for (int i = 0; i < l; ++i) {
            long k = this.index(i, i);
            this.A.put((long)this.m, this.A.get(k) + b[i]);
        }
        return this;
    }

    public BigMatrix add(double b) {
        for (int j = 0; j < this.n; ++j) {
            for (int i = 0; i < this.m; ++i) {
                this.add(i, j, b);
            }
        }
        return this;
    }

    public BigMatrix sub(double b) {
        for (int j = 0; j < this.n; ++j) {
            for (int i = 0; i < this.m; ++i) {
                this.sub(i, j, b);
            }
        }
        return this;
    }

    public BigMatrix mul(double b) {
        for (int j = 0; j < this.n; ++j) {
            for (int i = 0; i < this.m; ++i) {
                this.mul(i, j, b);
            }
        }
        return this;
    }

    public BigMatrix div(double b) {
        for (int j = 0; j < this.n; ++j) {
            for (int i = 0; i < this.m; ++i) {
                this.div(i, j, b);
            }
        }
        return this;
    }

    public BigMatrix add(BigMatrix B) {
        if (this.m != B.m || this.n != B.n) {
            throw new IllegalArgumentException("Matrix is not of same size.");
        }
        for (int j = 0; j < this.n; ++j) {
            for (int i = 0; i < this.m; ++i) {
                this.add(i, j, B.get(i, j));
            }
        }
        return this;
    }

    public BigMatrix sub(BigMatrix B) {
        if (this.m != B.m || this.n != B.n) {
            throw new IllegalArgumentException("Matrix is not of same size.");
        }
        for (int j = 0; j < this.n; ++j) {
            for (int i = 0; i < this.m; ++i) {
                this.sub(i, j, B.get(i, j));
            }
        }
        return this;
    }

    public BigMatrix mul(BigMatrix B) {
        if (this.m != B.m || this.n != B.n) {
            throw new IllegalArgumentException("Matrix is not of same size.");
        }
        for (int j = 0; j < this.n; ++j) {
            for (int i = 0; i < this.m; ++i) {
                this.mul(i, j, B.get(i, j));
            }
        }
        return this;
    }

    public BigMatrix div(BigMatrix B) {
        if (this.m != B.m || this.n != B.n) {
            throw new IllegalArgumentException("Matrix is not of same size.");
        }
        for (int j = 0; j < this.n; ++j) {
            for (int i = 0; i < this.m; ++i) {
                this.div(i, j, B.get(i, j));
            }
        }
        return this;
    }

    public BigMatrix add(double beta, BigMatrix B) {
        if (this.m != B.m || this.n != B.n) {
            throw new IllegalArgumentException("Matrix is not of same size.");
        }
        for (int j = 0; j < this.n; ++j) {
            for (int i = 0; i < this.m; ++i) {
                this.add(i, j, beta * B.get(i, j));
            }
        }
        return this;
    }

    public BigMatrix add(double alpha, BigMatrix A, double beta, BigMatrix B) {
        if (this.m != A.m || this.n != A.n) {
            throw new IllegalArgumentException("Matrix A is not of same size.");
        }
        if (this.m != B.m || this.n != B.n) {
            throw new IllegalArgumentException("Matrix B is not of same size.");
        }
        if (this.layout() == A.layout() && this.layout() == B.layout() && this.ld == A.ld && this.ld == B.ld) {
            long size = BigMatrix.length(this.A);
            for (long i = 0L; i < size; ++i) {
                double a = A.A.get(i);
                double b = B.A.get(i);
                this.A.put(i, alpha * a + beta * b);
            }
        } else {
            for (int j = 0; j < this.n; ++j) {
                for (int i = 0; i < this.m; ++i) {
                    this.set(i, j, alpha * A.get(i, j) + beta * B.get(i, j));
                }
            }
        }
        return this;
    }

    public BigMatrix add(double alpha, double beta, BigMatrix B) {
        if (this.m != B.m || this.n != B.n) {
            throw new IllegalArgumentException("Matrix B is not of same size.");
        }
        if (this.layout() == B.layout() && this.ld == B.ld) {
            long size = (int)BigMatrix.length(this.A);
            for (long i = 0L; i < size; ++i) {
                double a = this.A.get(i);
                double b = B.A.get(i);
                this.A.put(i, alpha * a + beta * b);
            }
        } else {
            for (int j = 0; j < this.n; ++j) {
                for (int i = 0; i < this.m; ++i) {
                    this.set(i, j, alpha * this.get(i, j) + beta * B.get(i, j));
                }
            }
        }
        return this;
    }

    public BigMatrix add2(double alpha, double beta, BigMatrix B) {
        if (this.m != B.m || this.n != B.n) {
            throw new IllegalArgumentException("Matrix B is not of same size.");
        }
        if (this.layout() == B.layout() && this.ld == B.ld) {
            long size = BigMatrix.length(this.A);
            for (long i = 0L; i < size; ++i) {
                double a = this.A.get(i);
                double b = B.A.get(i);
                this.A.put(i, alpha * a + beta * b * b);
            }
        } else {
            for (int j = 0; j < this.n; ++j) {
                for (int i = 0; i < this.m; ++i) {
                    this.set(i, j, alpha * this.get(i, j) + beta * B.get(i, j) * B.get(i, j));
                }
            }
        }
        return this;
    }

    public BigMatrix add(double alpha, double[] x, double[] y) {
        if (this.m != x.length || this.n != y.length) {
            throw new IllegalArgumentException("Matrix is not of same size.");
        }
        if (this.isSymmetric() && x == y) {
            BLAS.engine.syr(this.layout(), this.uplo, this.m, alpha, new DoublePointer(x), 1, this.A, this.ld);
        } else {
            BLAS.engine.ger(this.layout(), this.m, this.n, alpha, new DoublePointer(x), 1, new DoublePointer(x), 1, this.A, this.ld);
        }
        return this;
    }

    public BigMatrix replaceNaN(double x) {
        long length = BigMatrix.length(this.A);
        int i = 0;
        while ((long)i < length) {
            if (Double.isNaN(this.A.get((long)i))) {
                this.A.put((long)i, x);
            }
            ++i;
        }
        return this;
    }

    public double sum() {
        double s = 0.0;
        for (int j = 0; j < this.n; ++j) {
            for (int i = 0; i < this.m; ++i) {
                s += this.get(i, j);
            }
        }
        return s;
    }

    public double norm1() {
        double f = 0.0;
        for (int j = 0; j < this.n; ++j) {
            double s = 0.0;
            for (int i = 0; i < this.m; ++i) {
                s += Math.abs(this.get(i, j));
            }
            f = Math.max(f, s);
        }
        return f;
    }

    public double norm2() {
        return this.svd((boolean)false, (boolean)false).s.get(0L);
    }

    public double norm() {
        return this.norm2();
    }

    public double normInf() {
        double[] f = new double[this.m];
        for (int j = 0; j < this.n; ++j) {
            for (int i = 0; i < this.m; ++i) {
                int n = i;
                f[n] = f[n] + Math.abs(this.get(i, j));
            }
        }
        return MathEx.max(f);
    }

    public double normFro() {
        double f = 0.0;
        for (int j = 0; j < this.n; ++j) {
            for (int i = 0; i < this.m; ++i) {
                f = Math.hypot(f, this.get(i, j));
            }
        }
        return f;
    }

    public double xAx(double[] x) {
        if (this.m != this.n) {
            throw new IllegalArgumentException(String.format("The matrix is not square: %d x %d", this.m, this.n));
        }
        if (this.n != x.length) {
            throw new IllegalArgumentException(String.format("Matrix: %d x %d, Vector: %d", this.m, this.n, x.length));
        }
        double[] Ax = this.mv(x);
        return MathEx.dot(x, Ax);
    }

    public double[] rowSums() {
        double[] x = new double[this.m];
        for (int j = 0; j < this.n; ++j) {
            for (int i = 0; i < this.m; ++i) {
                int n = i;
                x[n] = x[n] + this.get(i, j);
            }
        }
        return x;
    }

    public double[] rowMeans() {
        double[] x = this.rowSums();
        int i = 0;
        while (i < this.m) {
            int n = i++;
            x[n] = x[n] / (double)this.n;
        }
        return x;
    }

    public double[] rowSds() {
        double[] x = new double[this.m];
        double[] x2 = new double[this.m];
        for (int j = 0; j < this.n; ++j) {
            int i = 0;
            while (i < this.m) {
                double a = this.get(i, j);
                int n = i;
                x[n] = x[n] + a;
                int n2 = i++;
                x2[n2] = x2[n2] + a * a;
            }
        }
        for (int i = 0; i < this.m; ++i) {
            double mu = x[i] / (double)this.n;
            double variance = Math.max(x2[i] / (double)this.n - mu * mu, 0.0);
            x[i] = Math.sqrt(variance);
        }
        return x;
    }

    public double[] colSums() {
        double[] x = new double[this.n];
        for (int j = 0; j < this.n; ++j) {
            for (int i = 0; i < this.m; ++i) {
                int n = j;
                x[n] = x[n] + this.get(i, j);
            }
        }
        return x;
    }

    public double[] colMeans() {
        double[] x = new double[this.n];
        int j = 0;
        while (j < this.n) {
            for (int i = 0; i < this.m; ++i) {
                int n = j;
                x[n] = x[n] + this.get(i, j);
            }
            int n = j++;
            x[n] = x[n] / (double)this.m;
        }
        return x;
    }

    public double[] colSds() {
        double[] x = new double[this.n];
        for (int j = 0; j < this.n; ++j) {
            double mu = 0.0;
            double sumsq = 0.0;
            for (int i = 0; i < this.m; ++i) {
                double a = this.get(i, j);
                mu += a;
                sumsq += a * a;
            }
            double variance = Math.max(sumsq / (double)this.m - (mu /= (double)this.m) * mu, 0.0);
            x[j] = Math.sqrt(variance);
        }
        return x;
    }

    public BigMatrix standardize() {
        double[] center = this.colMeans();
        double[] scale = this.colSds();
        return this.scale(center, scale);
    }

    public BigMatrix scale(double[] center, double[] scale) {
        if (center == null && scale == null) {
            throw new IllegalArgumentException("Both center and scale are null");
        }
        BigMatrix matrix = new BigMatrix(this.m, this.n);
        if (center == null) {
            for (int j = 0; j < this.n; ++j) {
                for (int i = 0; i < this.m; ++i) {
                    matrix.set(i, j, this.get(i, j) / scale[j]);
                }
            }
        } else if (scale == null) {
            for (int j = 0; j < this.n; ++j) {
                for (int i = 0; i < this.m; ++i) {
                    matrix.set(i, j, this.get(i, j) - center[j]);
                }
            }
        } else {
            for (int j = 0; j < this.n; ++j) {
                for (int i = 0; i < this.m; ++i) {
                    matrix.set(i, j, (this.get(i, j) - center[j]) / scale[j]);
                }
            }
        }
        return matrix;
    }

    public BigMatrix inverse() {
        if (this.m != this.n) {
            throw new IllegalArgumentException(String.format("The matrix is not square: %d x %d", this.m, this.n));
        }
        try (BigMatrix lu = this.copy();){
            BigMatrix inv = BigMatrix.eye(this.n);
            IntPointer ipiv = new IntPointer((long)this.n);
            if (this.isSymmetric()) {
                info = LAPACK.engine.sysv(lu.layout(), this.uplo, this.n, this.n, lu.A, lu.ld, ipiv, inv.A, inv.ld);
                if (info != 0) {
                    throw new ArithmeticException("SYSV fails: " + info);
                }
            } else {
                info = LAPACK.engine.gesv(lu.layout(), this.n, this.n, lu.A, lu.ld, ipiv, inv.A, inv.ld);
                if (info != 0) {
                    throw new ArithmeticException("GESV fails: " + info);
                }
            }
            BigMatrix bigMatrix = inv;
            return bigMatrix;
        }
    }

    private void mv(Transpose trans, double alpha, DoublePointer x, double beta, DoublePointer y) {
        if (this.uplo != null) {
            if (this.diag != null) {
                if (alpha == 1.0 && beta == 0.0 && x == y) {
                    BLAS.engine.trmv(this.layout(), this.uplo, trans, this.diag, this.m, this.A, this.ld, y, 1);
                } else {
                    BLAS.engine.gemv(this.layout(), trans, this.m, this.n, alpha, this.A, this.ld, x, 1, beta, y, 1);
                }
            } else {
                BLAS.engine.symv(this.layout(), this.uplo, this.m, alpha, this.A, this.ld, x, 1, beta, y, 1);
            }
        } else {
            BLAS.engine.gemv(this.layout(), trans, this.m, this.n, alpha, this.A, this.ld, x, 1, beta, y, 1);
        }
    }

    @Override
    public void mv(Transpose trans, double alpha, double[] x, double beta, double[] y) {
        DoublePointer xp = new DoublePointer(x);
        DoublePointer yp = new DoublePointer(y);
        this.mv(trans, alpha, xp, beta, yp);
        yp.get(y);
    }

    @Override
    public void mv(double[] work, int inputOffset, int outputOffset) {
        try (DoublePointer pointer = new DoublePointer(work);){
            DoublePointer xb = pointer.getPointer((long)inputOffset).limit((long)this.n);
            DoublePointer yb = pointer.getPointer((long)outputOffset).limit((long)this.m);
            this.mv(Transpose.NO_TRANSPOSE, 1.0, xb, 0.0, yb);
            pointer.get(work);
        }
    }

    @Override
    public void tv(double[] work, int inputOffset, int outputOffset) {
        try (DoublePointer pointer = new DoublePointer(work);){
            DoublePointer xb = pointer.getPointer((long)inputOffset).limit((long)this.m);
            DoublePointer yb = pointer.getPointer((long)outputOffset).limit((long)this.n);
            this.mv(Transpose.TRANSPOSE, 1.0, xb, 0.0, yb);
            pointer.get(work);
        }
    }

    public BigMatrix mm(Transpose transA, BigMatrix A, Transpose transB, BigMatrix B) {
        return this.mm(transA, A, transB, B, 1.0, 0.0);
    }

    public BigMatrix mm(Transpose transA, BigMatrix A, Transpose transB, BigMatrix B, double alpha, double beta) {
        if (A.isSymmetric() && transB == Transpose.NO_TRANSPOSE && B.layout() == this.layout()) {
            BLAS.engine.symm(this.layout(), Side.LEFT, A.uplo, this.m, this.n, alpha, A.A, A.ld, B.A, B.ld, beta, this.A, this.ld);
        } else if (B.isSymmetric() && transA == Transpose.NO_TRANSPOSE && A.layout() == this.layout()) {
            BLAS.engine.symm(this.layout(), Side.RIGHT, B.uplo, this.m, this.n, alpha, B.A, B.ld, A.A, A.ld, beta, this.A, this.ld);
        } else {
            if (this.layout() != A.layout()) {
                transA = BigMatrix.flip(transA);
                A = A.transpose();
            }
            if (this.layout() != B.layout()) {
                transB = BigMatrix.flip(transB);
                B = B.transpose();
            }
            int k = transA == Transpose.NO_TRANSPOSE ? A.n : A.m;
            BLAS.engine.gemm(this.layout(), transA, transB, this.m, this.n, k, alpha, A.A, A.ld, B.A, B.ld, beta, this.A, this.ld);
        }
        return this;
    }

    public BigMatrix ata() {
        BigMatrix C = new BigMatrix(this.n, this.n);
        C.mm(Transpose.TRANSPOSE, this, Transpose.NO_TRANSPOSE, this);
        C.uplo(UPLO.LOWER);
        return C;
    }

    public BigMatrix aat() {
        BigMatrix C = new BigMatrix(this.m, this.m);
        C.mm(Transpose.NO_TRANSPOSE, this, Transpose.TRANSPOSE, this);
        C.uplo(UPLO.LOWER);
        return C;
    }

    public static BigMatrix adb(Transpose transA, BigMatrix A, double[] D, Transpose transB, BigMatrix B) {
        BigMatrix AD;
        int m = A.m;
        int n = A.n;
        if (transA == Transpose.NO_TRANSPOSE) {
            AD = new BigMatrix(m, n);
            for (j = 0; j < n; ++j) {
                dj = D[j];
                for (int i = 0; i < m; ++i) {
                    AD.set(i, j, dj * A.get(i, j));
                }
            }
        } else {
            AD = new BigMatrix(n, m);
            for (j = 0; j < m; ++j) {
                dj = D[j];
                for (int i = 0; i < n; ++i) {
                    AD.set(i, j, dj * A.get(j, i));
                }
            }
        }
        try (BigMatrix bigMatrix = AD;){
            BigMatrix bigMatrix2 = transB == Transpose.NO_TRANSPOSE ? AD.mm(B) : AD.mt(B);
            return bigMatrix2;
        }
    }

    public BigMatrix mm(BigMatrix B) {
        if (this.n != B.m) {
            throw new IllegalArgumentException(String.format("Matrix multiplication A * B: %d x %d vs %d x %d", this.m, this.n, B.m, B.n));
        }
        BigMatrix C = new BigMatrix(this.m, B.n);
        C.mm(Transpose.NO_TRANSPOSE, this, Transpose.NO_TRANSPOSE, B);
        return C;
    }

    public BigMatrix mt(BigMatrix B) {
        if (this.n != B.n) {
            throw new IllegalArgumentException(String.format("Matrix multiplication A * B': %d x %d vs %d x %d", this.m, this.n, B.m, B.n));
        }
        BigMatrix C = new BigMatrix(this.m, B.m);
        C.mm(Transpose.NO_TRANSPOSE, this, Transpose.TRANSPOSE, B);
        return C;
    }

    public BigMatrix tm(BigMatrix B) {
        if (this.m != B.m) {
            throw new IllegalArgumentException(String.format("Matrix multiplication A' * B: %d x %d vs %d x %d", this.m, this.n, B.m, B.n));
        }
        BigMatrix C = new BigMatrix(this.n, B.n);
        C.mm(Transpose.TRANSPOSE, this, Transpose.NO_TRANSPOSE, B);
        return C;
    }

    public BigMatrix tt(BigMatrix B) {
        if (this.m != B.n) {
            throw new IllegalArgumentException(String.format("Matrix multiplication A' * B': %d x %d vs %d x %d", this.m, this.n, B.m, B.n));
        }
        BigMatrix C = new BigMatrix(this.n, B.m);
        C.mm(Transpose.TRANSPOSE, this, Transpose.TRANSPOSE, B);
        return C;
    }

    public LU lu() {
        return this.lu(false);
    }

    public LU lu(boolean overwrite) {
        BigMatrix lu = overwrite ? this : this.copy();
        IntPointer ipiv = new IntPointer((long)Math.min(this.m, this.n));
        int info = LAPACK.engine.getrf(lu.layout(), lu.m, lu.n, lu.A, lu.ld, ipiv);
        if (info < 0) {
            logger.error("LAPACK GETRF error code: {}", (Object)info);
            throw new ArithmeticException("LAPACK GETRF error code: " + info);
        }
        lu.uplo = null;
        return new LU(lu, ipiv, info);
    }

    public Cholesky cholesky() {
        return this.cholesky(false);
    }

    public Cholesky cholesky(boolean overwrite) {
        if (this.uplo == null) {
            throw new IllegalArgumentException("The matrix is not symmetric");
        }
        BigMatrix lu = overwrite ? this : this.copy();
        int info = LAPACK.engine.potrf(lu.layout(), lu.uplo, lu.n, lu.A, lu.ld);
        if (info != 0) {
            logger.error("LAPACK POTRF error code: {}", (Object)info);
            throw new ArithmeticException("LAPACK POTRF error code: " + info);
        }
        return new Cholesky(lu);
    }

    public QR qr() {
        return this.qr(false);
    }

    public QR qr(boolean overwrite) {
        BigMatrix qr = overwrite ? this : this.copy();
        DoublePointer tau = new DoublePointer((long)Math.min(this.m, this.n));
        int info = LAPACK.engine.geqrf(qr.layout(), qr.m, qr.n, qr.A, qr.ld, tau);
        if (info != 0) {
            logger.error("LAPACK GEQRF error code: {}", (Object)info);
            throw new ArithmeticException("LAPACK GEQRF error code: " + info);
        }
        qr.uplo = null;
        return new QR(qr, tau);
    }

    public SVD svd() {
        return this.svd(true, false);
    }

    public SVD svd(boolean vectors, boolean overwrite) {
        BigMatrix W;
        int k = Math.min(this.m, this.n);
        DoublePointer s = new DoublePointer((long)k);
        BigMatrix bigMatrix = W = overwrite ? this : this.copy();
        if (vectors) {
            BigMatrix U = new BigMatrix(this.m, k);
            BigMatrix VT = new BigMatrix(k, this.n);
            int info = LAPACK.engine.gesdd(W.layout(), SVDJob.COMPACT, W.m, W.n, W.A, W.ld, s, U.A, U.ld, VT.A, VT.ld);
            if (W != this) {
                W.close();
            }
            if (info != 0) {
                logger.error("LAPACK GESDD with COMPACT error code: {}", (Object)info);
                throw new ArithmeticException("LAPACK GESDD with COMPACT error code: " + info);
            }
            return new SVD(s, U, VT.transpose());
        }
        try (BigMatrix U = new BigMatrix(1, 1);){
            BigMatrix VT = new BigMatrix(1, 1);
            try {
                int info = LAPACK.engine.gesdd(W.layout(), SVDJob.NO_VECTORS, W.m, W.n, W.A, W.ld, s, U.A, U.ld, VT.A, VT.ld);
                if (W != this) {
                    W.close();
                }
                if (info != 0) {
                    logger.error("LAPACK GESDD with NO_VECTORS error code: {}", (Object)info);
                    throw new ArithmeticException("LAPACK GESDD with NO_VECTORS error code: " + info);
                }
                SVD sVD = new SVD(this.m, this.n, s);
                VT.close();
                return sVD;
            }
            catch (Throwable throwable) {
                try {
                    VT.close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
        }
    }

    public EVD eigen() {
        return this.eigen(false, true, false);
    }

    public EVD eigen(boolean vl, boolean vr, boolean overwrite) {
        BigMatrix eig;
        if (this.m != this.n) {
            throw new IllegalArgumentException(String.format("The matrix is not square: %d x %d", this.m, this.n));
        }
        BigMatrix bigMatrix = eig = overwrite ? this : this.copy();
        if (this.isSymmetric()) {
            DoublePointer w = new DoublePointer((long)this.n);
            int info = LAPACK.engine.syevd(eig.layout(), vr ? EVDJob.VECTORS : EVDJob.NO_VECTORS, eig.uplo, this.n, eig.A, eig.ld, w);
            if (eig != this && !vr) {
                eig.close();
            }
            if (info != 0) {
                logger.error("LAPACK SYEV error code: {}", (Object)info);
                throw new ArithmeticException("LAPACK SYEV error code: " + info);
            }
            eig.uplo = null;
            return new EVD(w, vr ? eig : null);
        }
        DoublePointer wr = new DoublePointer((long)this.n);
        DoublePointer wi = new DoublePointer((long)this.n);
        BigMatrix Vl = vl ? new BigMatrix(this.n, this.n) : new BigMatrix(1, 1);
        BigMatrix Vr = vr ? new BigMatrix(this.n, this.n) : new BigMatrix(1, 1);
        int info = LAPACK.engine.geev(eig.layout(), vl ? EVDJob.VECTORS : EVDJob.NO_VECTORS, vr ? EVDJob.VECTORS : EVDJob.NO_VECTORS, this.n, eig.A, eig.ld, wr, wi, Vl.A, Vl.ld, Vr.A, Vr.ld);
        if (eig != this && !vr) {
            eig.close();
        }
        if (info != 0) {
            logger.error("LAPACK GEEV error code: {}", (Object)info);
            throw new ArithmeticException("LAPACK GEEV error code: " + info);
        }
        return new EVD(wr, wi, vl ? Vl : null, vr ? Vr : null);
    }

    private static class RowMajor
    extends BigMatrix {
        RowMajor(int m, int n, int ld, DoublePointer A) {
            super(m, n, ld, A);
        }

        @Override
        public Layout layout() {
            return Layout.ROW_MAJOR;
        }

        @Override
        protected long index(int i, int j) {
            return (long)i * (long)this.ld + (long)j;
        }
    }

    public static class SVD
    implements Serializable {
        private static final long serialVersionUID = 2L;
        public final int m;
        public final int n;
        public final DoublePointer s;
        public final BigMatrix U;
        public final BigMatrix V;

        public SVD(int m, int n, DoublePointer s) {
            this.m = m;
            this.n = n;
            this.s = s;
            this.U = null;
            this.V = null;
        }

        public SVD(DoublePointer s, BigMatrix U, BigMatrix V) {
            this.m = U.m;
            this.n = V.m;
            this.s = s;
            this.U = U;
            this.V = V;
        }

        public BigMatrix diag() {
            BigMatrix S = new BigMatrix(this.m, this.n);
            long length = BigMatrix.length(this.s);
            int i = 0;
            while ((long)i < length) {
                S.set(i, i, this.s.get((long)i));
                ++i;
            }
            return S;
        }

        public double norm() {
            return this.s.get(0L);
        }

        private double rcond() {
            return 0.5 * Math.sqrt(this.m + this.n + 1) * this.s.get(0L) * MathEx.EPSILON;
        }

        public int rank() {
            if (BigMatrix.length(this.s) != (long)Math.min(this.m, this.n)) {
                throw new UnsupportedOperationException("The operation cannot be called on a partial SVD.");
            }
            int r = 0;
            double tol = this.rcond();
            long length = BigMatrix.length(this.s);
            int i = 0;
            while ((long)i < length) {
                double si = this.s.get((long)i);
                if (si > tol) {
                    ++r;
                }
                ++i;
            }
            return r;
        }

        public int nullity() {
            return Math.min(this.m, this.n) - this.rank();
        }

        public double condition() {
            int k = Math.min(this.m, this.n);
            if (BigMatrix.length(this.s) != (long)k) {
                throw new UnsupportedOperationException("The operation cannot be called on a partial SVD.");
            }
            return this.s.get(0L) <= 0.0 || this.s.get((long)(k - 1)) <= 0.0 ? Double.POSITIVE_INFINITY : this.s.get(0L) / this.s.get((long)(k - 1));
        }

        public BigMatrix range() {
            if (BigMatrix.length(this.s) != (long)Math.min(this.m, this.n)) {
                throw new UnsupportedOperationException("The operation cannot be called on a partial SVD.");
            }
            if (this.U == null) {
                throw new IllegalStateException("The left singular vectors are not available.");
            }
            int r = this.rank();
            if (r == 0) {
                return null;
            }
            BigMatrix R = new BigMatrix(this.m, r);
            for (int j = 0; j < r; ++j) {
                for (int i = 0; i < this.m; ++i) {
                    R.set(i, j, this.U.get(i, j));
                }
            }
            return R;
        }

        public BigMatrix nullspace() {
            if (BigMatrix.length(this.s) != (long)Math.min(this.m, this.n)) {
                throw new UnsupportedOperationException("The operation cannot be called on a partial SVD.");
            }
            if (this.V == null) {
                throw new IllegalStateException("The right singular vectors are not available.");
            }
            int nr = this.nullity();
            if (nr == 0) {
                return null;
            }
            BigMatrix N = new BigMatrix(this.n, nr);
            for (int j = 0; j < nr; ++j) {
                for (int i = 0; i < this.n; ++i) {
                    N.set(i, j, this.V.get(i, this.n - j - 1));
                }
            }
            return N;
        }

        public BigMatrix pinv() {
            if (this.U == null || this.V == null) {
                throw new IllegalStateException("The singular vectors are not available.");
            }
            int k = (int)BigMatrix.length(this.s);
            double[] sigma = new double[k];
            int r = this.rank();
            for (int i = 0; i < r; ++i) {
                sigma[i] = 1.0 / this.s.get((long)i);
            }
            return BigMatrix.adb(Transpose.NO_TRANSPOSE, this.V, sigma, Transpose.TRANSPOSE, this.U);
        }

        public double[] solve(double[] b) {
            if (this.U == null || this.V == null) {
                throw new IllegalStateException("The singular vectors are not available.");
            }
            if (b.length != this.m) {
                throw new IllegalArgumentException(String.format("Row dimensions do not agree: A is %d x %d, but B is %d x 1", this.m, this.n, b.length));
            }
            int r = this.rank();
            double[] Utb = new double[(int)BigMatrix.length(this.s)];
            this.U.submatrix(0, 0, this.m - 1, r - 1).tv(b, Utb);
            for (int i = 0; i < r; ++i) {
                int n = i;
                Utb[n] = Utb[n] / this.s.get((long)i);
            }
            return this.V.mv(Utb);
        }
    }

    public static class LU
    implements Serializable {
        private static final long serialVersionUID = 2L;
        public final BigMatrix lu;
        public final IntPointer ipiv;
        public final int info;

        public LU(BigMatrix lu, IntPointer ipiv, int info) {
            this.lu = lu;
            this.ipiv = ipiv;
            this.info = info;
        }

        public boolean isSingular() {
            return this.info > 0;
        }

        public double det() {
            int j;
            int m = this.lu.m;
            int n = this.lu.n;
            if (m != n) {
                throw new IllegalArgumentException(String.format("The matrix is not square: %d x %d", m, n));
            }
            double d = 1.0;
            for (j = 0; j < n; ++j) {
                d *= this.lu.get(j, j);
            }
            for (j = 0; j < n; ++j) {
                if (j + 1 == this.ipiv.get((long)j)) continue;
                d = -d;
            }
            return d;
        }

        public BigMatrix inverse() {
            BigMatrix inv = BigMatrix.eye(this.lu.n);
            this.solve(inv);
            return inv;
        }

        public double[] solve(double[] b) {
            BigMatrix x = BigMatrix.column(b);
            this.solve(x);
            double[] y = new double[b.length];
            x.A.get(y);
            return y;
        }

        public void solve(BigMatrix B) {
            if (this.lu.m != this.lu.n) {
                throw new IllegalArgumentException(String.format("The matrix is not square: %d x %d", this.lu.m, this.lu.n));
            }
            if (B.m != this.lu.m) {
                throw new IllegalArgumentException(String.format("Row dimensions do not agree: A is %d x %d, but B is %d x %d", this.lu.m, this.lu.n, B.m, B.n));
            }
            if (this.lu.layout() != B.layout()) {
                throw new IllegalArgumentException("The matrix layout is inconsistent.");
            }
            if (this.info > 0) {
                throw new RuntimeException("The matrix is singular.");
            }
            int ret = LAPACK.engine.getrs(this.lu.layout(), Transpose.NO_TRANSPOSE, this.lu.n, B.n, this.lu.A, this.lu.ld, this.ipiv, B.A, B.ld);
            if (ret != 0) {
                logger.error("LAPACK GETRS error code: {}", (Object)ret);
                throw new ArithmeticException("LAPACK GETRS error code: " + ret);
            }
        }
    }

    public static class Cholesky
    implements Serializable {
        private static final long serialVersionUID = 2L;
        public final BigMatrix lu;

        public Cholesky(BigMatrix lu) {
            if (lu.nrow() != lu.ncol()) {
                throw new UnsupportedOperationException("Cholesky constructor on a non-square matrix");
            }
            this.lu = lu;
        }

        public double det() {
            int n = this.lu.n;
            double d = 1.0;
            for (int i = 0; i < n; ++i) {
                d *= this.lu.get(i, i);
            }
            return d * d;
        }

        public double logdet() {
            int n = this.lu.n;
            double d = 0.0;
            for (int i = 0; i < n; ++i) {
                d += Math.log(this.lu.get(i, i));
            }
            return 2.0 * d;
        }

        public BigMatrix inverse() {
            BigMatrix inv = BigMatrix.eye(this.lu.n);
            this.solve(inv);
            return inv;
        }

        public double[] solve(double[] b) {
            BigMatrix x = BigMatrix.column(b);
            this.solve(x);
            double[] y = new double[b.length];
            x.A.get(y);
            return y;
        }

        public void solve(BigMatrix B) {
            if (B.m != this.lu.m) {
                throw new IllegalArgumentException(String.format("Row dimensions do not agree: A is %d x %d, but B is %d x %d", this.lu.m, this.lu.n, B.m, B.n));
            }
            int info = LAPACK.engine.potrs(this.lu.layout(), this.lu.uplo, this.lu.n, B.n, this.lu.A, this.lu.ld, B.A, B.ld);
            if (info != 0) {
                logger.error("LAPACK POTRS error code: {}", (Object)info);
                throw new ArithmeticException("LAPACK POTRS error code: " + info);
            }
        }
    }

    public static class QR
    implements Serializable {
        private static final long serialVersionUID = 2L;
        public final BigMatrix qr;
        public final DoublePointer tau;

        public QR(BigMatrix qr, DoublePointer tau) {
            this.qr = qr;
            this.tau = tau;
        }

        public Cholesky CholeskyOfAtA() {
            int n = this.qr.n;
            BigMatrix L = new BigMatrix(n, n);
            for (int i = 0; i < n; ++i) {
                for (int j = 0; j <= i; ++j) {
                    L.set(i, j, this.qr.get(j, i));
                }
            }
            L.uplo(UPLO.LOWER);
            return new Cholesky(L);
        }

        public BigMatrix R() {
            int n = this.qr.n;
            BigMatrix R = BigMatrix.diag(this.tau);
            for (int i = 0; i < n; ++i) {
                for (int j = i; j < n; ++j) {
                    R.set(i, j, this.qr.get(i, j));
                }
            }
            return R;
        }

        public BigMatrix Q() {
            int m = this.qr.m;
            int n = this.qr.n;
            int k = Math.min(m, n);
            BigMatrix Q = this.qr.copy();
            int info = LAPACK.engine.orgqr(this.qr.layout(), m, n, k, Q.A, this.qr.ld, this.tau);
            if (info != 0) {
                logger.error("LAPACK ORGRQ error code: {}", (Object)info);
                throw new ArithmeticException("LAPACK ORGRQ error code: " + info);
            }
            return Q;
        }

        public double[] solve(double[] b) {
            if (b.length != this.qr.m) {
                throw new IllegalArgumentException(String.format("Row dimensions do not agree: A is %d x %d, but B is %d x 1", this.qr.m, this.qr.n, b.length));
            }
            BigMatrix x = BigMatrix.column(b);
            this.solve(x);
            double[] y = new double[this.qr.n];
            x.A.get(y);
            return y;
        }

        public void solve(BigMatrix B) {
            if (B.m != this.qr.m) {
                throw new IllegalArgumentException(String.format("Row dimensions do not agree: A is %d x %d, but B is %d x %d", this.qr.nrow(), this.qr.nrow(), B.nrow(), B.ncol()));
            }
            int m = this.qr.m;
            int n = this.qr.n;
            int k = Math.min(m, n);
            int info = LAPACK.engine.ormqr(this.qr.layout(), Side.LEFT, Transpose.TRANSPOSE, B.nrow(), B.ncol(), k, this.qr.A, this.qr.ld, this.tau, B.A, B.ld);
            if (info != 0) {
                logger.error("LAPACK ORMQR error code: {}", (Object)info);
                throw new IllegalArgumentException("LAPACK ORMQR error code: " + info);
            }
            info = LAPACK.engine.trtrs(this.qr.layout(), UPLO.UPPER, Transpose.NO_TRANSPOSE, Diag.NON_UNIT, this.qr.n, B.n, this.qr.A, this.qr.ld, B.A, B.ld);
            if (info != 0) {
                logger.error("LAPACK TRTRS error code: {}", (Object)info);
                throw new IllegalArgumentException("LAPACK TRTRS error code: " + info);
            }
        }
    }

    public static class EVD
    implements Serializable {
        private static final long serialVersionUID = 2L;
        public final DoublePointer wr;
        public final DoublePointer wi;
        public final BigMatrix Vl;
        public final BigMatrix Vr;

        public EVD(DoublePointer w, BigMatrix V) {
            this.wr = w;
            this.wi = null;
            this.Vl = V;
            this.Vr = V;
        }

        public EVD(DoublePointer wr, DoublePointer wi, BigMatrix Vl, BigMatrix Vr) {
            this.wr = wr;
            this.wi = wi;
            this.Vl = Vl;
            this.Vr = Vr;
        }

        public BigMatrix diag() {
            BigMatrix D = BigMatrix.diag(this.wr);
            if (this.wi != null) {
                int n = (int)BigMatrix.length(this.wr);
                for (int i = 0; i < n; ++i) {
                    if (this.wi.get((long)i) > 0.0) {
                        D.set(i, i + 1, this.wi.get((long)i));
                        continue;
                    }
                    if (!(this.wi.get((long)i) < 0.0)) continue;
                    D.set(i, i - 1, this.wi.get((long)i));
                }
            }
            return D;
        }

        public EVD sort() {
            int n = (int)BigMatrix.length(this.wr);
            double[] w = new double[n];
            if (this.wi != null) {
                for (i = 0; i < n; ++i) {
                    w[i] = -(this.wr.get((long)i) * this.wr.get((long)i) + this.wi.get((long)i) * this.wi.get((long)i));
                }
            } else {
                for (i = 0; i < n; ++i) {
                    w[i] = -(this.wr.get((long)i) * this.wr.get((long)i));
                }
            }
            int[] index = QuickSort.sort(w);
            DoublePointer wr2 = new DoublePointer((long)n);
            for (int j = 0; j < n; ++j) {
                wr2.put((long)j, this.wr.get((long)index[j]));
            }
            DoublePointer wi2 = null;
            if (this.wi != null) {
                wi2 = new DoublePointer((long)n);
                for (int j = 0; j < n; ++j) {
                    wi2.put((long)j, this.wi.get((long)index[j]));
                }
            }
            BigMatrix Vl2 = null;
            if (this.Vl != null) {
                int m = this.Vl.m;
                Vl2 = new BigMatrix(m, n);
                for (int j = 0; j < n; ++j) {
                    for (int i = 0; i < m; ++i) {
                        Vl2.set(i, j, this.Vl.get(i, index[j]));
                    }
                }
            }
            BigMatrix Vr2 = null;
            if (this.Vr != null) {
                int m = this.Vr.m;
                Vr2 = new BigMatrix(m, n);
                for (int j = 0; j < n; ++j) {
                    for (int i = 0; i < m; ++i) {
                        Vr2.set(i, j, this.Vr.get(i, index[j]));
                    }
                }
            }
            return new EVD(wr2, wi2, Vl2, Vr2);
        }
    }
}

