/*
 * Decompiled with CFR 0.152.
 */
package com.github.alexdlaird.ngrok.process;

import com.github.alexdlaird.exception.JavaNgrokSecurityException;
import com.github.alexdlaird.exception.NgrokException;
import com.github.alexdlaird.http.DefaultHttpClient;
import com.github.alexdlaird.http.HttpClient;
import com.github.alexdlaird.http.Response;
import com.github.alexdlaird.ngrok.conf.JavaNgrokConfig;
import com.github.alexdlaird.ngrok.installer.NgrokInstaller;
import com.github.alexdlaird.ngrok.installer.NgrokVersion;
import com.github.alexdlaird.ngrok.process.NgrokLog;
import com.github.alexdlaird.ngrok.protocol.Tunnels;
import com.github.alexdlaird.util.StringUtils;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.logging.Level;
import java.util.logging.Logger;

public class NgrokProcess {
    private static final Logger LOGGER = Logger.getLogger(String.valueOf(NgrokProcess.class));
    private final JavaNgrokConfig javaNgrokConfig;
    private final NgrokInstaller ngrokInstaller;
    private Process process;
    private ProcessMonitor processMonitor;

    public NgrokProcess(JavaNgrokConfig javaNgrokConfig, NgrokInstaller ngrokInstaller) {
        this.javaNgrokConfig = javaNgrokConfig;
        this.ngrokInstaller = ngrokInstaller;
        if (!Files.exists(javaNgrokConfig.getNgrokPath(), new LinkOption[0])) {
            ngrokInstaller.installNgrok(javaNgrokConfig.getNgrokPath(), javaNgrokConfig.getNgrokVersion());
        }
        if (!Files.exists(javaNgrokConfig.getConfigPath(), new LinkOption[0])) {
            ngrokInstaller.installDefaultConfig(javaNgrokConfig.getConfigPath(), Collections.emptyMap(), javaNgrokConfig.getNgrokVersion());
        }
    }

    public void start() {
        if (this.isRunning()) {
            return;
        }
        if (!Files.exists(this.javaNgrokConfig.getNgrokPath(), new LinkOption[0])) {
            throw new NgrokException(String.format("ngrok binary was not found. Be sure to call \"NgrokInstaller.installNgrok()\" first for \"ngrokPath\": %s", this.javaNgrokConfig.getNgrokPath()));
        }
        this.ngrokInstaller.validateConfig(this.javaNgrokConfig.getConfigPath());
        ProcessBuilder processBuilder = new ProcessBuilder(new String[0]);
        processBuilder.redirectErrorStream(true);
        processBuilder.inheritIO().redirectOutput(ProcessBuilder.Redirect.PIPE);
        ArrayList<String> command = new ArrayList<String>();
        command.add(this.javaNgrokConfig.getNgrokPath().toString());
        command.add("start");
        command.add("--none");
        command.add("--log=stdout");
        if (Objects.nonNull(this.javaNgrokConfig.getConfigPath())) {
            LOGGER.info(String.format("Starting ngrok with config file: %s", this.javaNgrokConfig.getConfigPath()));
            command.add(String.format("--config=%s", this.javaNgrokConfig.getConfigPath().toString()));
        }
        if (Objects.nonNull(this.javaNgrokConfig.getAuthToken())) {
            LOGGER.info("Overriding default auth token");
            command.add(String.format("--authtoken=%s", this.javaNgrokConfig.getAuthToken()));
        }
        if (Objects.nonNull((Object)this.javaNgrokConfig.getRegion())) {
            LOGGER.info(String.format("Starting ngrok in region: %s", new Object[]{this.javaNgrokConfig.getRegion()}));
            command.add(String.format("--region=%s", new Object[]{this.javaNgrokConfig.getRegion()}));
        }
        processBuilder.command(command);
        try {
            this.process = processBuilder.start();
            Runtime.getRuntime().addShutdownHook(new Thread(this::stop));
            LOGGER.fine(String.format("ngrok process starting with PID: %s", this.process.pid()));
            this.processMonitor = new ProcessMonitor(this.process, this.javaNgrokConfig);
            new Thread(this.processMonitor).start();
            Calendar timeout = Calendar.getInstance();
            timeout.add(13, this.javaNgrokConfig.getStartupTime());
            while (Calendar.getInstance().before(timeout)) {
                if (this.processMonitor.isHealthy()) {
                    LOGGER.info(String.format("ngrok process has started with API URL: %s", this.processMonitor.apiUrl));
                    this.processMonitor.startupError = null;
                    break;
                }
                if (this.process.isAlive()) continue;
            }
            if (!this.processMonitor.isHealthy()) {
                this.stop();
                if (Objects.nonNull(this.processMonitor.startupError)) {
                    throw new NgrokException(String.format("The ngrok process errored on start: %s.", this.processMonitor.startupError), this.processMonitor.logs, this.processMonitor.startupError);
                }
                throw new NgrokException("The ngrok process was unable to start.", this.processMonitor.logs);
            }
        }
        catch (IOException e) {
            throw new NgrokException("An error occurred while starting ngrok.", e);
        }
    }

    public boolean isRunning() {
        return Objects.nonNull(this.process) && this.process.isAlive();
    }

    public void stop() {
        if (!this.isRunning()) {
            LOGGER.info(String.format("\"ngrokPath\" %s is not running a process", this.javaNgrokConfig.getNgrokPath()));
            return;
        }
        LOGGER.info(String.format("Killing ngrok process: %s", this.process.pid()));
        this.processMonitor.stop();
        this.process.descendants().forEach(ProcessHandle::destroy);
        this.process.destroy();
        this.process = null;
    }

    public void setAuthToken(String authToken) {
        ProcessBuilder processBuilder = new ProcessBuilder(new String[0]);
        processBuilder.redirectErrorStream(true);
        processBuilder.inheritIO().redirectOutput(ProcessBuilder.Redirect.PIPE);
        ArrayList<String> command = new ArrayList<String>();
        command.add(this.javaNgrokConfig.getNgrokPath().toString());
        if (this.javaNgrokConfig.getNgrokVersion() == NgrokVersion.V2) {
            command.add("authtoken");
            command.add(authToken);
        } else {
            command.add("config");
            command.add("add-authtoken");
            command.add(authToken);
        }
        command.add("--log=stdout");
        if (Objects.nonNull(this.javaNgrokConfig.getConfigPath())) {
            command.add(String.format("--config=%s", this.javaNgrokConfig.getConfigPath().toString()));
        }
        LOGGER.info(String.format("Updating authtoken for \"configPath\": %s", this.javaNgrokConfig.getConfigPath()));
        processBuilder.command(command);
        try {
            Process process = processBuilder.start();
            process.waitFor();
            BufferedReader reader = new BufferedReader(new InputStreamReader(process.getInputStream()));
            String result = this.captureOutput(reader);
            if (!result.contains("Authtoken saved")) {
                throw new NgrokException(String.format("An error occurred while setting the auth token: %s", result));
            }
        }
        catch (IOException | InterruptedException e) {
            throw new NgrokException("An error occurred while setting the auth token for ngrok.", e);
        }
    }

    private String captureOutput(BufferedReader reader) throws IOException {
        String line;
        StringBuilder builder = new StringBuilder();
        while ((line = reader.readLine()) != null) {
            builder.append(line).append("\n");
        }
        return builder.toString().trim();
    }

    public void update() {
        ProcessBuilder processBuilder = new ProcessBuilder(new String[0]);
        processBuilder.redirectErrorStream(true);
        processBuilder.inheritIO().redirectOutput(ProcessBuilder.Redirect.PIPE);
        List<String> command = List.of(this.javaNgrokConfig.getNgrokPath().toString(), "update", "--log=stdout");
        processBuilder.command(command);
        try {
            Process process = processBuilder.start();
            process.waitFor();
        }
        catch (IOException | InterruptedException e) {
            throw new NgrokException("An error occurred while trying to update ngrok.", e);
        }
    }

    public String getVersion() {
        ProcessBuilder processBuilder = new ProcessBuilder(new String[0]);
        processBuilder.redirectErrorStream(true);
        processBuilder.inheritIO().redirectOutput(ProcessBuilder.Redirect.PIPE);
        List<String> command = List.of(this.javaNgrokConfig.getNgrokPath().toString(), "--version");
        processBuilder.command(command);
        try {
            Process process = processBuilder.start();
            process.waitFor();
            BufferedReader reader = new BufferedReader(new InputStreamReader(process.getInputStream()));
            String result = this.captureOutput(reader);
            return result.split("version ")[1];
        }
        catch (IOException | ArrayIndexOutOfBoundsException | InterruptedException e) {
            throw new NgrokException("An error occurred while trying to update ngrok.", e);
        }
    }

    public String getApiUrl() {
        if (!this.isRunning() || !this.processMonitor.isHealthy()) {
            return null;
        }
        return this.processMonitor.apiUrl;
    }

    public NgrokInstaller getNgrokInstaller() {
        return this.ngrokInstaller;
    }

    public ProcessMonitor getProcessMonitor() {
        return this.processMonitor;
    }

    public static class ProcessMonitor
    implements Runnable {
        private final Process process;
        private final JavaNgrokConfig javaNgrokConfig;
        private final HttpClient httpClient;
        private String apiUrl;
        private boolean tunnelStarted;
        private boolean clientConnected;
        private String startupError;
        private final List<NgrokLog> logs = new ArrayList<NgrokLog>();
        private boolean alive = true;

        public ProcessMonitor(Process process, JavaNgrokConfig javaNgrokConfig) {
            this(process, javaNgrokConfig, new DefaultHttpClient.Builder().build());
        }

        protected ProcessMonitor(Process process, JavaNgrokConfig javaNgrokConfig, HttpClient httpClient) {
            this.process = process;
            this.javaNgrokConfig = javaNgrokConfig;
            this.httpClient = httpClient;
        }

        @Override
        public void run() {
            try {
                String line;
                BufferedReader reader = new BufferedReader(new InputStreamReader(this.process.getInputStream()));
                while ((line = reader.readLine()) != null) {
                    this.logStartupLine(line);
                    if (this.isHealthy()) break;
                    if (!Objects.nonNull(this.startupError)) continue;
                    this.alive = false;
                    break;
                }
                while (this.alive && this.process.isAlive() && this.javaNgrokConfig.isKeepMonitoring() && (line = reader.readLine()) != null) {
                    this.logLine(line);
                }
                this.alive = false;
            }
            catch (IOException e) {
                throw new NgrokException("An error occurred in the ngrok process.", e);
            }
        }

        public List<NgrokLog> getLogs() {
            return List.of(this.logs.toArray(new NgrokLog[0]));
        }

        public boolean isMonitoring() {
            return this.alive;
        }

        private void stop() {
            this.alive = false;
        }

        private boolean isHealthy() {
            if (Objects.isNull(this.apiUrl) || !this.tunnelStarted || !this.clientConnected) {
                return false;
            }
            if (!this.apiUrl.toLowerCase().startsWith("http")) {
                throw new JavaNgrokSecurityException(String.format("URL must start with \"http\": %s", this.apiUrl));
            }
            Response<Tunnels> tunnelsResponse = this.httpClient.get(String.format("%s/api/tunnels", this.apiUrl), Tunnels.class);
            if (tunnelsResponse.getStatusCode() != 200) {
                return false;
            }
            return this.process.isAlive();
        }

        private void logStartupLine(String line) {
            NgrokLog ngrokLog = this.logLine(line);
            if (Objects.isNull(ngrokLog)) {
                return;
            }
            if (Objects.nonNull(ngrokLog.getLvl()) && ngrokLog.getLvl().equals(Level.SEVERE.getName())) {
                this.startupError = ngrokLog.getErr();
            } else if (Objects.nonNull(ngrokLog.getMsg())) {
                if (ngrokLog.getMsg().contains("starting web service") && Objects.nonNull(ngrokLog.getAddr())) {
                    this.apiUrl = String.format("http://%s", ngrokLog.getAddr());
                } else if (ngrokLog.getMsg().contains("tunnel session started")) {
                    this.tunnelStarted = true;
                } else if (ngrokLog.getMsg().contains("client session established")) {
                    this.clientConnected = true;
                }
            }
        }

        private NgrokLog logLine(String line) {
            NgrokLog ngrokLog = new NgrokLog(line);
            if (StringUtils.isBlank(ngrokLog.getLine())) {
                return null;
            }
            LOGGER.log(Level.parse(ngrokLog.getLvl()), ngrokLog.getLine());
            this.logs.add(ngrokLog);
            if (this.logs.size() > this.javaNgrokConfig.getMaxLogs()) {
                this.logs.remove(0);
            }
            if (Objects.nonNull(this.javaNgrokConfig.getLogEventCallback())) {
                this.javaNgrokConfig.getLogEventCallback().apply(ngrokLog);
            }
            return ngrokLog;
        }
    }
}

