/*
 * Decompiled with CFR 0.152.
 */
package com.github.alexdlaird.ngrok.installer;

import com.github.alexdlaird.exception.JavaNgrokException;
import com.github.alexdlaird.exception.JavaNgrokInstallerException;
import com.github.alexdlaird.exception.JavaNgrokSecurityException;
import com.github.alexdlaird.ngrok.installer.NgrokCDNUrl;
import com.github.alexdlaird.ngrok.installer.NgrokV2CDNUrl;
import com.github.alexdlaird.ngrok.installer.NgrokV3CDNUrl;
import com.github.alexdlaird.ngrok.installer.NgrokVersion;
import com.github.alexdlaird.util.StringUtils;
import com.google.gson.JsonParseException;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.StringWriter;
import java.io.Writer;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.PosixFileAttributes;
import java.nio.file.attribute.PosixFilePermission;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.logging.Logger;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import org.yaml.snakeyaml.Yaml;

public class NgrokInstaller {
    private static final Logger LOGGER = Logger.getLogger(String.valueOf(NgrokInstaller.class));
    public static final String MAC = "DARWIN";
    public static final String WINDOWS = "WINDOWS";
    public static final String LINUX = "LINUX";
    public static final String FREEBSD = "FREEBSD";
    public static final List<String> UNIX_BINARIES = List.of("DARWIN", "LINUX", "FREEBSD");
    public static final Path DEFAULT_NGROK_PATH = Paths.get(System.getProperty("user.home"), ".ngrok2", NgrokInstaller.getNgrokBin());
    public static final Path DEFAULT_CONFIG_PATH = Paths.get(System.getProperty("user.home"), ".ngrok2", "ngrok.yml");
    private static final List<String> VALID_LOG_LEVELS = List.of("info", "debug");
    private final Yaml yaml = new Yaml();
    private final Map<String, Map<String, Object>> configCache = new HashMap<String, Map<String, Object>>();

    public static String getNgrokBin() {
        String system = NgrokInstaller.getSystem();
        if (UNIX_BINARIES.contains(system)) {
            return "ngrok";
        }
        return "ngrok.exe";
    }

    public void installDefaultConfig(Path configPath, Map<String, Object> data) {
        this.installDefaultConfig(configPath, data, NgrokVersion.V2);
    }

    public void installDefaultConfig(Path configPath, Map<String, Object> data, NgrokVersion ngrokVersion) {
        try {
            Files.createDirectories(configPath.getParent(), new FileAttribute[0]);
            if (!Files.exists(configPath, new LinkOption[0])) {
                Files.createFile(configPath, new FileAttribute[0]);
            }
            Map<String, Object> config = this.getNgrokConfig(configPath, false, ngrokVersion);
            config.putAll(this.getDefaultConfig(ngrokVersion));
            config.putAll(data);
            this.validateConfig(config);
            LOGGER.fine(String.format("Installing default config to %s ...", configPath));
            FileOutputStream out = new FileOutputStream(configPath.toFile());
            StringWriter writer = new StringWriter();
            this.yaml.dump(config, (Writer)writer);
            out.write(writer.toString().getBytes());
            out.close();
        }
        catch (IOException e) {
            throw new JavaNgrokInstallerException(String.format("An error while installing the default ngrok config to %s.", configPath), e);
        }
    }

    public void installNgrok(Path ngrokPath) {
        this.installNgrok(ngrokPath, NgrokVersion.V2);
    }

    public void installNgrok(Path ngrokPath, NgrokVersion ngrokVersion) {
        NgrokCDNUrl ngrokCDNUrl = this.getNgrokCDNUrl(ngrokVersion);
        LOGGER.fine(String.format("Installing ngrok %s to %s%s ...", new Object[]{ngrokVersion, ngrokPath, Files.exists(ngrokPath, new LinkOption[0]) ? ", overwriting" : ""}));
        Path ngrokZip = Paths.get(ngrokPath.getParent().toString(), "ngrok.zip");
        this.downloadFile(ngrokCDNUrl.getUrl(), ngrokZip);
        this.installNgrokZip(ngrokZip, ngrokPath);
    }

    public NgrokCDNUrl getNgrokCDNUrl() {
        return this.getNgrokCDNUrl(NgrokVersion.V3);
    }

    public NgrokCDNUrl getNgrokCDNUrl(NgrokVersion ngrokVersion) {
        String arch = this.getArch();
        String system = NgrokInstaller.getSystem();
        String plat = String.format("%s_%s", system, arch);
        LOGGER.fine(String.format("Platform to download: %s", plat));
        if (ngrokVersion == NgrokVersion.V2) {
            return NgrokV2CDNUrl.valueOf(plat);
        }
        return NgrokV3CDNUrl.valueOf(plat);
    }

    public void validateConfig(Path configPath) {
        Map<String, Object> config = this.getNgrokConfig(configPath);
        this.validateConfig(config);
    }

    public void validateConfig(Map<String, Object> data) {
        if (data.getOrDefault("web_addr", "127.0.0.1:4040").equals("false")) {
            throw new JavaNgrokException("\"web_addr\" cannot be false, as the ngrok API is a dependency for java-ngrok");
        }
        if (data.getOrDefault("log_format", "term").equals("json")) {
            throw new JavaNgrokException("\"log_format\" must be \"term\" to be compatible with java-ngrok");
        }
        if (!VALID_LOG_LEVELS.contains((String)data.getOrDefault("log_level", "info"))) {
            throw new JavaNgrokException("\"log_level\" must be \"info\" to be compatible with java-ngrok");
        }
    }

    public static String getSystem() {
        String os = System.getProperty("os.name").replaceAll(" ", "").toLowerCase();
        if (os.startsWith("mac")) {
            return MAC;
        }
        if (os.startsWith("windows") || os.contains("cygwin")) {
            return WINDOWS;
        }
        if (os.startsWith("linux")) {
            return LINUX;
        }
        if (os.startsWith("freebsd")) {
            return FREEBSD;
        }
        throw new JavaNgrokInstallerException(String.format("Unknown os.name: %s", os));
    }

    public Map<String, Object> getNgrokConfig(Path configPath, boolean useCache, NgrokVersion ngrokVersion) {
        String key = configPath.toString();
        if (!this.configCache.containsKey(key) || !useCache) {
            try {
                String config = Files.readString(configPath);
                if (StringUtils.isBlank(config)) {
                    this.configCache.put(key, this.getDefaultConfig(ngrokVersion));
                } else {
                    this.configCache.put(key, (Map)this.yaml.load(config));
                }
            }
            catch (JsonParseException | IOException e) {
                throw new JavaNgrokInstallerException(String.format("An error occurred while parsing the config file: %s", configPath), e);
            }
        }
        return this.configCache.get(key);
    }

    public Map<String, Object> getDefaultConfig(NgrokVersion ngrokVersion) {
        if (ngrokVersion == NgrokVersion.V2) {
            return new HashMap<String, Object>();
        }
        HashMap<String, Object> config = new HashMap<String, Object>();
        config.put("version", "2");
        config.put("region", "us");
        return config;
    }

    public Map<String, Object> getNgrokConfig(Path configPath) {
        return this.getNgrokConfig(configPath, true);
    }

    public Map<String, Object> getNgrokConfig(Path configPath, boolean useCache) {
        return this.getNgrokConfig(configPath, useCache, NgrokVersion.V2);
    }

    private void installNgrokZip(Path zipPath, Path ngrokPath) {
        try {
            ZipEntry zipEntry;
            Path dir = ngrokPath.getParent();
            LOGGER.fine(String.format("Extracting ngrok binary from %s to %s ...", zipPath, ngrokPath));
            Files.createDirectories(dir, new FileAttribute[0]);
            byte[] buffer = new byte[1024];
            ZipInputStream in = new ZipInputStream(new FileInputStream(zipPath.toFile()));
            while ((zipEntry = in.getNextEntry()) != null) {
                int len;
                Path file = Paths.get(dir.toString(), zipEntry.getName());
                if (!file.normalize().startsWith(dir)) {
                    throw new JavaNgrokSecurityException("Bad zip entry, paths don't match");
                }
                if (zipEntry.isDirectory()) {
                    if (Files.isDirectory(file, new LinkOption[0])) continue;
                    Files.createDirectories(file, new FileAttribute[0]);
                    continue;
                }
                Path parent = file.getParent();
                if (!Files.isDirectory(parent, new LinkOption[0])) {
                    Files.createDirectories(parent, new FileAttribute[0]);
                }
                FileOutputStream out = new FileOutputStream(file.toFile());
                while ((len = in.read(buffer)) > 0) {
                    out.write(buffer, 0, len);
                }
                out.close();
            }
            in.closeEntry();
            in.close();
            if (ngrokPath.getFileSystem().supportedFileAttributeViews().contains("posix")) {
                Set<PosixFilePermission> perms = Files.readAttributes(ngrokPath, PosixFileAttributes.class, new LinkOption[0]).permissions();
                perms.add(PosixFilePermission.OWNER_EXECUTE);
                perms.add(PosixFilePermission.GROUP_EXECUTE);
                perms.add(PosixFilePermission.OTHERS_EXECUTE);
                Files.setPosixFilePermissions(ngrokPath, perms);
            }
        }
        catch (IOException e) {
            throw new JavaNgrokInstallerException("An error occurred while unzipping ngrok.", e);
        }
    }

    private void downloadFile(String url, Path dest) {
        try {
            Files.createDirectories(dest.getParent(), new FileAttribute[0]);
            LOGGER.fine(String.format("Download ngrok from %s ...", url));
            InputStream in = new URL(url).openStream();
            Files.copy(in, dest, StandardCopyOption.REPLACE_EXISTING);
        }
        catch (IOException e) {
            throw new JavaNgrokInstallerException(String.format("An error occurred while downloading the file from %s.", url), e);
        }
    }

    private String getArch() {
        String archProperty = System.getProperty("os.arch");
        StringBuilder arch = new StringBuilder();
        if (archProperty.contains("x86_64")) {
            arch.append("x86_64");
        } else {
            arch.append("i386");
        }
        if (archProperty.startsWith("arm") || archProperty.startsWith("aarch64")) {
            arch.append("_arm");
        }
        return arch.toString();
    }
}

