/*
 * Decompiled with CFR 0.152.
 */
package com.extollit.time;

import com.extollit.collect.RingBuffer;
import com.extollit.temporal.Duration;
import java.io.IOException;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.Calendar;
import java.util.Collections;
import java.util.Date;
import java.util.Iterator;
import java.util.Properties;
import org.apache.commons.net.ntp.NTPUDPClient;
import org.apache.commons.net.ntp.TimeInfo;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TimeSystem {
    private static final Logger logger = LoggerFactory.getLogger((String)"Time System");
    private static final TimeSystem INSTANCE;
    private static final int CONFIGURED_OFFSET;
    private final NTPClientBackground ntp;
    private volatile long offset;

    private TimeSystem(int period, String ... ntpHosts) {
        this.ntp = new NTPClientBackground(period, ntpHosts);
    }

    public static void start() {
        if (INSTANCE != null) {
            logger.info("Time system starting up with hosts {}", (Object)TimeSystem.INSTANCE.ntp.hostNames);
            TimeSystem.INSTANCE.ntp.start();
        } else {
            logger.error("Cannot start NTP background worker, failed to initialize");
        }
    }

    public static void stop() {
        if (INSTANCE != null) {
            TimeSystem.INSTANCE.ntp.stopIt();
        }
    }

    private static int configuredOffset() {
        String property = System.getProperty(TimeSystem.class.getPackage().getName() + ".offset");
        if (property == null) {
            return 0;
        }
        try {
            return Integer.parseInt(property);
        }
        catch (NumberFormatException e) {
            logger.error("Wrong format for time offset: {}", (Object)property);
            return 0;
        }
    }

    private static TimeSystem acquire() throws IOException {
        Properties props = new Properties();
        props.load(Thread.currentThread().getContextClassLoader().getResourceAsStream("ntp.properties"));
        String template = "%s.ntp.%s";
        String namespace = TimeSystem.class.getPackage().getName();
        String[] hostNames = props.getProperty(String.format("%s.ntp.%s", namespace, "hosts"), "").split("\\s*,\\s*");
        int period = 60;
        try {
            period = Integer.parseInt(props.getProperty(String.format("%s.ntp.%s", namespace, "period"), "60"));
        }
        catch (NumberFormatException e) {
            logger.error("Invalid numeric format for period ({}), using default of one minute instead", (Object)e.getMessage());
        }
        return new TimeSystem(period, hostNames);
    }

    private static long getOffset() {
        return INSTANCE == null ? (long)CONFIGURED_OFFSET : TimeSystem.INSTANCE.offset;
    }

    public static long currentTimeMillis() {
        return System.currentTimeMillis() + TimeSystem.getOffset();
    }

    public static Calendar createCalendar() {
        Calendar calendar = Calendar.getInstance();
        calendar.add(14, (int)TimeSystem.getOffset());
        return calendar;
    }

    public static Date addDate(Duration duration) {
        return new Date(TimeSystem.currentTimeMillis() + duration.millis());
    }

    public static Date createDate(long millis) {
        return new Date(millis + TimeSystem.getOffset());
    }

    public static Date createDate() {
        return new Date(TimeSystem.currentTimeMillis());
    }

    static {
        CONFIGURED_OFFSET = TimeSystem.configuredOffset();
        TimeSystem instance = null;
        try {
            instance = TimeSystem.acquire();
        }
        catch (IOException e) {
            logger.error("Could not initialize time system, will use configured fallback", (Throwable)e);
        }
        INSTANCE = instance;
    }

    private final class NTPClientBackground
    extends Thread {
        private final RingBuffer<String> hostNames;
        private final NTPUDPClient client;
        private final int sleep;

        private NTPClientBackground(int period, String ... hostNames) {
            super("NTP Sync");
            this.client = new NTPUDPClient();
            this.sleep = period * 1000;
            this.hostNames = new RingBuffer(hostNames.length);
            Collections.addAll(this.hostNames, hostNames);
        }

        @Override
        public void run() {
            TimeSystem.this.offset = CONFIGURED_OFFSET;
            if (this.hostNames.isEmpty()) {
                logger.error("Time system shutting down, no NTP hosts configured");
            }
            try {
                block6: while (true) {
                    Iterator<String> iterator = this.hostNames.iterator();
                    while (true) {
                        if (!iterator.hasNext()) continue block6;
                        String hostName = iterator.next();
                        try {
                            InetAddress host = InetAddress.getByName(hostName);
                            TimeInfo time = this.client.getTime(host);
                            time.computeDetails();
                            Long offset = time.getOffset();
                            if (offset == null) {
                                throw new IllegalArgumentException("Missing time offset");
                            }
                            TimeSystem.this.offset = offset;
                        }
                        catch (UnknownHostException e) {
                            logger.warn("Unable to resolve host {}, skipping", (Object)hostName);
                        }
                        catch (IOException e) {
                            logger.warn("I/O error from host {}, skipping - {}", (Object)hostName, (Object)e.getMessage());
                        }
                        catch (IllegalArgumentException e) {
                            logger.error("Unsatisfied condition, illegal argument computing time delta from host {}, skipping - {}", (Object)hostName, (Object)e.getMessage());
                        }
                        NTPClientBackground.sleep(this.sleep);
                    }
                    break;
                }
            }
            catch (InterruptedException e) {
                logger.warn("Time system shutting down, interrupted");
                return;
            }
        }

        public void stopIt() {
            this.interrupt();
        }
    }
}

