package com.destroystokyo.paper.event.profile;

import com.google.common.collect.Multimap;
import com.google.common.collect.Sets;
import com.mojang.authlib.GameProfile;
import com.mojang.authlib.GameProfileRepository;
import com.mojang.authlib.ProfileLookupCallback;
import com.mojang.authlib.properties.Property;
import com.mojang.authlib.properties.PropertyMap;
import org.bukkit.Bukkit;
import org.bukkit.event.Event;
import org.bukkit.event.HandlerList;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Set;
import java.util.UUID;

/**
 * Allows a plugin to intercept a Profile Lookup for a Profile by name
 *
 * At the point of event fire, the UUID and properties are unset.
 *
 * If a plugin sets the UUID, and optionally the properties, the API call to look up the profile may be skipped.
 *
 * No guarantees are made about thread execution context for this event. If you need to know, check
 * event.isAsync()
 */
public class PreLookupProfileEvent extends Event {

    private static final HandlerList handlers = new HandlerList();

    private final String name;

    private UUID uuid;
    private PropertyMap properties = new PropertyMap();

    public PreLookupProfileEvent(@Nonnull String name) {
        super(!Bukkit.isPrimaryThread());
        this.name = name;
    }

    /**
     * @return Name of the profile
     */
    @Nonnull
    public String getName() {
        return name;
    }

    /**
     *  @return The UUID of the profile if it has already been provided by a plugin
     *
     * If this value is left null by the completion of the event call, then the server will
     * trigger a call to the Mojang API to look up the UUID (Network Request), and subsequently, fire a
     * {@link LookupProfileEvent}
     */
    @Nullable
    public UUID getUUID() {
        return uuid;
    }

    /**
     * Sets the UUID for this player name. This will skip the initial API call to find the players UUID.
     *
     * However, if Profile Properties are needed by the server, you must also set them or else an API call will still be made.
     *
     * @param uuid the UUID to set on the {@link GameProfile} or null to reset
     */
    public void setUUID(@Nullable UUID uuid) {
        this.uuid = uuid;
    }

    /**
     * Get the properties for this profile
     *
     * @return the property map to attach to the new {@link GameProfile}
     */
    @Nonnull
    public Multimap<String, Property> getProperties() {
        return properties;
    }

    /**
     * Completely replaces all Properties with the new provided properties
     * @param properties the properties to set on the new profile
     */
    public void setProperties(Multimap<String, Property> properties) {
        this.properties = new PropertyMap();
        this.properties.putAll(properties);
    }

    /**
     * Adds additional properties, without removing the original properties
     * @param properties the properties to add to the existing properties
     */
    public void addProperties(Multimap<String, Property> properties) {
        this.properties.putAll(properties);
    }

    public HandlerList getHandlers() {
        return handlers;
    }

    public static HandlerList getHandlerList() {
        return handlers;
    }

    /*
     * Wraps the Profile Repository so we can intercept all lookups
     */
    public static GameProfileRepository wrapProfileRepository(final GameProfileRepository orig) {
        return (names, agent, callback) -> {
            Set<String> unfoundNames = Sets.newHashSet();
            for (String name : names) {
                PreLookupProfileEvent event = new PreLookupProfileEvent(name);
                event.callEvent();
                if (event.getUUID() != null) {
                    // Plugin provided UUI, we can skip network call.
                    GameProfile gameprofile = new GameProfile(event.getUUID(), name);
                    // We might even have properties!
                    gameprofile.getProperties().putAll(event.getProperties());
                    callback.onProfileLookupSucceeded(gameprofile);
                } else {
                    unfoundNames.add(name);
                }
            }

            // Some things were not found.... Proceed to look up.
            if (!unfoundNames.isEmpty() && orig != null) {
                String[] namesArr = unfoundNames.toArray(new String[unfoundNames.size()]);
                orig.findProfilesByNames(namesArr, agent, new PreProfileLookupCallback(callback));
            }
        };
    }

    private static class PreProfileLookupCallback implements ProfileLookupCallback {
        private final ProfileLookupCallback callback;

        PreProfileLookupCallback(ProfileLookupCallback callback) {
            this.callback = callback;
        }

        @Override
        public void onProfileLookupSucceeded(GameProfile gameProfile) {
            new LookupProfileEvent(gameProfile).callEvent();
            callback.onProfileLookupSucceeded(gameProfile);
        }

        @Override
        public void onProfileLookupFailed(GameProfile gameProfile, Exception e) {
            callback.onProfileLookupFailed(gameProfile, e);
        }
    }
}
