/*
 * Decompiled with CFR 0.152.
 */
package won.protocol.util.linkeddata;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.Serializable;
import java.lang.invoke.MethodHandles;
import java.net.URI;
import java.text.MessageFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.EnumSet;
import java.util.Locale;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.CountDownLatch;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import net.sf.ehcache.Cache;
import net.sf.ehcache.CacheException;
import net.sf.ehcache.Ehcache;
import net.sf.ehcache.Element;
import org.apache.jena.query.Dataset;
import org.apache.jena.query.DatasetFactory;
import org.apache.jena.riot.Lang;
import org.apache.jena.riot.RDFDataMgr;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.cache.ehcache.EhCacheCacheManager;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import won.protocol.rest.DatasetResponseWithStatusCodeAndHeaders;
import won.protocol.util.linkeddata.CrawlerCallback;
import won.protocol.util.linkeddata.LinkedDataSource;
import won.protocol.util.linkeddata.LinkedDataSourceBase;

@Qualifier(value="default")
public class CachingLinkedDataSource
extends LinkedDataSourceBase
implements LinkedDataSource,
InitializingBean {
    private static final String CACHE_NAME = "linkedDataCache";
    private static final String HTTP_DATE_FORMAT = "EEE, dd MMM yyyy HH:mm:ss z";
    private static final int DEFAULT_EXPIRY_PERIOD = 600;
    private static final int DEFAULT_BYTE_ARRAY_SIZE = 500;
    private static final Logger logger = LoggerFactory.getLogger(MethodHandles.lookup().lookupClass());
    @Autowired(required=true)
    private EhCacheCacheManager cacheManager;
    private Ehcache cache;
    private CrawlerCallback crawlerCallback = null;
    private boolean sharedCache = true;
    private ConcurrentMap<String, CountDownLatch> countDownLatchMap = new ConcurrentHashMap<String, CountDownLatch>(10);

    public void setSharedCache(boolean sharedCache) {
        this.sharedCache = sharedCache;
    }

    public boolean isSharedCache() {
        return this.sharedCache;
    }

    public void invalidate(URI resource) {
        if (resource == null) {
            return;
        }
        logger.debug("invalidating cached resource {}", (Object)resource);
        this.cache.remove((Serializable)((Object)this.makeCacheKey(resource, null)));
    }

    public void invalidate(URI resource, URI requesterWebID) {
        if (resource == null || requesterWebID == null) {
            return;
        }
        logger.debug("invalidating cached resource {} for webid {}", (Object)resource, (Object)requesterWebID);
        this.cache.remove((Serializable)((Object)this.makeCacheKey(resource, requesterWebID)));
    }

    public void clear() {
        this.cache.removeAll();
    }

    public void addToCache(Dataset dataset, URI resource, URI requesterWebID) {
        String key = this.makeCacheKey(resource, requesterWebID);
        LinkedDataCacheEntry entry = new LinkedDataCacheEntry(CachingLinkedDataSource.writeDatasetToByteArray(dataset));
        this.cache.put(new Element((Object)this.makeCacheKey(resource, requesterWebID), (Object)entry));
    }

    public void addToCache(Dataset dataset, URI resource) {
        this.addToCache(dataset, resource, null);
    }

    @Override
    public Dataset getDataForResource(URI resource, URI requesterWebID) {
        if (resource == null) {
            throw new IllegalArgumentException("resource cannot be null");
        }
        Element element = null;
        try {
            element = this.cache.get((Serializable)((Object)this.makeCacheKey(resource, requesterWebID)));
        }
        catch (CacheException e) {
            logger.warn(String.format("Couldn't fetch resource %s", resource));
            logger.debug("Exception is:", (Throwable)e);
            return DatasetFactory.createGeneral();
        }
        LinkedDataCacheEntry linkedDataCacheEntry = null;
        if (element != null) {
            Object cachedObject = element.getObjectValue();
            if (!(cachedObject instanceof LinkedDataCacheEntry)) {
                throw new IllegalStateException(new MessageFormat("The underlying linkedDataCache should only contain Datasets, but we got a {0} for URI {1}").format(new Object[]{cachedObject.getClass(), resource}));
            }
            linkedDataCacheEntry = (LinkedDataCacheEntry)cachedObject;
        }
        return this.fetchOrUseCached(resource, requesterWebID, linkedDataCacheEntry).getDataset();
    }

    private DatasetResponseWithStatusCodeAndHeaders fetchOrUseCached(URI resource, URI requesterWebID, LinkedDataCacheEntry linkedDataCacheEntry) {
        DatasetResponseWithStatusCodeAndHeaders responseData = null;
        HttpHeaders headers = new HttpHeaders();
        if (linkedDataCacheEntry != null) {
            Date now = new Date();
            if (linkedDataCacheEntry.isExpiredAtDate(now)) {
                this.cache.remove((Serializable)((Object)this.makeCacheKey(resource, requesterWebID)));
                logger.debug("cache item {} expired, fetching again.", (Object)resource);
                return this.fetchOnlyOnce(resource, requesterWebID, linkedDataCacheEntry, headers);
            }
            if (linkedDataCacheEntry.getCacheControlFlags().contains((Object)CacheControlFlag.PRIVATE) && this.isSharedCache()) {
                logger.debug("cache item {} is Cache-Control:private and we are a shared cache. Will return cached copy only after server checks ETAG (and client cert), therefore sending request to server.", (Object)resource);
                return this.fetchOnlyOnce(resource, requesterWebID, linkedDataCacheEntry, headers);
            }
            logger.debug("returning cached version of {}", (Object)resource);
            return linkedDataCacheEntry.recreateResponse();
        }
        logger.debug("Nothing found in cache for {}, fetching remotely", (Object)resource);
        responseData = this.fetchOnlyOnce(resource, requesterWebID, null, headers);
        if (this.crawlerCallback != null) {
            try {
                this.crawlerCallback.onDatasetCrawled(resource, responseData.getDataset());
            }
            catch (Exception e) {
                logger.info(String.format("error during callback execution for dataset %s", resource.toString()), (Throwable)e);
            }
        }
        return responseData;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private DatasetResponseWithStatusCodeAndHeaders fetchOnlyOnce(URI resource, URI requesterWebID, LinkedDataCacheEntry linkedDataCacheEntry, HttpHeaders headers) {
        String cacheKey = this.makeCacheKey(resource, requesterWebID);
        CountDownLatch latch = new CountDownLatch(1);
        CountDownLatch preExistingLatch = this.countDownLatchMap.putIfAbsent(cacheKey, latch);
        try {
            DatasetResponseWithStatusCodeAndHeaders datasetResponse;
            if (preExistingLatch != null) {
                logger.debug("resource " + cacheKey + " is being fetched in another thread, we wait for its result and use it if it turns out to be cacheable");
                try {
                    preExistingLatch.await();
                }
                catch (InterruptedException e) {
                    logger.warn("interrupted while waiting for another thread to fetch '" + resource + "'");
                }
                Element element = this.cache.get((Serializable)((Object)cacheKey));
                if (element != null) {
                    logger.debug("resource " + cacheKey + " turned out to be cacheable, using it");
                    LinkedDataCacheEntry entry = (LinkedDataCacheEntry)element.getObjectValue();
                    DatasetResponseWithStatusCodeAndHeaders datasetResponseWithStatusCodeAndHeaders = entry.recreateResponse();
                    return datasetResponseWithStatusCodeAndHeaders;
                }
                logger.debug("resource " + cacheKey + " did not turn out to be cacheable - fetching it, too");
            }
            DatasetResponseWithStatusCodeAndHeaders datasetResponseWithStatusCodeAndHeaders = datasetResponse = this.fetchAndCacheIfAppropriate(resource, requesterWebID, linkedDataCacheEntry, headers);
            return datasetResponseWithStatusCodeAndHeaders;
        }
        finally {
            this.countDownLatchMap.remove(cacheKey, latch);
            latch.countDown();
        }
    }

    private DatasetResponseWithStatusCodeAndHeaders fetchAndCacheIfAppropriate(URI resource, URI requesterWebID, LinkedDataCacheEntry linkedDataCacheEntry, HttpHeaders headers) {
        DatasetResponseWithStatusCodeAndHeaders responseData = this.fetchWithEtagValidation(resource, requesterWebID, linkedDataCacheEntry, headers);
        Date expires = this.parseCacheControlMaxAgeValue(resource, responseData);
        if (responseData.getDataset() == null) {
            throw new IllegalStateException("Could not load dataset for URI " + resource + " and requesterWebID " + requesterWebID);
        }
        if (expires == null && (expires = this.parseExpiresHeader(resource, responseData)) != null && expires.getTime() == 0L) {
            return responseData;
        }
        EnumSet<CacheControlFlag> cacheControlFlags = this.parseCacheControlHeaderFlags(resource, responseData);
        if (cacheControlFlags.contains((Object)CacheControlFlag.NO_STORE) || cacheControlFlags.contains((Object)CacheControlFlag.NO_CACHE)) {
            this.cache.remove((Serializable)((Object)this.makeCacheKey(resource, requesterWebID)));
            logger.debug("Fetched {}. Will not be cached due to Cache-Control headers sent by server", (Object)resource);
            return responseData;
        }
        Date responseDate = this.parseDateHeader(resource, responseData);
        if (responseDate != null && expires != null && (responseDate.equals(expires) || responseDate.after(expires))) {
            logger.debug("Fetched {}. Will not be cached due to Expires/Date header combination sent by server", (Object)resource);
            this.cache.remove((Serializable)((Object)this.makeCacheKey(resource, requesterWebID)));
            return responseData;
        }
        String etag = responseData.getResponseHeaders().getFirst("ETag");
        if (etag == null && responseData.getStatusCode() == HttpStatus.NOT_MODIFIED.value() && linkedDataCacheEntry != null) {
            etag = linkedDataCacheEntry.getEtag();
        }
        LinkedDataCacheEntry entry = new LinkedDataCacheEntry(etag, expires, CachingLinkedDataSource.writeDatasetToByteArray(responseData.getDataset()), cacheControlFlags, responseData.getResponseHeaders(), responseData.getStatusCode());
        this.cache.put(new Element((Object)this.makeCacheKey(resource, requesterWebID), (Object)entry));
        logger.debug("Fetched and cached {} ", (Object)resource);
        if (logger.isDebugEnabled()) {
            logger.debug("cache size: {} elements, in-memory size: {} bytes", (Object)this.cache.getSize(), (Object)this.cache.calculateInMemorySize());
        }
        return responseData;
    }

    private static Dataset readDatasetFromByteArray(byte[] datasetbytes) {
        Dataset dataset = DatasetFactory.create();
        RDFDataMgr.read((Dataset)dataset, (InputStream)new ByteArrayInputStream(datasetbytes), (Lang)Lang.NQUADS);
        return dataset;
    }

    private static byte[] writeDatasetToByteArray(Dataset dataset) {
        ByteArrayOutputStream out = new ByteArrayOutputStream(500);
        RDFDataMgr.write((OutputStream)out, (Dataset)dataset, (Lang)Lang.NQUADS);
        return out.toByteArray();
    }

    private DatasetResponseWithStatusCodeAndHeaders fetchWithEtagValidation(URI resource, URI requesterWebID, LinkedDataCacheEntry linkedDataCacheEntry, HttpHeaders headers) {
        if (linkedDataCacheEntry == null || linkedDataCacheEntry.getEtag() == null) {
            logger.debug("fetching from server without ETAG validation: {} ", (Object)resource);
            return this.fetch(resource, requesterWebID, headers);
        }
        HttpHeaders myHeaders = headers != null ? headers : new HttpHeaders();
        myHeaders.add("If-None-Match", linkedDataCacheEntry.getEtag());
        logger.debug("fetching from server with ETAG validation: {} ", (Object)resource);
        DatasetResponseWithStatusCodeAndHeaders datasetResponse = this.fetch(resource, requesterWebID, myHeaders);
        if (datasetResponse.getStatusCode() == HttpStatus.NOT_MODIFIED.value()) {
            logger.debug("server said our ETAG is still valid, using cached dataset for URI {} ", (Object)resource);
            datasetResponse = new DatasetResponseWithStatusCodeAndHeaders(CachingLinkedDataSource.readDatasetFromByteArray(linkedDataCacheEntry.getDataset()), datasetResponse.getStatusCode(), datasetResponse.getResponseHeaders());
        } else {
            logger.debug("server said our ETAG is not valid, not using cached result for URI {} ", (Object)resource);
        }
        return datasetResponse;
    }

    private DatasetResponseWithStatusCodeAndHeaders fetch(URI resource, URI requesterWebID, HttpHeaders headers) {
        DatasetResponseWithStatusCodeAndHeaders responseData;
        if (requesterWebID != null) {
            logger.debug("fetching linked data for URI {} with WebID {}", (Object)resource, (Object)requesterWebID);
            responseData = this.linkedDataRestClient.readResourceDataWithHeaders(resource, requesterWebID, headers);
            if (logger.isTraceEnabled()) {
                logger.trace("fetched resource {} with requesterWebID {}: ", (Object)resource, (Object)requesterWebID);
                RDFDataMgr.write((OutputStream)System.out, (Dataset)responseData.getDataset(), (Lang)Lang.TRIG);
            }
        } else {
            logger.debug("fetching linked data for URI {} without WebID", (Object)resource, (Object)requesterWebID);
            responseData = this.linkedDataRestClient.readResourceDataWithHeaders(resource, headers);
            if (logger.isTraceEnabled()) {
                logger.trace("fetched resource {} without requesterWebID:", (Object)resource, (Object)requesterWebID);
                RDFDataMgr.write((OutputStream)System.out, (Dataset)responseData.getDataset(), (Lang)Lang.TRIG);
            }
        }
        return responseData;
    }

    private Date parseExpiresHeader(URI resource, DatasetResponseWithStatusCodeAndHeaders responseData) {
        String expiresHeader = responseData.getResponseHeaders().getFirst("Expires");
        if (expiresHeader == null) {
            return null;
        }
        expiresHeader = expiresHeader.trim();
        SimpleDateFormat format = new SimpleDateFormat(HTTP_DATE_FORMAT, Locale.ENGLISH);
        Date expires = null;
        try {
            expires = format.parse(expiresHeader);
        }
        catch (ParseException e) {
            logger.debug("could not parse 'Expires' header ' " + expiresHeader + "' obtained for '" + resource + "', marking as already expired");
            return new Date(0L);
        }
        return expires;
    }

    private Date addNSecondsToNow(int seconds) {
        Calendar cal = Calendar.getInstance();
        cal.setTime(new Date());
        cal.add(13, seconds);
        Date expires = cal.getTime();
        return expires;
    }

    private Date parseDateHeader(URI resource, DatasetResponseWithStatusCodeAndHeaders responseData) {
        String dateHeader = responseData.getResponseHeaders().getFirst("Date");
        if (dateHeader == null) {
            return null;
        }
        SimpleDateFormat format = new SimpleDateFormat(HTTP_DATE_FORMAT, Locale.ENGLISH);
        Date date = null;
        try {
            date = format.parse(dateHeader);
        }
        catch (ParseException e) {
            date = new Date();
            logger.warn("could not parse 'Date' header ' " + dateHeader + "' obtained for '" + resource + "', using current date");
        }
        return date;
    }

    private EnumSet<CacheControlFlag> parseCacheControlHeaderFlags(URI resource, DatasetResponseWithStatusCodeAndHeaders responseData) {
        String[] values;
        String cacheControlHeaderValue = responseData.getResponseHeaders().getFirst("Cache-Control");
        EnumSet<CacheControlFlag> cacheControlFlags = EnumSet.noneOf(CacheControlFlag.class);
        if (cacheControlHeaderValue == null) {
            return cacheControlFlags;
        }
        for (String value : values = cacheControlHeaderValue.split(",")) {
            CacheControlFlag flag = CacheControlFlag.forName(value.trim());
            if (flag == null) continue;
            cacheControlFlags.add(flag);
        }
        return cacheControlFlags;
    }

    private Date parseCacheControlMaxAgeValue(URI resource, DatasetResponseWithStatusCodeAndHeaders responseData) {
        String cacheControlHeaderValue = responseData.getResponseHeaders().getFirst("Cache-Control");
        if (cacheControlHeaderValue == null) {
            return null;
        }
        Pattern maxagePattern = Pattern.compile("[^\\s,]*max-age\\s*=\\s*(\\d+)[$\\s,]");
        Matcher m = maxagePattern.matcher(cacheControlHeaderValue);
        if (!m.find()) {
            return null;
        }
        String maxAgeValueString = m.group(1);
        int maxAgeInt = 3600;
        try {
            maxAgeInt = Integer.parseInt(maxAgeValueString);
        }
        catch (NumberFormatException e) {
            logger.warn("could not parse 'Expires' header ' " + cacheControlHeaderValue + "' obtained for '" + resource + "' using default expiry period of 1 hour", (Throwable)e);
        }
        Calendar cal = Calendar.getInstance();
        cal.setTime(new Date());
        cal.add(13, maxAgeInt);
        return cal.getTime();
    }

    @Override
    public Dataset getDataForResource(URI resource) {
        return this.getDataForResource(resource, null);
    }

    public void afterPropertiesSet() throws Exception {
        Cache baseCache = this.cacheManager.getCacheManager().getCache(CACHE_NAME);
        if (baseCache == null) {
            throw new IllegalArgumentException(String.format("could not find a cache with name '%s' in ehcache config", CACHE_NAME));
        }
        this.cache = baseCache;
    }

    public void setCacheManager(EhCacheCacheManager cacheManager) {
        this.cacheManager = cacheManager;
    }

    @Autowired(required=false)
    public void setCrawlerCallback(CrawlerCallback crawlerCallback) {
        this.crawlerCallback = crawlerCallback;
    }

    private String makeCacheKey(URI resource, URI requesterWebID) {
        return resource.toString() + (requesterWebID == null ? " (no Web ID)" : requesterWebID.toString());
    }

    public static class LinkedDataCacheEntry {
        private String etag = null;
        private Date expires = null;
        private byte[] dataset = null;
        private EnumSet<CacheControlFlag> cacheControlFlags = EnumSet.noneOf(CacheControlFlag.class);
        private HttpHeaders headers;
        private int statusCode;

        public LinkedDataCacheEntry(String etag, Date expires, byte[] dataset, EnumSet<CacheControlFlag> cacheControlFlags, HttpHeaders headers, int statusCode) {
            this.etag = etag;
            this.expires = expires;
            this.dataset = dataset;
            this.cacheControlFlags = cacheControlFlags != null ? cacheControlFlags : EnumSet.noneOf(CacheControlFlag.class);
            this.headers = headers;
            this.statusCode = statusCode;
        }

        public LinkedDataCacheEntry(byte[] dataset) {
            this(null, null, dataset, null, null, 0);
        }

        private static Date inOneYear() {
            Calendar cal = Calendar.getInstance();
            cal.setTime(new Date());
            cal.add(1, 1);
            return cal.getTime();
        }

        public DatasetResponseWithStatusCodeAndHeaders recreateResponse() {
            return new DatasetResponseWithStatusCodeAndHeaders(CachingLinkedDataSource.readDatasetFromByteArray(this.dataset), this.statusCode, this.headers);
        }

        public String getEtag() {
            return this.etag;
        }

        public byte[] getDataset() {
            return this.dataset;
        }

        public Date getExpires() {
            return this.expires;
        }

        public EnumSet<CacheControlFlag> getCacheControlFlags() {
            return this.cacheControlFlags;
        }

        public boolean isExpiredAtDate(Date when) {
            if (this.expires == null) {
                return false;
            }
            return this.expires.before(when);
        }
    }

    public static enum CacheControlFlag {
        PUBLIC("public"),
        PRIVATE("private"),
        NO_CACHE("no-cache"),
        NO_STORE("no-store"),
        MUST_REVALIDATE("must-revalidate");

        private String name;

        private CacheControlFlag(String name) {
            this.name = name;
        }

        public static CacheControlFlag forName(String name) {
            switch (name) {
                case "public": {
                    return PUBLIC;
                }
                case "private": {
                    return PRIVATE;
                }
                case "no-cache": {
                    return NO_CACHE;
                }
                case "no-store": {
                    return NO_STORE;
                }
                case "must-revalidate": {
                    return MUST_REVALIDATE;
                }
            }
            return null;
        }

        public String getName() {
            return this.name;
        }
    }
}

