/*
 * Decompiled with CFR 0.152.
 */
package com.spotify.helios.client;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.JavaType;
import com.fasterxml.jackson.databind.type.TypeFactory;
import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.base.Preconditions;
import com.google.common.base.Supplier;
import com.google.common.base.Suppliers;
import com.google.common.base.Throwables;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.util.concurrent.AsyncFunction;
import com.google.common.util.concurrent.FutureFallback;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListeningExecutorService;
import com.google.common.util.concurrent.MoreExecutors;
import com.spotify.helios.common.HeliosException;
import com.spotify.helios.common.Json;
import com.spotify.helios.common.Resolver;
import com.spotify.helios.common.VersionCompatibility;
import com.spotify.helios.common.descriptors.Deployment;
import com.spotify.helios.common.descriptors.DeploymentGroup;
import com.spotify.helios.common.descriptors.HostStatus;
import com.spotify.helios.common.descriptors.Job;
import com.spotify.helios.common.descriptors.JobId;
import com.spotify.helios.common.descriptors.JobStatus;
import com.spotify.helios.common.descriptors.RolloutOptions;
import com.spotify.helios.common.protocol.CreateDeploymentGroupResponse;
import com.spotify.helios.common.protocol.CreateJobResponse;
import com.spotify.helios.common.protocol.DeploymentGroupStatusResponse;
import com.spotify.helios.common.protocol.HostDeregisterResponse;
import com.spotify.helios.common.protocol.JobDeleteResponse;
import com.spotify.helios.common.protocol.JobDeployResponse;
import com.spotify.helios.common.protocol.JobUndeployResponse;
import com.spotify.helios.common.protocol.RemoveDeploymentGroupResponse;
import com.spotify.helios.common.protocol.RollingUpdateRequest;
import com.spotify.helios.common.protocol.RollingUpdateResponse;
import com.spotify.helios.common.protocol.SetGoalResponse;
import com.spotify.helios.common.protocol.TaskStatusEvents;
import com.spotify.helios.common.protocol.VersionResponse;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.Field;
import java.net.ConnectException;
import java.net.HttpURLConnection;
import java.net.SocketTimeoutException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.UnknownHostException;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.zip.GZIPInputStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class HeliosClient
implements AutoCloseable {
    private static final Logger log = LoggerFactory.getLogger(HeliosClient.class);
    private static final long RETRY_TIMEOUT_MILLIS = TimeUnit.SECONDS.toMillis(60L);
    private static final long HTTP_TIMEOUT_MILLIS = TimeUnit.SECONDS.toMillis(10L);
    private final AtomicBoolean versionWarningLogged = new AtomicBoolean();
    private final String user;
    private final Supplier<List<URI>> endpointSupplier;
    private final ListeningExecutorService executorService;

    HeliosClient(String user, Supplier<List<URI>> endpointSupplier, ListeningExecutorService executorService) {
        this.user = (String)Preconditions.checkNotNull((Object)user);
        this.endpointSupplier = (Supplier)Preconditions.checkNotNull(endpointSupplier);
        this.executorService = (ListeningExecutorService)Preconditions.checkNotNull((Object)executorService);
    }

    HeliosClient(String user, List<URI> endpoints, ListeningExecutorService executorService) {
        this(user, (Supplier<List<URI>>)Suppliers.ofInstance(endpoints), executorService);
    }

    HeliosClient(String user, Supplier<List<URI>> endpointSupplier) {
        this(user, endpointSupplier, MoreExecutors.listeningDecorator((ExecutorService)MoreExecutors.getExitingExecutorService((ThreadPoolExecutor)((ThreadPoolExecutor)Executors.newFixedThreadPool(4)), (long)0L, (TimeUnit)TimeUnit.SECONDS)));
    }

    HeliosClient(String user, List<URI> endpoints) {
        this(user, (Supplier<List<URI>>)Suppliers.ofInstance(endpoints));
    }

    @Override
    public void close() {
        this.executorService.shutdownNow();
    }

    private URI uri(String path) {
        return this.uri(path, Collections.emptyMap());
    }

    private URI uri(String path, Map<String, String> query) {
        Preconditions.checkArgument((boolean)path.startsWith("/"));
        HashMap queryWithUser = Maps.newHashMap(query);
        queryWithUser.put("user", this.user);
        String queryPart = Joiner.on((char)'&').withKeyValueSeparator("=").join((Map)queryWithUser);
        try {
            return new URI("http", "helios", path, queryPart, null);
        }
        catch (URISyntaxException e) {
            throw Throwables.propagate((Throwable)e);
        }
    }

    private String path(String resource, Object ... params) {
        String path;
        if (params.length == 0) {
            path = resource;
        } else {
            ArrayList encodedParams = Lists.newArrayList();
            for (Object param : params) {
                URI u;
                try {
                    String p = param.toString().replace("/", "%2F");
                    u = new URI("http", "ignore", "/" + p, "");
                }
                catch (URISyntaxException e) {
                    throw Throwables.propagate((Throwable)e);
                }
                encodedParams.add(u.getRawPath().substring(1));
            }
            path = String.format(resource, encodedParams.toArray());
        }
        return path;
    }

    private ListenableFuture<Response> request(URI uri, String method) {
        return this.request(uri, method, null);
    }

    private ListenableFuture<Response> request(final URI uri, final String method, Object entity) {
        byte[] entityBytes;
        final HashMap headers = Maps.newHashMap();
        headers.put("Helios-Version", Arrays.asList("0.9.12"));
        if (entity != null) {
            headers.put("Content-Type", Arrays.asList("application/json"));
            headers.put("Charset", Arrays.asList("utf-8"));
            entityBytes = Json.asBytesUnchecked(entity);
        } else {
            entityBytes = new byte[]{};
        }
        return this.executorService.submit((Callable)new Callable<Response>(){

            @Override
            public Response call() throws Exception {
                HttpURLConnection connection = HeliosClient.this.connect(uri, method, entityBytes, headers);
                int status = connection.getResponseCode();
                InputStream rawStream = status / 100 != 2 ? connection.getErrorStream() : connection.getInputStream();
                boolean gzip = this.isGzipCompressed(connection);
                InputStream stream = gzip ? new GZIPInputStream(rawStream) : rawStream;
                ByteArrayOutputStream payload = new ByteArrayOutputStream();
                if (stream != null) {
                    int n;
                    byte[] buffer = new byte[4096];
                    while ((n = stream.read(buffer, 0, buffer.length)) != -1) {
                        payload.write(buffer, 0, n);
                    }
                }
                URI realUri = connection.getURL().toURI();
                if (log.isTraceEnabled()) {
                    log.trace("rep: {} {} {} {} {} gzip:{}", new Object[]{method, realUri, status, payload.size(), HeliosClient.this.decode(payload), gzip});
                } else {
                    log.debug("rep: {} {} {} {} gzip:{}", new Object[]{method, realUri, status, payload.size(), gzip});
                }
                HeliosClient.this.checkprotocolVersionStatus(connection);
                return new Response(method, uri, status, payload.toByteArray());
            }

            private boolean isGzipCompressed(HttpURLConnection connection) {
                List<String> encodings = connection.getHeaderFields().get("Content-Encoding");
                if (encodings == null) {
                    return false;
                }
                for (String encoding : encodings) {
                    if (!"gzip".equals(encoding)) continue;
                    return true;
                }
                return false;
            }
        });
    }

    private void checkprotocolVersionStatus(HttpURLConnection connection) {
        VersionCompatibility.Status versionStatus = this.getVersionStatus(connection);
        if (versionStatus == null) {
            log.debug("Server didn't return a version header!");
            return;
        }
        String serverVersion = connection.getHeaderField("Helios-Server-Version");
        if (versionStatus == VersionCompatibility.Status.MAYBE && this.versionWarningLogged.compareAndSet(false, true)) {
            log.warn("Your Helios client version [{}] is ahead of the server [{}].  This will probably work ok but there is the potential for weird things.  If in doubt, contact the Helios team if you think the cluster you're connecting to is out of date and should be upgraded.", (Object)"0.9.12", (Object)serverVersion);
        }
    }

    private VersionCompatibility.Status getVersionStatus(HttpURLConnection connection) {
        String status = connection.getHeaderField("Helios-Version-Status");
        if (status != null) {
            return VersionCompatibility.Status.valueOf(status);
        }
        return null;
    }

    private String decode(ByteArrayOutputStream payload) {
        byte[] bytes = payload.toByteArray();
        try {
            return Json.asPrettyString(Json.read(bytes, new TypeReference<Map<String, Object>>(){}));
        }
        catch (IOException e) {
            return new String(bytes, StandardCharsets.UTF_8);
        }
    }

    private HttpURLConnection connect(URI uri, String method, byte[] entity, Map<String, List<String>> headers) throws URISyntaxException, IOException, TimeoutException, InterruptedException, HeliosException {
        long deadline = System.currentTimeMillis() + RETRY_TIMEOUT_MILLIS;
        int offset = ThreadLocalRandom.current().nextInt();
        while (System.currentTimeMillis() < deadline) {
            List endpoints = (List)this.endpointSupplier.get();
            if (endpoints.isEmpty()) {
                throw new RuntimeException("failed to resolve master");
            }
            log.info("endpoint uris are {}", (Object)endpoints);
            for (int i = 0; i < endpoints.size() && System.currentTimeMillis() < deadline; ++i) {
                URI endpoint = (URI)endpoints.get(this.positive(offset + i) % endpoints.size());
                String fullpath = endpoint.getPath() + uri.getPath();
                String host = endpoint.getHost();
                int port = endpoint.getPort();
                if (host == null || port == -1) {
                    throw new HeliosException("Master endpoints must be of the form \"http[s]://heliosmaster.domain.net:<port>\"");
                }
                URI realUri = new URI("http", host + ":" + port, fullpath, uri.getQuery(), null);
                try {
                    log.info("connecting to {}", (Object)realUri);
                    return this.connect0(realUri, method, entity, headers);
                }
                catch (ConnectException | SocketTimeoutException | UnknownHostException e) {
                    log.debug(e.getClass().getSimpleName() + " - " + e.getMessage());
                    Thread.sleep(200L);
                    continue;
                }
            }
            log.warn("Failed to connect, retrying in 5 seconds.");
            Thread.sleep(5000L);
        }
        throw new TimeoutException("Timed out connecting to master");
    }

    private HttpURLConnection connect0(URI uri, String method, byte[] entity, Map<String, List<String>> headers) throws IOException {
        if (log.isTraceEnabled()) {
            log.trace("req: {} {} {} {} {} {}", new Object[]{method, uri, headers.size(), Joiner.on((char)',').withKeyValueSeparator("=").join(headers), entity.length, Json.asPrettyStringUnchecked(entity)});
        } else {
            log.debug("req: {} {} {} {}", new Object[]{method, uri, headers.size(), entity.length});
        }
        HttpURLConnection connection = (HttpURLConnection)uri.toURL().openConnection();
        connection.setRequestProperty("Accept-Encoding", "gzip");
        connection.setInstanceFollowRedirects(false);
        connection.setConnectTimeout((int)HTTP_TIMEOUT_MILLIS);
        connection.setReadTimeout((int)HTTP_TIMEOUT_MILLIS);
        for (Map.Entry<String, List<String>> header : headers.entrySet()) {
            for (String value : header.getValue()) {
                connection.addRequestProperty(header.getKey(), value);
            }
        }
        if (entity.length > 0) {
            connection.setDoOutput(true);
            connection.getOutputStream().write(entity);
        }
        this.setRequestMethod(connection, method);
        connection.getResponseCode();
        return connection;
    }

    private int positive(int value) {
        return value < 0 ? value + Integer.MAX_VALUE : value;
    }

    private void setRequestMethod(HttpURLConnection connection, String method) {
        Class<?> httpURLConnectionClass = connection.getClass();
        try {
            Field methodField = httpURLConnectionClass.getSuperclass().getDeclaredField("method");
            methodField.setAccessible(true);
            methodField.set(connection, method);
        }
        catch (IllegalAccessException | NoSuchFieldException e) {
            throw Throwables.propagate((Throwable)e);
        }
    }

    private <T> ListenableFuture<T> get(URI uri, TypeReference<T> typeReference) {
        return this.get(uri, Json.type(typeReference));
    }

    private <T> ListenableFuture<T> get(URI uri, Class<T> clazz) {
        return this.get(uri, Json.type(clazz));
    }

    private <T> ListenableFuture<T> get(URI uri, JavaType javaType) {
        return Futures.transform(this.request(uri, "GET"), new ConvertResponseToPojo(javaType));
    }

    private ListenableFuture<Integer> put(URI uri) {
        return this.status(this.request(uri, "PUT"));
    }

    public ListenableFuture<JobDeployResponse> deploy(Deployment job, String host) {
        return this.deploy(job, host, "");
    }

    public ListenableFuture<JobDeployResponse> deploy(Deployment job, String host, String token) {
        ImmutableSet deserializeReturnCodes = ImmutableSet.of((Object)200, (Object)404, (Object)405, (Object)400, (Object)403);
        return Futures.transform(this.request(this.uri(this.path("/hosts/%s/jobs/%s", host, job.getJobId()), (Map<String, String>)ImmutableMap.of((Object)"token", (Object)token)), "PUT", job), ConvertResponseToPojo.create(JobDeployResponse.class, (Set<Integer>)deserializeReturnCodes));
    }

    public ListenableFuture<SetGoalResponse> setGoal(Deployment job, String host) {
        return this.setGoal(job, host, "");
    }

    public ListenableFuture<SetGoalResponse> setGoal(Deployment job, String host, String token) {
        return Futures.transform(this.request(this.uri(this.path("/hosts/%s/jobs/%s", host, job.getJobId()), (Map<String, String>)ImmutableMap.of((Object)"token", (Object)token)), "PATCH", job), ConvertResponseToPojo.create(SetGoalResponse.class, (Set<Integer>)ImmutableSet.of((Object)200, (Object)404, (Object)403)));
    }

    private ListenableFuture<Integer> status(ListenableFuture<Response> req) {
        return Futures.transform(req, (Function)new Function<Response, Integer>(){

            public Integer apply(Response reply) {
                return reply.status;
            }
        });
    }

    public ListenableFuture<Deployment> deployment(String host, JobId job) {
        return this.get(this.uri(this.path("/hosts/%s/jobs/%s", host, job)), Deployment.class);
    }

    public ListenableFuture<HostStatus> hostStatus(String host) {
        return this.get(this.uri(this.path("/hosts/%s/status", host)), HostStatus.class);
    }

    public ListenableFuture<Map<String, HostStatus>> hostStatuses(List<String> hosts) {
        ConvertResponseToPojo converter = ConvertResponseToPojo.create((JavaType)TypeFactory.defaultInstance().constructMapType(Map.class, String.class, HostStatus.class), (Set<Integer>)ImmutableSet.of((Object)200));
        return Futures.transform(this.request(this.uri("/hosts/statuses"), "POST", hosts), converter);
    }

    public ListenableFuture<Integer> registerHost(String host, String id) {
        return this.put(this.uri(this.path("/hosts/%s", host), (Map<String, String>)ImmutableMap.of((Object)"id", (Object)id)));
    }

    public ListenableFuture<JobDeleteResponse> deleteJob(JobId id) {
        return this.deleteJob(id, "");
    }

    public ListenableFuture<JobDeleteResponse> deleteJob(JobId id, String token) {
        return Futures.transform(this.request(this.uri(this.path("/jobs/%s", id), (Map<String, String>)ImmutableMap.of((Object)"token", (Object)token)), "DELETE"), ConvertResponseToPojo.create(JobDeleteResponse.class, (Set<Integer>)ImmutableSet.of((Object)200, (Object)404, (Object)400, (Object)403)));
    }

    public ListenableFuture<JobUndeployResponse> undeploy(JobId jobId, String host) {
        return this.undeploy(jobId, host, "");
    }

    public ListenableFuture<JobUndeployResponse> undeploy(JobId jobId, String host, String token) {
        return Futures.transform(this.request(this.uri(this.path("/hosts/%s/jobs/%s", host, jobId), (Map<String, String>)ImmutableMap.of((Object)"token", (Object)token)), "DELETE"), ConvertResponseToPojo.create(JobUndeployResponse.class, (Set<Integer>)ImmutableSet.of((Object)200, (Object)404, (Object)400, (Object)403)));
    }

    public ListenableFuture<HostDeregisterResponse> deregisterHost(String host) {
        return Futures.transform(this.request(this.uri(this.path("/hosts/%s", host)), "DELETE"), ConvertResponseToPojo.create(HostDeregisterResponse.class, (Set<Integer>)ImmutableSet.of((Object)200, (Object)404)));
    }

    public ListenableFuture<List<String>> listHosts() {
        return this.get(this.uri("/hosts/"), new TypeReference<List<String>>(){});
    }

    public ListenableFuture<List<String>> listMasters() {
        return this.get(this.uri("/masters/"), new TypeReference<List<String>>(){});
    }

    public ListenableFuture<VersionResponse> version() {
        ListenableFuture futureWithFallback = Futures.withFallback(this.request(this.uri("/version/"), "GET"), (FutureFallback)new FutureFallback<Response>(){

            public ListenableFuture<Response> create(Throwable t) throws Exception {
                return Futures.immediateFuture(null);
            }
        });
        return Futures.transform((ListenableFuture)futureWithFallback, (AsyncFunction)new AsyncFunction<Response, VersionResponse>(){

            public ListenableFuture<VersionResponse> apply(Response reply) throws Exception {
                String masterVersion = reply == null ? "Unable to connect to master" : (reply.status == 200 ? Json.read(reply.payload, String.class) : "Master replied with error code " + reply.status);
                return Futures.immediateFuture((Object)new VersionResponse("0.9.12", masterVersion));
            }
        });
    }

    public ListenableFuture<CreateJobResponse> createJob(Job descriptor) {
        return Futures.transform(this.request(this.uri("/jobs/"), "POST", descriptor), ConvertResponseToPojo.create(CreateJobResponse.class, (Set<Integer>)ImmutableSet.of((Object)200, (Object)400)));
    }

    public ListenableFuture<Map<JobId, Job>> jobs(String query) {
        return this.get(this.uri("/jobs", (Map<String, String>)ImmutableMap.of((Object)"q", (Object)query)), new TypeReference<Map<JobId, Job>>(){});
    }

    public ListenableFuture<Map<JobId, Job>> jobs() {
        return this.get(this.uri("/jobs"), new TypeReference<Map<JobId, Job>>(){});
    }

    public ListenableFuture<TaskStatusEvents> jobHistory(JobId jobId) {
        return Futures.transform(this.request(this.uri(this.path("/history/jobs/%s", jobId.toString())), "GET"), ConvertResponseToPojo.create(TaskStatusEvents.class, (Set<Integer>)ImmutableSet.of((Object)200, (Object)404)));
    }

    public ListenableFuture<JobStatus> jobStatus(JobId jobId) {
        return this.get(this.uri(this.path("/jobs/%s/status", jobId)), JobStatus.class);
    }

    public ListenableFuture<Map<JobId, JobStatus>> jobStatuses(Set<JobId> jobs) {
        ConvertResponseToPojo converter = ConvertResponseToPojo.create((JavaType)TypeFactory.defaultInstance().constructMapType(Map.class, JobId.class, JobStatus.class), (Set<Integer>)ImmutableSet.of((Object)200));
        return Futures.transform(this.request(this.uri("/jobs/statuses"), "POST", jobs), converter);
    }

    public ListenableFuture<DeploymentGroup> deploymentGroup(String name) {
        return this.get(this.uri("/deployment-group/" + name), new TypeReference<DeploymentGroup>(){});
    }

    public ListenableFuture<List<String>> listDeploymentGroups() {
        return this.get(this.uri("/deployment-group/"), new TypeReference<List<String>>(){});
    }

    public ListenableFuture<DeploymentGroupStatusResponse> deploymentGroupStatus(String name) {
        return this.get(this.uri(this.path("/deployment-group/%s/status", name)), new TypeReference<DeploymentGroupStatusResponse>(){});
    }

    public ListenableFuture<CreateDeploymentGroupResponse> createDeploymentGroup(DeploymentGroup descriptor) {
        return Futures.transform(this.request(this.uri("/deployment-group/"), "POST", descriptor), ConvertResponseToPojo.create(CreateDeploymentGroupResponse.class, (Set<Integer>)ImmutableSet.of((Object)200, (Object)400)));
    }

    public ListenableFuture<RemoveDeploymentGroupResponse> removeDeploymentGroup(String name) {
        return Futures.transform(this.request(this.uri("/deployment-group/" + name), "DELETE"), ConvertResponseToPojo.create(RemoveDeploymentGroupResponse.class, (Set<Integer>)ImmutableSet.of((Object)200, (Object)400)));
    }

    public ListenableFuture<RollingUpdateResponse> rollingUpdate(String deploymentGroupName, JobId job, RolloutOptions options) {
        return Futures.transform(this.request(this.uri(this.path("/deployment-group/%s/rolling-update", deploymentGroupName)), "POST", new RollingUpdateRequest(job, options)), ConvertResponseToPojo.create(RollingUpdateResponse.class, (Set<Integer>)ImmutableSet.of((Object)200, (Object)400)));
    }

    public ListenableFuture<Integer> stopDeploymentGroup(String deploymentGroupName) {
        return this.status(this.request(this.uri(this.path("/deployment-group/%s/stop", deploymentGroupName)), "POST"));
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    public static HeliosClient create(String domain, String user) {
        return HeliosClient.newBuilder().setDomain(domain).setUser(user).build();
    }

    private static class Response {
        private final String method;
        private final URI uri;
        private final int status;
        private final byte[] payload;

        public Response(String method, URI uri, int status, byte[] payload) {
            this.method = method;
            this.uri = uri;
            this.status = status;
            this.payload = payload;
        }

        public String toString() {
            return "Response{method='" + this.method + '\'' + ", uri=" + this.uri + ", status=" + this.status + ", payload=" + this.decode(this.payload) + '}';
        }

        private String decode(byte[] payload) {
            if (payload == null) {
                return "";
            }
            int length = Math.min(payload.length, 1024);
            return new String(payload, 0, length, StandardCharsets.UTF_8);
        }
    }

    public static class Builder {
        private String user;
        private Supplier<List<URI>> endpointSupplier;

        public Builder setUser(String user) {
            this.user = user;
            return this;
        }

        public Builder setDomain(String domain) {
            return this.setEndpointSupplier(Resolver.supplier("helios", domain));
        }

        public Builder setEndpoints(List<URI> endpoints) {
            return this.setEndpointSupplier((Supplier<List<URI>>)Suppliers.ofInstance(endpoints));
        }

        public Builder setEndpoints(URI ... endpoints) {
            return this.setEndpointSupplier((Supplier<List<URI>>)Suppliers.ofInstance(Arrays.asList(endpoints)));
        }

        public Builder setEndpoints(String ... endpoints) {
            return this.setEndpointStrings(Arrays.asList(endpoints));
        }

        public Builder setEndpointStrings(List<String> endpoints) {
            ArrayList uris = Lists.newArrayList();
            for (String endpoint : endpoints) {
                uris.add(URI.create(endpoint));
            }
            return this.setEndpoints(uris);
        }

        public Builder setEndpointSupplier(Supplier<List<URI>> endpointSupplier) {
            this.endpointSupplier = endpointSupplier;
            return this;
        }

        public HeliosClient build() {
            return new HeliosClient(this.user, this.endpointSupplier);
        }
    }

    private static final class ConvertResponseToPojo<T>
    implements AsyncFunction<Response, T> {
        private final JavaType javaType;
        private final Set<Integer> decodeableStatusCodes;

        private ConvertResponseToPojo(JavaType javaType) {
            this(javaType, (Set<Integer>)ImmutableSet.of((Object)200));
        }

        public ConvertResponseToPojo(JavaType type, Set<Integer> decodeableStatusCodes) {
            this.javaType = type;
            this.decodeableStatusCodes = decodeableStatusCodes;
        }

        public static <T> ConvertResponseToPojo<T> create(JavaType type, Set<Integer> decodeableStatusCodes) {
            return new ConvertResponseToPojo<T>(type, decodeableStatusCodes);
        }

        public static <T> ConvertResponseToPojo<T> create(Class<T> clazz, Set<Integer> decodeableStatusCodes) {
            return new ConvertResponseToPojo<T>(Json.type(clazz), decodeableStatusCodes);
        }

        public ListenableFuture<T> apply(Response reply) throws HeliosException {
            Object result;
            if (reply.status == 404 && !this.decodeableStatusCodes.contains(404)) {
                return Futures.immediateFuture(null);
            }
            if (!this.decodeableStatusCodes.contains(reply.status)) {
                throw new HeliosException("request failed: " + reply);
            }
            if (reply.payload.length == 0) {
                throw new HeliosException("bad reply: " + reply);
            }
            try {
                result = Json.read(reply.payload, this.javaType);
            }
            catch (IOException e) {
                throw new HeliosException("bad reply: " + reply, e);
            }
            return Futures.immediateFuture(result);
        }
    }
}

